/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: EntityTransformer.java 2010 2010-06-13 18:18:35Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.4/topia-persistence/src/main/java/org/nuiton/topia/generator/EntityTransformer.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.generator;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModelAssociationClass;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.EntityVisitor;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityAbstract;

import java.beans.Introspector;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import static org.nuiton.eugene.GeneratorUtil.getSimpleName;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.STEREOTYPE_ENTITY;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.TAG_ANNOTATION;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.TAG_DB_NAME;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.hasUnidirectionalRelationOnAbstractType;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.isDateType;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.isPrimitiveType;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.shouldBeAbstract;






/**
 * A template to generate all the {@link TopiaEntity} api for all classifier
 * with a {@code entity} stereotype.
 *
 * For example, given a {@code House} entity, it will generates :
 * <ul>
 * <li>{@code House} : contract of entity</li>
 * <li>{@code AbstractHouse} : default abstract implementation of entity</li>
 * <li>{@code HouseImpl} : default impl of abstract entity</li>
 * </ul>
 *
 * <b>Note: </b> The impl will ony be generated in these cases :
 * <ul>
 * <li>There is no abstract method</li>
 * <li>There is no already defined such class in class-path</li>
 * </ul>
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.3.4
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.EntityTransformer"
 */
public class EntityTransformer extends ObjectModelTransformerToJava {

    /** Logger */
    private static final Log log = LogFactory.getLog(EntityTransformer.class);

    protected ObjectModelInterface outputInterface;

    protected ObjectModelClass outputAbstract;

    protected ObjectModelClass outputImpl;

    protected void clean() {
        outputInterface = null;
        outputAbstract = null;
        outputImpl = null;
    }

    @Override
    public void transformFromClass(ObjectModelClass input) {

        if (!input.hasStereotype(STEREOTYPE_ENTITY)) {

            // not an entity, skip class.
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("for entity : "+input.getQualifiedName());
            log.info("Will use classLoader "+ getClassLoader());
        }

        // fix once for all the constant prefix to use

        String prefix = getConstantPrefix(input, "");

        if (StringUtils.isEmpty(prefix)) {

            // no specific prefix, so no prefix
            if (log.isWarnEnabled()) {
                log.warn("[" + input.getName() + "] Will generate constants with NO prefix, not a good idea...");
            }
        }

        setConstantPrefix(prefix);

        String clazzName = input.getName();
        String packageName = input.getPackageName();

        Collection<ObjectModelAttribute> attributes = input.getAttributes();

        Collection<ObjectModelOperation> operations = input.getOperations();

        // Create classifiers : Interface, Abstract
        outputInterface = createInterface(clazzName, packageName);
        outputAbstract = createAbstractClass(clazzName + "Abstract", packageName);

        generateInterface(input, outputInterface, attributes, operations);
        generateAbstract(input, outputAbstract, attributes, operations);

//        ObjectModelClassifier output;
//
//        // generate interface
//
//        {
//            outputInterface = createInterface(clazzName, packageName);
//            generateInterface(input, outputInterface, attributes, operations);
//        }
//
//        // generate abstract
//
//        {
//            outputAbstract = createAbstractClass( clazzName + "Abstract", packageName);
//            generateAbstract(input, outputAbstract, attributes, operations);
//
//        }

        boolean generateImpl = isGenerateImpl(input, operations);

        if (generateImpl) {

            String implName = clazzName + "Impl";
            if (isVerbose()) {
                log.info("Will generate [" + implName + "]");
            }

            if (isAbstract(input)) {
                outputImpl = createAbstractClass(implName, packageName);
            } else {
                outputImpl = createClass(implName, packageName);
            }

            // generate impl
            generateImpl(input, outputImpl);
        }

        // Clean data output after transformation
        clean();
    }

    protected boolean isGenerateImpl(ObjectModelClass input,
                                     Collection<ObjectModelOperation> operations) {

        String fqn = input.getQualifiedName() + "Impl";

        URL fileLocation = getFileInClassPath(fqn);

        if (fileLocation != null) {

            // there is already a existing file in class-path, skip
            if (isVerbose()) {
                log.info("Will not generate [" + fqn + "], found existing file in class-path : " + fileLocation);
            }
            return false;
        }

        // On ne génère pas le impl si l'entité a des opérations qui ne sont
        // pas seulement pour le DAO
        if (!operations.isEmpty()) {

            if (isVerbose()) {
                log.info("Will not generate [" + fqn + "], there is some operations and not only DAO ones");
            }
            return false;
        }

        //De même, on ne génère pas le impl si il y a des opérations venant des
        // superclasses non implémentées
        for (ObjectModelOperation otherOp : input.getAllOtherOperations(false)) {
            if (otherOp.isAbstract()) {
                if (isVerbose()) {
                    log.info("Will not generate [" + fqn + "], there is a abstract operation [" + otherOp.getName() + "] in allOtherOperations.");
                }
                return false;
            }
        }

        return true;
    }

    protected void generateInterface(ObjectModelClass input,
                                     ObjectModelInterface output,
                                     Collection<ObjectModelAttribute> attributes,
                                     Collection<ObjectModelOperation> operations) {

        addImport(output, TopiaEntity.class);

        Set<String> constants = addConstantsFromDependency(input, output);

        if (log.isDebugEnabled()) {
            log.debug("Add constants from dependency : " + constants);
        }

        if (TopiaGeneratorUtil.hasDocumentation(input)) {
            setDocumentation(output,input.getDocumentation());
        }

        // super classes

        for (ObjectModelClassifier parent : input.getInterfaces()) {
            addInterface(output,parent.getQualifiedName());
        }

        boolean needTopiaEntity = true;
        for (ObjectModelClassifier parent : input.getSuperclasses()) {
            if (parent.hasStereotype(STEREOTYPE_ENTITY)) {
                addInterface(output,parent.getQualifiedName());
                needTopiaEntity = false;
                break;
            }
        }

        if (needTopiaEntity) {
            addInterface(output, TopiaEntity.class);
        }

        generateInterfaceStaticColumnNames(input, output);

        // attributes

        for (ObjectModelAttribute attr : attributes) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (!attr.isNavigable() &&
                    !TopiaGeneratorUtil.hasUnidirectionalRelationOnAbstractType(
                            reverse, model)) {
                continue;
            }

            if (attr.hasAssociationClass()) {

                addInterfaceAssociationAttribute(output, attr);

            } else {

                addInterfaceNoneAssociationAttribute(output, attr);
            }
        }

        //Méthodes d'accès aux attributs d'une classe d'associations

        if (input instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc =
                    (ObjectModelAssociationClass) input;
            for (ObjectModelAttribute attr :
                    assoc.getParticipantsAttributes()) {
                if (attr != null) {
                    String type = attr.getType();
                    String name = attr.getName();
                    generateInterfaceAssociationAccessors(output, name, type);
                    if (attr.getReverseAttribute() == null) {
                        type = ((ObjectModelClassifier)
                                attr.getDeclaringElement()).getQualifiedName();
                        name = attr.getDeclaringElement().getName();
                        generateInterfaceAssociationAccessors(output, name, type);
                    }
                }
            }
        }

        for (ObjectModelOperation operation : operations) {
            generateInterfaceOperation(operation, output);
        }
    }

    protected void generateAbstract(ObjectModelClass input,
                                    ObjectModelClass output,
                                    Collection<ObjectModelAttribute> attributes,
                                    Collection<ObjectModelOperation> operations) {

        String clazzName = input.getName();
        String clazzFQN = TopiaGeneratorUtil.getSimpleName(
                input.getQualifiedName());

        addInterface(output, clazzName);

        addImport(output, ArrayList.class);
        addImport(output, List.class);
        addImport(output, TopiaEntity.class);
        addImport(output, TopiaContextImplementor.class);

        // javadoc

        StringBuilder doc = new StringBuilder();
        doc.append("Implantation POJO pour l'entité {@link ");
        doc.append(StringUtils.capitalize(clazzFQN));
        doc.append("}\n");

        {
            String dbName = input.getTagValue(TAG_DB_NAME);
            if (dbName != null) {
                doc.append("<p>Nom de l'entité en BD : ");
                doc.append(dbName);
                doc.append(".</p>");
            }
        }

        setDocumentation(output, doc.toString());

        // super classes

        for (ObjectModelClass parent : input.getSuperclasses()) {
            String extendClass = parent.getQualifiedName();
            //Si une des classes parentes définies des méthodes abstraites, son
            // impl ne sera pas créé
            boolean abstractParent = shouldBeAbstract(parent);
            if (parent.hasStereotype(STEREOTYPE_ENTITY)) {
                if (abstractParent) {
                    extendClass += "Abstract";
                } else {
                    extendClass += "Impl";
                }
            }
            setSuperClass(output, extendClass);
        }

        if (output.getSuperclasses().isEmpty()) {
            setSuperClass(output, TopiaEntityAbstract.class);
        }

        // serialVersionUID

        String svUID = TopiaGeneratorUtil.findTagValue("serialVersionUID",
                input, model);
        if (svUID != null) {

            addConstant(output, "serialVersionUID", long.class, svUID,
                         ObjectModelModifier.PRIVATE
            );
        }

        ObjectModelParameter attr2;

        for (ObjectModelAttribute attr : attributes) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();

            // pour les asso quoi qu'il arrive il faut les lier des 2 cotes
            // pour pouvoir supprimer en cascade l'asso lors de la suppression
            // d'un des cotes
            if (!(attr.isNavigable()
                    || hasUnidirectionalRelationOnAbstractType(reverse, model)
                    || attr.hasAssociationClass())) {
                continue;
            }

            String type;
            String name;

            if (!attr.hasAssociationClass()) {
                String attrName = attr.getName();
                name = attrName;
                type = attr.getType();
            } else {
                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                //TODO THIMEL : Je pense que les
                // GeneratorUtil.toLowerCaseFirstLetter sont inutiles
                // ici, ou alors il faudrait le faire partout
                name = GeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
                type = attr.getAssociationClass().getQualifiedName();
            }

            if (GeneratorUtil.isNMultiplicity(attr)) {
                String collectionType =
                        TopiaGeneratorUtil.getNMultiplicityInterfaceType(attr);
                type = collectionType + '<' + type + '>';
            }

            String attrVisibility = attr.getVisibility();

            attr2 = addAttribute(output, name, type, null,
                    ObjectModelModifier.toValue(attrVisibility),
                    ObjectModelModifier.PROTECTED
            );

            doc = new StringBuilder();

            if (TopiaGeneratorUtil.hasDocumentation(attr) ||
                    attr.hasTagValue(TAG_DB_NAME)) {
                if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                    String attrDocumentation = attr.getDocumentation();
                    doc.append(attrDocumentation).append('\n');
                }
                if (attr.hasTagValue(TAG_DB_NAME)) {
                    String dbName =
                            attr.getTagValue(TAG_DB_NAME);
                    doc.append("Nom de l'attribut en BD : ");
                    doc.append(dbName);
                    doc.append('\n');
                }
            }

            setDocumentation(attr2, doc.toString());


            if (attr.hasTagValue(TAG_ANNOTATION)) {
                String annotation =  attr.getTagValue(TAG_ANNOTATION);
                //FIXME Make annotation works...
                //TODO tchemit 20100513 Test it still works
                addAnnotation(output, attr2, annotation);
            }
        }

        //Déclaration des attributs d'une classe d'associations
        if (input instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc =
                    (ObjectModelAssociationClass) input;
            for (ObjectModelAttribute attr : assoc.getParticipantsAttributes()) {
                if (attr != null) {
                    String attrVisibility = attr.getVisibility();
                    String attrType = attr.getType();
                    String attrName = attr.getName();
                    addAttribute(output,
                            GeneratorUtil.toLowerCaseFirstLetter(attrName),
                            attrType, null,
                            ObjectModelModifier.toValue(attrVisibility));
                }
            }
        }

        ObjectModelOperation op = addOperation(output, "update", "void",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        setDocumentation(op,"@deprecated since 2.3.4, use the DAO api instead.");
        addAnnotation(output, op, Deprecated.class.getSimpleName());
        addAnnotation(output, op, Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        ((TopiaContextImplementor)getTopiaContext()).getDAO("+clazzName+".class).update(this);\n"
+""
        );

        op = addOperation(output, "delete", "void", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        setDocumentation(op,"@deprecated since 2.3.4, use the DAO api instead.");
        addAnnotation(output, op, Deprecated.class.getSimpleName());
        addAnnotation(output, op, Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        ((TopiaContextImplementor)getTopiaContext()).getDAO("+clazzName+".class).delete(this);\n"
+""
        );

        generateAcceptMethod(output, input);

        generateAggregateMethod(output, input);

        generateCompositeMethod(output, input);


        for (ObjectModelAttribute attr : attributes) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();

            if (!(attr.isNavigable()
                    || hasUnidirectionalRelationOnAbstractType(reverse, model))) {
                continue;
            }

            transformAttribute(output, attr, reverse);
        }


        //Méthodes d'accès aux attributs d'une classe d'associations
        if (input instanceof ObjectModelAssociationClass) {

            for (ObjectModelAttribute attr :
                    ((ObjectModelAssociationClass) input).getParticipantsAttributes()) {
                if (attr != null) {
                    String attrType = TopiaGeneratorUtil.getSimpleName(attr.getType());
                    String attrName = attr.getName();
                    generateAssociationAccessors(output, attrName, attrType);
                }
            }
        }

        generateAbstractMethods(output, input);

        boolean doGenerateToString = TopiaGeneratorUtil.generateToString(input,
                model);
        if (doGenerateToString) {
            addImport(output, ToStringBuilder.class);
            generateToStringMethod(output, input);
        }

        String i18nPrefix = TopiaGeneratorUtil.getI18nPrefix(input, model);
        if (!StringUtils.isEmpty(i18nPrefix)) {
            // generate i18n prefix
            generateI18n(output, i18nPrefix, input);
        }
    }

    protected void generateImpl(ObjectModelClass input,
                                ObjectModelClass output) {

        setDocumentation(output, "Implantation des operations pour l'entité " +
                                 input.getName() + ".");
        setSuperClass(output, input.getQualifiedName() + "Abstract");
    }


    // -------------------------------------------------------------------------
    // Interface generation
    // -------------------------------------------------------------------------

    private void generateInterfaceStaticColumnNames(ObjectModelClass input,
                                                    ObjectModelInterface output) {

        for (ObjectModelAttribute attr : input.getAttributes()) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (!(attr.isNavigable() ||
                    TopiaGeneratorUtil.hasUnidirectionalRelationOnAbstractType(
                            reverse, model)
                    || attr.hasAssociationClass())) {
                continue;
            }
            String attrName;
            if (!attr.hasAssociationClass()) {
                attrName = attr.getName();
            } else {
                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                attrName = GeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
            }
            String attrColName = getConstantName(attrName);
            addAttribute(output,
                    attrColName,
                    String.class,
                    "\"" + attrName + "\"",
                    ObjectModelModifier.PACKAGE);
        }

        //Déclaration des noms des champs des attributs d'une classe d'associations
        if (input instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc = (ObjectModelAssociationClass) input;
            for (ObjectModelAttribute attr : assoc.getParticipantsAttributes()) {
                if (attr != null) {
                    String attrName = attr.getName();
                    String attrColName = getConstantName(attrName);
                    addAttribute(output,
                            attrColName,
                            String.class,
                            "\"" + attrName + "\"",
                            ObjectModelModifier.PACKAGE);
                }
            }
        }
    }

    protected void addInterfaceNoneAssociationAttribute(ObjectModelInterface output,
                                                        ObjectModelAttribute attr) {
        String attrName = attr.getName();
        String attrType = attr.getType();
        ObjectModelOperation op;
        ObjectModelParameter attr2;

        if (!GeneratorUtil.isNMultiplicity(attr)) {

            // setXXX

            op = addOperation(output,
                    "set" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            attr2 = addParameter(op, attrType,
                                 GeneratorUtil.toLowerCaseFirstLetter(attrName));
            setDocumentation(attr2, "La valeur de l'attribut " + attrName
                    + " à positionner.");

            // getXXX

            // Add getter for simple property.
            // Only one call for this method, no need for abstract generation
            // TODO-fdesbois-2010-05-27 : manage all attribute cases in this method
            addSimpleGetterOperation(attr, null);

        } else {
            String collectionInterface =
                    TopiaGeneratorUtil.getNMultiplicityInterfaceType(attr);

            // addXXX

            op = addOperation(output,
                    "add" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            attr2 = addParameter(op, attrType,
                                 GeneratorUtil.toLowerCaseFirstLetter(attrName));
            setDocumentation(attr2,
                             "L'instance de " + attrName + " à ajouter");

            // addAllXXX

            op = addOperation(output,
                    "addAll" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            attr2 = addParameter(op,
                                 collectionInterface + "<" + attrType + ">",
                                 GeneratorUtil.toLowerCaseFirstLetter(attrName));
            setDocumentation(attr2,
                             "Les instances de " + attrName + " à ajouter");

            // setXXX

            op = addOperation(output,
                    "set" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            attr2 = addParameter(op,
                                 collectionInterface + "<" + attrType + ">",
                                 GeneratorUtil.toLowerCaseFirstLetter(attrName));
            setDocumentation(attr2,
                             "La Collection de " + attrName + " à ajouter");

            // removeXXX

            op = addOperation(output,
                    "remove" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            attr2 = addParameter(op,
                                 attrType,
                                 GeneratorUtil.toLowerCaseFirstLetter(attrName));
            setDocumentation(attr2,
                             "L'instance de " + attrName + " à retirer");

            // clearXXX

            op = addOperation(output,
                    "clear" + StringUtils.capitalize(attrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, attr.getDocumentation());
            }
            setDocumentation(attr2, "Vide la Collection de " + attrName);

            // getXXX

            op = addOperation(output,
                    "get" + StringUtils.capitalize(attrName),
                    collectionInterface + "<" + attrType + ">",
                    ObjectModelModifier.PACKAGE);
            if (TopiaGeneratorUtil.hasDocumentation(attr)) {
                setDocumentation(op, "Retourne la collection.");
            }

            if (!TopiaGeneratorUtil.isPrimitiveType(attr) &&
                !TopiaGeneratorUtil.isDateType(attr)) {

                // getXXXByTopiaId

                op = addOperation(output,
                        "get" + StringUtils.capitalize(attrName) + "ByTopiaId",
                        attrType,
                        ObjectModelModifier.PACKAGE);
                setDocumentation(op, "Recupère l'attribut " + attrName + " à partir de son topiaId");
                attr2 = addParameter(op, String.class, "topiaId");
                setDocumentation(attr2, "le topia id de l'entité recherchée");
            }

            // sizeXXX

            op = addOperation(output,
                    "size" + StringUtils.capitalize(attrName),
                    int.class,
                    ObjectModelModifier.PACKAGE);
            setDocumentation(op, "Retourne le nombre d'éléments de la collection " + attrName);

            // isXXXEmpty

            op = addOperation(output,
                    "is" + StringUtils.capitalize(attrName) + "Empty",
                    boolean.class,
                    ObjectModelModifier.PACKAGE);
            setDocumentation(op, "Retourne {@code true} si la collection " + attrName + " est vide.");
        }
    }

    protected void addInterfaceAssociationAttribute(ObjectModelInterface output,
                                                    ObjectModelAttribute attr) {
        String attrName = attr.getName();
        String attrType = attr.getType();
        String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
        String assocClassFQN = attr.getAssociationClass().getQualifiedName();
        String assocClassName = attr.getAssociationClass().getName();

        ObjectModelOperation op;
        ObjectModelParameter attr2;

        if (!GeneratorUtil.isNMultiplicity(attr)) {

            // setXXX

            op = addOperation(output,
                    "set" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            attr2 = addParameter(op, assocClassFQN, GeneratorUtil.toLowerCaseFirstLetter(assocClassName));
            setDocumentation(attr2, "La valeur de l'attribut " + assocClassName + " à positionner");

            // getXXX : getter interface for association attribute with unique multiplicity

            addOperation(output,
                    "get" + StringUtils.capitalize(assocAttrName),
                    assocClassFQN,
                    ObjectModelModifier.PACKAGE);

        } else {
            String collectionInterface = TopiaGeneratorUtil.getNMultiplicityInterfaceType(attr);

            // addXXX

            op = addOperation(output,
                    "add" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            attr2 = addParameter(op, assocClassFQN, GeneratorUtil.toLowerCaseFirstLetter(assocClassName));
            setDocumentation(attr2, "L'instance de " + assocClassName + " à ajouter");

            // addAllXXX

            op = addOperation(output,
                    "addAll" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            attr2 = addParameter(op, collectionInterface + "<" + assocClassFQN + ">", GeneratorUtil.toLowerCaseFirstLetter(assocClassName));
            setDocumentation(attr2, "Les instances de " + assocClassName + " à ajouter");

            // setXXX

            op = addOperation(output,
                    "set" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            attr2 = addParameter(op, collectionInterface + "<" + assocClassFQN + ">", GeneratorUtil.toLowerCaseFirstLetter(assocClassName));
            setDocumentation(attr2, "La Collection de " + assocClassName + " à ajouter");

            // removeXXX

            op = addOperation(output,
                    "remove" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            attr2 = addParameter(op, assocClassFQN, GeneratorUtil.toLowerCaseFirstLetter(assocClassName));
            setDocumentation(attr2, "L'instance de " + assocClassName + " à retirer");

            // clearXXX

            op = addOperation(output,
                    "clear" + StringUtils.capitalize(assocAttrName),
                    "void",
                    ObjectModelModifier.PACKAGE);
            setDocumentation(op, "Vide la Collection de " + assocClassName + " .");

            // getXXX

            addOperation(output,
                    "get" + StringUtils.capitalize(assocAttrName),
                    collectionInterface + "<" + assocClassFQN + ">",
                    ObjectModelModifier.PACKAGE);

            if (!TopiaGeneratorUtil.isPrimitiveType(attr) && !TopiaGeneratorUtil.isDateType(attr)) {

                // getXXXByTopiaId

                op = addOperation(output,
                        "get" + StringUtils.capitalize(assocAttrName) + "ByTopiaId",
                        assocClassFQN,
                        ObjectModelModifier.PACKAGE);
                setDocumentation(op, "Recupère l'attribut " + attrName + " à partir de son topiaId");
                attr2 = addParameter(op, String.class, "topiaId");
                setDocumentation(attr2, "le topia id de l'entité recherchée");
            }

            // getXXX

            op = addOperation(output,
                    "get" + StringUtils.capitalize(assocAttrName),
                    assocClassFQN);
            addParameter(op, attrType, "value");


            // sizeXXX

            addOperation(output,
                    "size" + StringUtils.capitalize(assocAttrName),
                    int.class,
                    ObjectModelModifier.PACKAGE);


            // isXXXEmpty

            addOperation(output,
                    "is" + StringUtils.capitalize(assocAttrName) + "Empty",
                    boolean.class,
                    ObjectModelModifier.PACKAGE);
        }
    }

    private void generateInterfaceOperation(ObjectModelOperation op,
                                            ObjectModelInterface output) {


        String visibility = op.getVisibility();
        if (op.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO) ||
            !visibility.equals(ObjectModelModifier.PUBLIC.toString())) {
            // Pas de génération des signatures de méthodes pour celles à intégrer au DAO de l'entité
            return;
        }

        String opName = op.getName();
        String opType = op.getReturnType();

        ObjectModelOperation op2 = addOperation(output, opName, opType, ObjectModelModifier.PACKAGE);
        if (TopiaGeneratorUtil.hasDocumentation(op)) {
            setDocumentation(op2, op.getDocumentation());
        }

        for (ObjectModelParameter param : op.getParameters()) {
            String paramName = param.getName();
            String paramType = param.getType();
            ObjectModelParameter param2 = addParameter(op2, paramType, paramName);
            if (TopiaGeneratorUtil.hasDocumentation(param)) {
                setDocumentation(param2, param.getDocumentation());
            }
        }
        for (String exception : op.getExceptions()) {
            addException(op2, exception);
        }
    }

    private void generateInterfaceAssociationAccessors(ObjectModelInterface output,
                                              String attrName,
                                              String attrType) {

        ObjectModelOperation op;
        ObjectModelParameter param;

        op = addOperation(output,
                "set" + StringUtils.capitalize(attrName),
                "void",
                ObjectModelModifier.PACKAGE);
        param = addParameter(op, attrType, "value");
        setDocumentation(param, "La valeur de l'attribut " + attrName + " à positionner.");

        op = addOperation(output,
                "get" + StringUtils.capitalize(attrName),
                attrType,
                ObjectModelModifier.PACKAGE);
        setDocumentation(op, "Retourne la valeur de l'attribut " + attrName + ".");
    }

    // -------------------------------------------------------------------------
    // Abstract generation
    // -------------------------------------------------------------------------

     protected void transformAttribute(ObjectModelClass result,
                                      ObjectModelAttribute attr,
                                      ObjectModelAttribute reverse) {

        String attrName = attr.getName();
        String attrType = TopiaGeneratorUtil.getSimpleName(attr.getType());
        addImport(result, attrType);

        attrType = TopiaGeneratorUtil.getSimpleName(attrType);

        ObjectModelOperation op;

        if (!GeneratorUtil.isNMultiplicity(attr)) {

            if (attr.hasAssociationClass()) {
                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                String assocClassFQN = attr.getAssociationClass().getQualifiedName();
                addImport(result, assocClassFQN);
                assocClassFQN = TopiaGeneratorUtil.getSimpleName(assocClassFQN);
                String name = GeneratorUtil.toLowerCaseFirstLetter(assocAttrName);

                // setXXX

                op = addOperation(result,
                        "set" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, assocClassFQN, "association");
                setOperationBody(op, ""
+"\n"
+"        "+assocClassFQN+" _oldValue = this."+name+";\n"
+"        fireOnPreWrite("+getConstantName(name)+", _oldValue, association);\n"
+"        this."+name+" = association;\n"
+"        fireOnPostWrite("+getConstantName(name)+", _oldValue, association);\n"
+""
                );

                // getXXX : getter for association attribute with unique multiplicity

                op = addOperation(result,
                        "get" + StringUtils.capitalize(assocAttrName),
                        assocClassFQN, ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        return "+name+";\n"
+""
                );
            } else {

                // setXXX

                op = addOperation(result,
                        "set" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, attrType, "value");
                setOperationBody(op, ""
+"\n"
+"        "+attrType+" _oldValue = this."+attrName+";\n"
+"        fireOnPreWrite("+getConstantName(attrName)+", _oldValue, value);\n"
+"        this."+attrName+" = value;\n"
+"        fireOnPostWrite("+getConstantName(attrName)+", _oldValue, value);\n"
+""

                );

                // getXXX : getter for simple attribute (as bean convention)

                // Getter is already set in abstract when added in interface
                // see {@link #addSimpleGetterOperation(ObjectModelAttribute, String)}

            }
        } else { //NMultiplicity
            String collectionInterface = TopiaGeneratorUtil.getNMultiplicityInterfaceType(attr);
            String collectionObject = TopiaGeneratorUtil.getNMultiplicityObjectType(attr);
            addImport(result, collectionInterface);
            addImport(result, collectionObject);
            collectionInterface = TopiaGeneratorUtil.getSimpleName(collectionInterface);
            collectionObject = getSimpleName(collectionObject);

            if (!attr.hasAssociationClass()) {
                //Méthodes remplacées par des accesseurs sur les classes d'assoc

                // addXXX

                op = addOperation(result,
                        "add" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, attrType, attrName);
                StringBuilder body = new StringBuilder();

                body.append(""
+"\n"
+"        fireOnPreWrite("+getConstantName(attrName)+", null, "+attrName+");\n"
+"        if (this."+attrName+" == null) {\n"
+"            this."+attrName+" = new "+collectionObject+"<"+attrType+">();\n"
+"        }\n"
+""
                );

                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    String reverseAttrType = TopiaGeneratorUtil.getSimpleName(reverse.getType());
                    if (!GeneratorUtil.isNMultiplicity(reverse)) {
                        body.append(""
+"        "+attrName+".set"+StringUtils.capitalize(reverseAttrName)+"(this);\n"
+""
                        );
                    } else {
                        body.append(""
+"        if ("+attrName+".get"+StringUtils.capitalize(reverseAttrName)+"() == null) {\n"
+"            "+attrName+".set"+StringUtils.capitalize(reverseAttrName)+"(new "+collectionObject+"<"+reverseAttrType+">());\n"
+"        }\n"
+"        "+attrName+".get"+StringUtils.capitalize(reverseAttrName)+"().add(this);\n"
+""
                        );
                    }
                }
                body.append(""
+"        this."+attrName+".add("+attrName+");\n"
+"        fireOnPostWrite("+getConstantName(attrName)+", this."+attrName+".size(), null, "+attrName+");\n"
+""
                );
                setOperationBody(op, body.toString());

                // addAllXXX

                op = addOperation(result,
                        "addAll" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, collectionInterface + '<' + attrType + '>', "values");

                setOperationBody(op, ""
+"\n"
+"        if (values == null) {\n"
+"            return;\n"
+"        }\n"
+"        for ("+attrType+" item : values) {\n"
+"            add"+StringUtils.capitalize(attrName)+"(item);\n"
+"        }\n"
+""
                );

                if (!isPrimitiveType(attr) && !isDateType(attr)) {

                    op = addOperation(result,
                            "get" + StringUtils.capitalize(attrName) + "ByTopiaId",
                            attrType,
                            ObjectModelModifier.PUBLIC);
                    addParameter(op, String.class, "topiaId");
                    setOperationBody(op, ""
+"\n"
+"        return org.nuiton.topia.persistence.util.TopiaEntityHelper.getEntityByTopiaId("+attrName+", topiaId);\n"
+" "
                    );

                }

                // setXXX

                op = addOperation(result,
                        "set" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, collectionInterface + '<' + attrType + '>', "values");

                setOperationBody(op, ""
+"\n"
+"        "+collectionInterface+"<"+attrType+"> _oldValue = "+attrName+";\n"
+"        fireOnPreWrite("+getConstantName(attrName)+", _oldValue, values);\n"
+"        "+attrName+" = values;\n"
+"        fireOnPostWrite("+getConstantName(attrName)+", _oldValue, values);\n"
+""
                );


                // removeXXX


                op = addOperation(result,
                        "remove" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, attrType, "value");
                body = new StringBuilder();

                body.append(""
+"\n"
+"        fireOnPreWrite("+getConstantName(attrName)+", value, null);\n"
+"        if ((this."+attrName+" == null) || (!this."+attrName+".remove(value))) {\n"
+"            throw new IllegalArgumentException(\"List does not contain given element\");\n"
+"        }\n"
+""
                );

                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    if (!GeneratorUtil.isNMultiplicity(reverse)) {
                        body.append(""
+"        value.set"+StringUtils.capitalize(reverseAttrName)+"(null);\n"
+""
                        );
                    } else {
                        body.append(""
+"        value.get"+StringUtils.capitalize(reverseAttrName)+"().remove(this);\n"
+""
                        );
                    }
                }
                body.append(""
+"        fireOnPostWrite("+getConstantName(attrName)+", this."+attrName+".size()+1, value, null);\n"
+""
                );
                setOperationBody(op, body.toString());


                // clearXXX

                op = addOperation(result,
                        "clear" + StringUtils.capitalize(attrName),
                        "void",
                        ObjectModelModifier.PUBLIC);

                body = new StringBuilder();

                body.append(""
+"\n"
+"        if (this."+attrName+" == null) {\n"
+"            return;\n"
+"        }\n"
+""
                );

                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    body.append(""
+"        for ("+attrType+" item : this."+attrName+") {\n"
+""
                    );
                    if (!GeneratorUtil.isNMultiplicity(reverse)) {
                        body.append(""
+"            item.set"+StringUtils.capitalize(reverseAttrName)+"(null);\n"
+""
                        );
                    } else {
                        body.append(""
+"            item.get"+StringUtils.capitalize(reverseAttrName)+"().remove(this);\n"
+""
                        );
                    }
                    body.append(""
+"        }\n"
+""
                    );
                }
                body.append(""
+"        "+collectionInterface+"<"+attrType+"> _oldValue = new "+collectionObject+"<"+attrType+">(this."+attrName+");\n"
+"        fireOnPreWrite("+getConstantName(attrName)+", _oldValue, this."+attrName+");\n"
+"        this."+attrName+".clear();\n"
+"        fireOnPostWrite("+getConstantName(attrName)+", _oldValue, this."+attrName+");\n"
+""
                );

                setOperationBody(op, body.toString());

            } else {


                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                String assocClassFQN = attr.getAssociationClass().getQualifiedName();
//                String assocClassFQN = TopiaGeneratorUtil.getSimpleName(attr.getAssociationClass().getQualifiedName());

                // addXXX

                op = addOperation(result,
                        "add" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, assocClassFQN, "value");

                StringBuilder body = new StringBuilder();

                body.append(""
+"\n"
+"        fireOnPreWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", null, value);\n"
+"        if (this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" == null) {\n"
+"            this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" = new "+collectionObject+"<"+assocClassFQN+">();\n"
+"        }\n"
+""
                );
                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    body.append(""
+"        value.set"+StringUtils.capitalize(reverseAttrName)+"(this);\n"
+""
                    );
                }
                body.append(""
+"        this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".add(value);\n"
+"        fireOnPostWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".size(), null, value);\n"
+""
                );
                setOperationBody(op, body.toString());


                if (!isPrimitiveType(attr) && !isDateType(attr)) {

                    // getXXXByTopiaId

                    op = addOperation(result,
                            "get" + StringUtils.capitalize(assocAttrName) + "ByTopiaId",
                            assocClassFQN,
                            ObjectModelModifier.PUBLIC);
                    addParameter(op, String.class, "topiaId");
                    setOperationBody(op, ""
+"\n"
+"        return org.nuiton.topia.persistence.util.TopiaEntityHelper.getEntityByTopiaId("+assocAttrName+", topiaId);\n"
+""
                    );

                }

                // addAllXXX

                op = addOperation(result,
                        "addAll" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, collectionInterface + '<' + assocClassFQN + '>', "values");
                setOperationBody(op, ""
+"\n"
+"        if (values == null) {\n"
+"            return;\n"
+"        }\n"
+"        for ("+assocClassFQN+" item : values) {\n"
+"            add"+StringUtils.capitalize(assocAttrName)+"(item);\n"
+"        }\n"
+""
                );

                // setXXX

                op = addOperation(result,
                        "set" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, collectionInterface + '<' + assocClassFQN + '>', "values");
                setOperationBody(op, ""
+"\n"
+"//        clear"+StringUtils.capitalize(assocAttrName)+"();\n"
+"//        addAll"+StringUtils.capitalize(assocAttrName)+"(values);\n"
+"// FIXME\n"
+"        "+collectionInterface+"<"+assocClassFQN+"> _oldValue = "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+";\n"
+"        fireOnPreWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", _oldValue, values);\n"
+"        "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" = values;\n"
+"        fireOnPostWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", _oldValue, values);\n"
+""
                );

                // removeXXX

                op = addOperation(result,
                        "remove" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(op, assocClassFQN, "value");

                body = new StringBuilder();

                body.append(""
+"\n"
+"        fireOnPreWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", value, null);\n"
+"        if ((this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" == null) || (!this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".remove(value))) {\n"
+"            throw new IllegalArgumentException(\"List does not contain given element\");\n"
+"        }\n"
+""
                );
                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    body.append(""
+"        value.set"+StringUtils.capitalize(reverseAttrName)+"(null);\n"
+""
                    );
                }
                body.append(""
+"        fireOnPostWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".size()+1, value, null);\n"
+""
                );

                setOperationBody(op, body.toString());

                // clearXXX

                op = addOperation(result,
                        "clear" + StringUtils.capitalize(assocAttrName),
                        "void",
                        ObjectModelModifier.PUBLIC);

                body = new StringBuilder();

                body.append(""
+"\n"
+"        if (this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" == null) {\n"
+"            return;\n"
+"        }\n"
+""
                );

                if (reverse != null && (reverse.isNavigable() ||
                        hasUnidirectionalRelationOnAbstractType(attr, model))) {
                    String reverseAttrName = reverse.getName();
                    body.append(""
+"        for ("+assocClassFQN+" item : this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+") {\n"
+"            item.set"+StringUtils.capitalize(reverseAttrName)+"(null);\n"
+"        }\n"
+""
                    );
                }
                body.append(""
+"        "+collectionInterface+"<"+assocClassFQN+"> _oldValue = new "+collectionObject+"<"+assocClassFQN+">(this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+");\n"
+"        fireOnPreWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", _oldValue, null);\n"
+"        this."+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".clear();\n"
+"        fireOnPostWrite("+getConstantName(GeneratorUtil.toLowerCaseFirstLetter(assocAttrName))+", _oldValue, null);\n"
+""
                );
                setOperationBody(op, body.toString());
            }

            if (!attr.hasAssociationClass()) {

                // getXXX : getter for collection entity attribute (N multiplicity)

                op = addOperation(result,
                        "get" + StringUtils.capitalize(attrName),
                        collectionInterface + '<' + attrType + '>',
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        return "+attrName+";\n"
+""
                );

                // sizeXXX

                op = addOperation(result,
                        "size" + StringUtils.capitalize(attrName),
                        int.class,
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        if ("+attrName+" == null) {\n"
+"            return 0;\n"
+"        }\n"
+"        return "+attrName+".size();\n"
+""
                );

                // isXXXEmpty

                op = addOperation(result,
                        "is" + StringUtils.capitalize(attrName) + "Empty",
                        boolean.class,
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        int size = size"+StringUtils.capitalize(attrName)+"();\n"
+"        return size == 0;\n"
+""
                );

            } else {
                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                String assocClassFQN = attr.getAssociationClass().getQualifiedName();
//                String assocClassFQN = TopiaGeneratorUtil.getSimpleName(attr.getAssociationClass().getQualifiedName());

                // getXXX : getter for association attribute with no parameter
                //  (return a collection)

                op = addOperation(result,
                        "get" + StringUtils.capitalize(assocAttrName),
                        collectionInterface + '<' + assocClassFQN + '>',
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        return "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+";\n"
+""
                );

                // getXXX : getter for association attribute with one parameter
                //  (return a single element)

                op = addOperation(result,
                        "get" + StringUtils.capitalize(assocAttrName),
                        assocClassFQN,
                        ObjectModelModifier.PUBLIC);
                addParameter(op, attrType, "value");
                setOperationBody(op, ""
+"\n"
+"        if (value == null || "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" == null) {\n"
+"            return null;\n"
+"        }\n"
+"        for ("+assocClassFQN+" item : "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+") {\n"
+"            if (value.equals(item.get"+StringUtils.capitalize(attrName)+"())) {\n"
+"                return item;\n"
+"            }\n"
+"        }\n"
+"        return null;\n"
+""
                );


                // sizeXXX

                op = addOperation(result,
                        "size" + StringUtils.capitalize(assocAttrName),
                        int.class,
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        if ("+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+" == null) {\n"
+"            return 0;\n"
+"        }\n"
+"        return "+GeneratorUtil.toLowerCaseFirstLetter(assocAttrName)+".size();\n"
+""
                );

                //isXXXEmpty

                op = addOperation(result,
                        "is" + StringUtils.capitalize(assocAttrName) + "Empty",
                        boolean.class,
                        ObjectModelModifier.PUBLIC);
                setOperationBody(op, ""
+"\n"
+"        int size = size"+StringUtils.capitalize(assocAttrName)+"();\n"
+"        return size == 0;\n"
+""
                );

            }
        }
    }

    /**
     * Add getter for simple property (neither association nor multiple).
     * Will add two different operations for boolean case ('is' method and
     * 'get' method). This method add the operation in both {@code
     * outputAbstract} and {@code outputInterface}.
     *
     * @param attribute         ObjectModelAttribute for getter operation
     * @param operationPrefix   Operation prefix : 'get' by default, if prefix
     *                          is null
     */
    protected void addSimpleGetterOperation(ObjectModelAttribute attribute,
                                            String operationPrefix) {

        String attrName = attribute.getName();
        String attrType = attribute.getType();

        if (operationPrefix == null) {
            operationPrefix = TopiaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;
        }

        if (log.isDebugEnabled()) {
            log.debug("Add getter operation for " + attrName +
                      " prefix = " + operationPrefix);
        }

        // CONTRACT in Interface
        ObjectModelOperation contract = addOperation(outputInterface,
                operationPrefix + StringUtils.capitalize(attrName),
                attrType,
                ObjectModelModifier.PACKAGE);

        if (TopiaGeneratorUtil.hasDocumentation(attribute)) {
            setDocumentation(contract, attribute.getDocumentation());
        }

        // IMPLEMENTATION in Abstract
        ObjectModelOperation impl = addOperation(outputAbstract,
                        contract.getName(),
                        contract.getReturnType(),
                        ObjectModelModifier.PUBLIC);
                setOperationBody(impl, ""
+"\n"
+"        fireOnPreRead("+getConstantName(attrName)+", "+attrName+");\n"
+"        "+attrType+" result = this."+attrName+";\n"
+"        fireOnPostRead("+getConstantName(attrName)+", "+attrName+");\n"
+"        return result;\n"
+""
        );

        // Generate 'is' getter for boolean attributes
        if (attrType.toLowerCase().contains("boolean") &&
                !operationPrefix.equals(TopiaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX)) {
            addSimpleGetterOperation(attribute,
                    TopiaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX);
        }
    }

    protected void generateToStringMethod(ObjectModelClass output,
                                          ObjectModelClass clazz) {
        if (log.isDebugEnabled()) {
            log.debug("generate toString method for clazz " +
                    clazz.getQualifiedName());
        }
        ObjectModelOperation op = addOperation(output,
                "toString",
                String.class,
                ObjectModelModifier.PUBLIC);
        addAnnotation(output, op, Override.class.getSimpleName());
        StringBuilder body = new StringBuilder();


        body.append(""
+"\n"
+"        String result = new ToStringBuilder(this).\n"
+""
        );
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            //FIXME possibilité de boucles (non directes)
            ObjectModelClass attrEntity = null;
            if (model.hasClass(attr.getType())) {
                attrEntity = model.getClass(attr.getType());
            }
            boolean isEntity = attrEntity != null &&
                    attrEntity.hasStereotype(STEREOTYPE_ENTITY);
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (isEntity && (reverse == null || !reverse.isNavigable())
                    && !attr.hasAssociationClass() || !isEntity) {
                String attrName = attr.getName();
                body.append(""
+"            append("+getConstantName(attrName)+", this."+attrName+").\n"
+""
                );
            }
        }
        body.append(""
+"         toString();\n"
+"        return result;\n"
+""
        );
        setOperationBody(op, body.length() == 0 ? " " : body.toString());

    }

    protected void generateCompositeMethod(ObjectModelClass output,
                                           ObjectModelClass clazz) {

        ObjectModelOperation op = addOperation(output,
                "getComposite",
                List.class.getName() + '<' + TopiaEntity.class.getName() + '>',
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addAnnotation(output, op, Override.class.getSimpleName());
        StringBuilder body = new StringBuilder();
        body.append(""
+"\n"
+"        List<TopiaEntity> tmp = new ArrayList<TopiaEntity>();\n"
+"\n"
+"        // pour tous les attributs rechecher les composites et les class d'asso\n"
+"        // on les ajoute dans tmp\n"
+""
        );
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.referenceClassifier() &&
                    attr.getClassifier().hasStereotype(
                            STEREOTYPE_ENTITY)) {
                if (attr.isComposite()) {
                    String attrName = attr.getName();
                    String getterName = "get" + StringUtils.capitalize(attrName);
                    if (GeneratorUtil.isNMultiplicity(attr)) {
                        body.append(""
+"        if ("+getterName+"() != null) {\n"
+"              tmp.addAll("+getterName+"());\n"
+"           }\n"
+""
                        );
                    } else {
                        body.append(""
+"        tmp.add("+getterName+"());\n"
+""
                        );
                    }
                } else if (attr.hasAssociationClass()) {
                    String assocAttrName = GeneratorUtil.getAssocAttrName(
                            attr);
                    String assocClassFQN = TopiaGeneratorUtil.getSimpleName(
                            attr.getAssociationClass().getQualifiedName());
                    String ref = "this." + GeneratorUtil.toLowerCaseFirstLetter(
                            assocAttrName);
                    if (!GeneratorUtil.isNMultiplicity(attr)) {
                        body.append(""
+"\n"
+"        if ("+ref+" != null) {\n"
+"            tmp.add("+ref+");\n"
+"        }\n"
+""
                        );
                    } else {
                        ObjectModelAttribute reverse = attr.getReverseAttribute();
                        String reverseAttrName = reverse.getName();
                        // On utilise pas l'attribut car il est potentiellement
                        // pas a jour, car  pour les asso avec cardinalité
                        // personne ne fait de add. Ce qui est normal, mais
                        // pour pouvoir faire tout de meme des delete en cascade
                        // sur les asso, le champs est dans le mapping
                        // hibernate et donc il le faut aussi dans la classe
                        // sinon hibernate rale lorsqu'il charge l'objet
//                        if (<%=ref%> != null) {
//                            tmp.addAll(<%=ref%>);
//                        }
                        body.append(""
+"\n"
+"        {\n"
+"            org.nuiton.topia.persistence.TopiaDAO<"+assocClassFQN+"> dao = ((TopiaContextImplementor) getTopiaContext()).getDAO("+assocClassFQN+".class);\n"
+"            List<"+assocClassFQN+"> findAllByProperties = dao.findAllByProperties(\""+reverseAttrName+"\", this);\n"
+"            if (findAllByProperties != null) {\n"
+"                tmp.addAll(findAllByProperties);\n"
+"            }\n"
+"        }\n"
+""
                        );
                    }
                }
            }
        }
        body.append(""
+"\n"
+"        // on refait un tour sur chaque entity de tmp pour recuperer leur\n"
+"        // composite\n"
+"        List<TopiaEntity> result = new ArrayList<TopiaEntity>();\n"
+"        for (TopiaEntity entity : tmp) {\n"
+"            if (entity != null) {\n"
+"                result.add(entity);\n"
+"                result.addAll(entity.getComposite());\n"
+"            }\n"
+"        }\n"
+"\n"
+"        return result;\n"
+"    "
        );

        setOperationBody(op, body.length() == 0 ? " " : body.toString());
    }

    protected void generateAggregateMethod(ObjectModelClass output,
                                           ObjectModelClass clazz) {

        ObjectModelOperation op = addOperation(output,
                "getAggregate",
                List.class.getName() + '<' + TopiaEntity.class.getName() + '>',
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addAnnotation(output, op, Override.class.getSimpleName());
        StringBuilder body = new StringBuilder();
        body.append(""
+"\n"
+"        List<TopiaEntity> tmp = new ArrayList<TopiaEntity>();\n"
+"\n"
+"        // pour tous les attributs rechecher les composites et les class d'asso\n"
+"        // on les ajoute dans tmp\n"
+""
        );
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.referenceClassifier() &&
                    attr.getClassifier().hasStereotype(
                            STEREOTYPE_ENTITY)) {
                if (attr.isAggregate()) {
                    String attrName = attr.getName();
                    String getterName = "get" + StringUtils.capitalize(attrName);
                    if (GeneratorUtil.isNMultiplicity(attr)) {
                        body.append(""
+"        tmp.addAll("+getterName+"());\n"
+""
                        );
                    } else {
                        body.append(""
+"        tmp.add("+getterName+"());\n"
+""
                        );
                    }
                }
            }
        }
        body.append(""
+"\n"
+"        // on refait un tour sur chaque entity de tmp pour recuperer leur\n"
+"        // composite\n"
+"        List<TopiaEntity> result = new ArrayList<TopiaEntity>();\n"
+"        for (TopiaEntity entity : tmp) {\n"
+"            result.add(entity);\n"
+"            result.addAll(entity.getAggregate());\n"
+"        }\n"
+"\n"
+"        return result;\n"
+""
        );

        setOperationBody(op, body.length() == 0 ? " " : body.toString());
    }

    protected void generateAcceptMethod(ObjectModelClass output,
                                        ObjectModelClass clazz) {

        ObjectModelOperation op = addOperation(output,
                "accept",
                "void",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addAnnotation(output, op, Override.class.getSimpleName());
        setDocumentation(op, "Envoi via les methodes du visitor l'ensemble des " +
                "champs de l'entity\n" +
                "avec leur nom, type et valeur.");
        {
            ObjectModelParameter attr = addParameter(op, EntityVisitor.class,
                    "visitor");
            setDocumentation(attr, "le visiteur de l'entite.");
        }
        StringBuilder body = new StringBuilder();
        body.append(""
+"\n"
+"        visitor.start(this);\n"
+""
        );
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();

            // pour les asso quoi qu'il arrive il faut les lier des 2 cotes
            // pour pouvoir supprimer en cascade l'asso lors de la suppression
            // d'un des cotes
            if (!(attr.isNavigable()
                    || hasUnidirectionalRelationOnAbstractType(reverse, model)
                    || attr.hasAssociationClass())) {
                continue;
            }

            if (!attr.hasAssociationClass()) {
                String attrType = TopiaGeneratorUtil.getSimpleName(attr.getType());
                String attrName = attr.getName();
                if (!GeneratorUtil.isNMultiplicity(attr)) {
                    body.append(""
+"        visitor.visit(this, "+getConstantName(attrName)+", "+attrType+".class, "+attrName+");\n"
+""
                    );
                } else {
                    String collectionType = TopiaGeneratorUtil.getSimpleName(
                            TopiaGeneratorUtil.getNMultiplicityInterfaceType(
                                    attr));
                    body.append(""
+"        visitor.visit(this, "+getConstantName(attrName)+", "+collectionType+".class, "+attrType+".class, "+attrName+");\n"
+""
                    );
                }
            } else {
                String assocAttrName = GeneratorUtil.getAssocAttrName(attr);
                assocAttrName = GeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
                String assocClassFQN = TopiaGeneratorUtil.getSimpleName(
                        attr.getAssociationClass().getQualifiedName());
                if (!GeneratorUtil.isNMultiplicity(attr)) {
                    body.append(""
+"        visitor.visit(this, "+getConstantName(assocAttrName)+", "+assocClassFQN+".class, "+assocAttrName+");\n"
+""
                    );
                } else {
                    String collectionType = TopiaGeneratorUtil.getNMultiplicityInterfaceType(attr);
                    body.append(""
+"        visitor.visit(this, "+getConstantName(assocAttrName)+", "+collectionType+".class, "+assocClassFQN+".class, "+assocAttrName+");\n"
+""
                    );
                }
            }
        }

        //Déclaration des attributs d'une classe d'associations
        if (clazz instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc = (ObjectModelAssociationClass) clazz;
            for (ObjectModelAttribute attr : assoc.getParticipantsAttributes()) {
                if (attr != null) {
                    String attrType = TopiaGeneratorUtil.getSimpleName(
                            attr.getType());
                    String attrName = attr.getName();
                    attrName = GeneratorUtil.toLowerCaseFirstLetter(attrName);
                    body.append(""
+"        visitor.visit(this, "+getConstantName(attrName)+", "+attrType+".class, "+attrName+");\n"
+""
                    );
                }
            }
        }
        body.append(""
+"        visitor.end(this);\n"
+""
        );

        setOperationBody(op, body.length() == 0 ? " " : body.toString());
    }


    private void generateAssociationAccessors(ObjectModelClass output,
                                              String name, String type) {
        ObjectModelOperation op;
        op = addOperation(output,
                "set" + StringUtils.capitalize(name),
                "void",
                ObjectModelModifier.PUBLIC);
        addAnnotation(output, op, Override.class.getSimpleName());
        ObjectModelParameter param = addParameter(op, type, "value");
        setDocumentation(param, "La valeur de l'attribut " + name +
                " à positionner.");
        setOperationBody(op, ""
+"\n"
+"        "+type+" _oldValue = this."+GeneratorUtil.toLowerCaseFirstLetter(name)+";\n"
+"        fireOnPreWrite(\""+name+"\", _oldValue, value);\n"
+"        this."+GeneratorUtil.toLowerCaseFirstLetter(name)+" = value;\n"
+"        fireOnPostWrite(\""+name+"\", _oldValue, value);\n"
+""
        );

        op = addOperation(output,
                "get" + StringUtils.capitalize(name),
                type,
                ObjectModelModifier.PUBLIC);
        addAnnotation(output, op, Override.class.getSimpleName());
        setOperationBody(op, ""
+"\n"
+"        return "+GeneratorUtil.toLowerCaseFirstLetter(name)+";\n"
+""
        );

    }

    /**
     * Generate entity methods which have not a public visibility. In this case, they will not be
     * generated in the EntityInterface, so they will be generated here, in the EntityAbstract
     * with abstract modifier to keep consistency with the model.
     * @param output ObjectModelClass result corresponding to the EntityAbstract
     * @param input ObjectModelClass source from ObjectModel
     */
    private void generateAbstractMethods(ObjectModelClass output,
                                         ObjectModelClass input) {
        for (ObjectModelOperation op : input.getOperations()) {
            if (log.isDebugEnabled()) {
                log.debug("clazz : " + input.getQualifiedName() +
                        " - method : " + op.getName() +
                        " - visibility : " + op.getVisibility());
            }
            String visibility = op.getVisibility();
            ObjectModelModifier visibilityModifier = ObjectModelModifier.toValue(visibility);
            if (!visibilityModifier.equals(ObjectModelModifier.PUBLIC)) {
                addOperation(output,
                        op.getName(),
                        op.getReturnType(),
                        visibilityModifier,
                        ObjectModelModifier.ABSTRACT);
            }
        }
    }

    private void generateI18n(ObjectModelClass result, String i18nPrefix,
                              ObjectModelClass clazz) {

        StringBuilder buffer = new StringBuilder(300);
        addI18n(buffer, i18nPrefix, Introspector.decapitalize(clazz.getName()));
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            //TC-20100225 only treate navigable relations
            if (attr.isNavigable()) {
                addI18n(buffer, i18nPrefix, Introspector.decapitalize(attr.getName()));
            }
        }

        //FIXME : use a block extension for java
        ObjectModelOperation op = addBlock(result, ObjectModelModifier.STATIC);
        setOperationBody(op, buffer.toString());
    }

    private void addI18n(StringBuilder buffer, String i18nPrefix,
                         String suffix) {
        buffer.append("\n    org.nuiton.i18n.I18n.n_(\"");
        buffer.append(i18nPrefix);
        buffer.append(suffix);
        buffer.append("\");");
    }


    // -------------------------------------------------------------------------
    // Impl generation
    // -------------------------------------------------------------------------

    protected boolean isAbstract(ObjectModelClass clazz) {
        if (clazz.isAbstract()) {
            return true;
        }

        //Une classe peut être abstraite si elle a des méthodes définies dans
        // ses superinterface et non implantées dans ses superclasses
        Collection<ObjectModelOperation> allInterfaceOperations =
                clazz.getAllInterfaceOperations(true);
        allInterfaceOperations.removeAll(clazz.getAllOtherOperations(true));
        for (ObjectModelOperation op : allInterfaceOperations) {
            boolean implementationFound = false;
            for (ObjectModelClass superClazz : clazz.getSuperclasses()) {
                for (ObjectModelOperation matchingOp :
                        superClazz.getOperations(op.getName())) {
                    implementationFound = op.equals(matchingOp) &&
                                          !matchingOp.isAbstract();
                    if (implementationFound) {
                        break;
                    }
                }
                if (implementationFound) {
                    break;
                }
            }
            if (!implementationFound) {
                if (log.isDebugEnabled()) {
                    log.debug(clazz.getName() + " : abstract operation " + op);
                }
                return true;
            }
        }
        return false;
    }

}
