package jaxx.runtime.swing.editor;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2020 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.border.Border;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.binding.DefaultJAXXBinding;
import jaxx.runtime.binding.SimpleJAXXObjectBinding;
import jaxx.runtime.context.DefaultJAXXContext;
import jaxx.runtime.swing.JAXXButtonGroup;
import jaxx.runtime.swing.renderer.LocaleListCellRenderer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n.n;
import static org.nuiton.i18n.I18n.t;

public class I18nEditor extends JPanel implements PropertyChangeListener, ActionListener, JAXXObject {

    /*-----------------------------------------------------------------------*/
    /*---------------- Constants for all javaBean properties ----------------*/
    /*-----------------------------------------------------------------------*/

    public static final String PROPERTY_LOCALES = "locales";
    public static final String PROPERTY_NOT_SELECTED_TOOL_TIP_TEXT = "notSelectedToolTipText";
    public static final String PROPERTY_POPUP_BORDER = "popupBorder";
    public static final String PROPERTY_POPUP_VISIBLE = "popupVisible";
    public static final String PROPERTY_RENDERER = "renderer";
    public static final String PROPERTY_SELECTED_LOCALE = "selectedLocale";
    public static final String PROPERTY_SELECTED_TOOL_TIP_TEXT = "selectedToolTipText";
    public static final String PROPERTY_SHOW_ICON = "showIcon";
    public static final String PROPERTY_SHOW_POPUP_ICON = "showPopupIcon";
    public static final String PROPERTY_SHOW_POPUP_TEXT = "showPopupText";
    public static final String PROPERTY_SHOW_TEXT = "showText";

    /*-----------------------------------------------------------------------*/
    /*------------------ Constants for all public bindings ------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String BINDING_BUTTON_ICON = "button.icon";
    public static final String BINDING_BUTTON_SELECTED = "button.selected";
    public static final String BINDING_BUTTON_TEXT = "button.text";
    public static final String BINDING_BUTTON_TOOL_TIP_TEXT = "button.toolTipText";
    public static final String BINDING_POPUP_BORDER = "popup.border";
    public static final String BINDING_RENDERER_SHOW_ICON = "renderer.showIcon";
    public static final String BINDING_RENDERER_SHOW_TEXT = "renderer.showText";

    /*-----------------------------------------------------------------------*/
    /*------------------------- Other static fields -------------------------*/
    /*-----------------------------------------------------------------------*/

    private static final String $jaxxObjectDescriptor = "H4sIAAAAAAAAAKVVz08TQRSeVtrSHyCVSCBigujFYLaRkwkEkWIFUpRAYwg96LQ7tkuGnXFnVpaL8U/wT9C7FxNvnowHzx68GP8FYzx4Nb6Z3e22sNbG9tBu34/vfe+9mW/ffEcp4aArh9jzDMe1pXVEjK07+/sPGoekKdeJaDoWl8xB/ieRRMk6ypsdu5Doar2q0ktBeqnMjjizid2VvVRFOSFPKBFtQqREl3szmkKU9jruJY+7TojaIRWH+urnj+RL88XrJEIeB3ZFaGXuX1lRJyNVlLRMiS5ApWe4RLHdAhqOZbeA75iylSkW4j4+Ik/Rc5SpojTHDoBJND94yxpD53tcokyT2VJDXOshKo6hrEFMC1KNzZu37Lv6kXOdmobENrZNShyJFgZJ3PCjo/xJQShslJg1xmjN4jXiAYkJ1aWhOjf8zqOEKZvJvbM5yluMovKccZevMcdU1GYUnBdwamij4fuijFHRZsd+9WJUfQ0qEGyfCtuEaan/M5F9TNl3VNGQTawzJrOgqT60hNWg5JQvQ1kTw/mTaFxzcqVFjaolZBQyHg6wqkPD2fmR2tRFHk4JNK1GshizrdAb5Kk6ZULpbmBWINkOVkrThn10z3ZLN7lNbDcqmtOBVdwgFM50T7Q2qrCFTnS64UoJZ7F3Z1s11mpRsqZ9vTwyFrDz1IjmY1pSmuFn3XMYkOJwFS/1xMElN6JLHt3CRB2lHFfPfrp+Vhd2weUrwvQpRVCA2vt7avLL+2/vKqEMFKD2xdjQLhWD68kdxmFFlip93tcAtc3SNuZLdZT1960lbjaG2F7gBnJQrxgdhg0s2gCRynz98HHq8edzKFlBOcqwWcEqfhNlZduBKTBqevz2qmZUOB6F7wnFDVZD8QmDWaLxZRNLPNeAycOMVzwYwmzMEDpMGtlPvyb33q6Gg0gAsZm/hkfDSB2gtGVTyyZaEQOxi1XAPBfENVkkanEyl1C/OR4czWX9vRrXac8lL3uu+lnXxNVTReNcHxQn1IMhcNK+ZvVBuTEASobYGCRGvVmW/bHOYQnqCjeOrMTDlgaAHZFBgwf/j1F4wpqu2MEWvITMPliLA2BlNRYOxHQIoBGr/wkYBGM0VOehcJT50ZBM8rL3ZdkH6g8SOKddfgkAAA==";
    private static final Log log = LogFactory.getLog(I18nEditor.class);
    private static final long serialVersionUID = 1L;

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Internal states ---------------------------*/
    /*-----------------------------------------------------------------------*/

    protected List<Object> $activeBindings = new ArrayList<Object>();
    protected Map<String, Object> $bindingSources = new HashMap<String, Object>();
    protected final Map<String, JAXXBinding> $bindings = new TreeMap<String, JAXXBinding>();
    protected Map<String, Object> $objectMap = new HashMap<String, Object>();
    protected Map<?, ?> $previousValues = new HashMap<Object, Object>();
    private boolean allComponentsCreated;
    protected final JAXXContext delegateContext = new DefaultJAXXContext();

    /*-----------------------------------------------------------------------*/
    /*------------------------ Protected components  ------------------------*/
    /*-----------------------------------------------------------------------*/

    protected JToggleButton button;
    protected I18nEditor content = this;
    protected I18nEditorHandler handler;
    protected JAXXButtonGroup indexes;
    protected List<Locale> locales;
    protected String notSelectedToolTipText;
    protected JPopupMenu popup;
    protected Border popupBorder;
    protected JLabel popupLabel;
    protected Boolean popupVisible;
    protected LocaleListCellRenderer renderer;
    protected Locale selectedLocale;
    protected String selectedToolTipText;
    protected Boolean showIcon;
    protected Boolean showPopupIcon;
    protected Boolean showPopupText;
    protected Boolean showText;

    /*-----------------------------------------------------------------------*/
    /*---------------------- Raw body code from script ----------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String DEFAULT_SELECTED_TOOLTIP = n("i18neditor.selected");
    public static final String DEFAULT_NOT_SELECTED_TOOLTIP = n("i18neditor.unselected");
    public static final String LOCALES_PROPERTY = "locales";
    public static final String SELECTED_LOCALE_PROPERTY = "selectedLocale";
    public static final String SHOW_ICON_PROPERTY = "showIcon";
    public static final String SHOW_TEXT_PROPERTY = "showText";
    public static final String SHOW_POPUP_ICON_PROPERTY = "showPopupIcon";
    public static final String SHOW_POPUP_TEXT_PROPERTY = "showPopupText";
    public static final String POPUP_BORDER_PROPERTY = "popupBorder";
    public static final String POPUP_VISIBLE_PROPERTY = "popupVisible";
    
    
    
    
    void $afterCompleteSetup() {
        handler.$afterCompleteSetup();
    }
    
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        handler.propertyChange(evt);
    }
    
    @Override
    public void actionPerformed(ActionEvent event) {
        handler.actionPerformed(event);
    }
    
    public void loadI18nBundles() {
        handler.loadI18nBundles();
    }
    
    protected void rebuildPopup() {
        handler.rebuildPopup();
    }
    
    protected String getTip(Locale l) {
        return handler.getTip(l);
    }
    
    protected String getSelectedTip(Locale l) {
        return handler.getSelectedTip(l);
    }
    
    protected String getNotSelectedTip(Locale l) {
        return handler.getNotSelectedTip(l);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Constructors  ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public I18nEditor(LayoutManager param0) {
        super(param0);
        $initialize();
    }

    public I18nEditor(JAXXContext parentContext, LayoutManager param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public I18nEditor(LayoutManager param0, boolean param1) {
        super(param0 ,param1);
        $initialize();
    }

    public I18nEditor(JAXXContext parentContext, LayoutManager param1, boolean param2) {
        super(param1 ,param2);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public I18nEditor() {
        $initialize();
    }

    public I18nEditor(JAXXContext parentContext) {
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    public I18nEditor(boolean param0) {
        super(param0);
        $initialize();
    }

    public I18nEditor(JAXXContext parentContext, boolean param1) {
        super(param1);
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Statics methods ---------------------------*/
    /*-----------------------------------------------------------------------*/

    public static JAXXObjectDescriptor $getJAXXObjectDescriptor() {
        return JAXXUtil.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------- JAXXObject implementation ----------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public void applyDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).applyDataBinding();
        }
        processDataBinding($binding);
    }

    @Override
    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        super.firePropertyChange(propertyName, oldValue, newValue);
    }

    @Override
    public Map<String, Object> get$objectMap() {
        return $objectMap;
    }

    @Override
    public JAXXBinding getDataBinding(String bindingId) {
        return $bindings.get(bindingId);
    }

    @Override
    public JAXXBinding[] getDataBindings() {
        return $bindings.values().toArray(new JAXXBinding[$bindings.size()]);
    }

    @Override
    public Object getObjectById(String id) {
        return $objectMap.get(id);
    }

    @Override
    public void processDataBinding(String $binding, boolean $force) {
        if (!$force && $activeBindings.contains($binding)) { 
            return;
        }
        $activeBindings.add($binding);
        try {
            if (allComponentsCreated && $bindings.containsKey($binding)) {
                getDataBinding($binding).processDataBinding();
            }
        } finally {
            $activeBindings.remove($binding);
        }
    }

    @Override
    public void processDataBinding(String $binding) {
        processDataBinding($binding, false);
    }

    @Override
    public void registerDataBinding(JAXXBinding binding) {
        $bindings.put(binding.getId(), binding);
    }

    @Override
    public void removeDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).removeDataBinding();
        }
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- JAXXContext implementation  ---------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return delegateContext.getContextValue(clazz, null);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        return delegateContext.getContextValue(clazz, name);
    }

    @Override
    public JAXXContext getDelegateContext() {
        return delegateContext;
    }

    @Override
    public <O extends Container> O getParentContainer(Class<O> clazz) {
        return SwingUtil.getParentContainer(this, clazz);
    }

    @Override
    public <O extends Container> O getParentContainer(Object source, Class<O> clazz) {
        return SwingUtil.getParentContainer(source, clazz);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz) {
        delegateContext.removeContextValue(clazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz, String name) {
        delegateContext.removeContextValue(clazz, name);
    }

    @Override
    public <T> void setContextValue(T o) {
        delegateContext.setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        delegateContext.setContextValue(o, name);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Event methods ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public void doFocusGained__on__content(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        button.requestFocus();
    }

    public void doFocusLost__on__content(FocusEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setPopupVisible(false);
    }

    public void doItemStateChanged__on__button(ItemEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        if (event.getStateChange()  == ItemEvent.SELECTED) { setPopupVisible(true); } else { popupVisible = false; }
    }

    public void doPopupMenuCanceled__on__popup(PopupMenuEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        button.setSelected(false);
    }

    public void doPopupMenuWillBecomeInvisible__on__popup(PopupMenuEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        button.setSelected(false);
    }

    public void doStateChanged__on__indexes(ChangeEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        log.info(indexes.getSelectedValue());
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public acessor methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public JToggleButton getButton() {
        return button;
    }

    public I18nEditorHandler getHandler() {
        return handler;
    }

    public JAXXButtonGroup getIndexes() {
        return indexes;
    }

    public List<Locale> getLocales() {
        return locales;
    }

    public String getNotSelectedToolTipText() {
        return notSelectedToolTipText;
    }

    public JPopupMenu getPopup() {
        return popup;
    }

    public Border getPopupBorder() {
        return popupBorder;
    }

    public JLabel getPopupLabel() {
        return popupLabel;
    }

    public Boolean getPopupVisible() {
        return popupVisible;
    }

    public LocaleListCellRenderer getRenderer() {
        return renderer;
    }

    public Locale getSelectedLocale() {
        return selectedLocale;
    }

    public String getSelectedToolTipText() {
        return selectedToolTipText;
    }

    public Boolean getShowIcon() {
        return showIcon;
    }

    public Boolean getShowPopupIcon() {
        return showPopupIcon;
    }

    public Boolean getShowPopupText() {
        return showPopupText;
    }

    public Boolean getShowText() {
        return showText;
    }

    public Boolean isPopupVisible() {
        return popupVisible !=null && popupVisible;
    }

    public Boolean isShowIcon() {
        return showIcon !=null && showIcon;
    }

    public Boolean isShowPopupIcon() {
        return showPopupIcon !=null && showPopupIcon;
    }

    public Boolean isShowPopupText() {
        return showPopupText !=null && showPopupText;
    }

    public Boolean isShowText() {
        return showText !=null && showText;
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public mutator methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public void setLocales(List<Locale> locales) {
        List<Locale> oldValue = this.locales;
        this.locales = locales;
        firePropertyChange(PROPERTY_LOCALES, oldValue, locales);
    }

    public void setNotSelectedToolTipText(String notSelectedToolTipText) {
        String oldValue = this.notSelectedToolTipText;
        this.notSelectedToolTipText = notSelectedToolTipText;
        firePropertyChange(PROPERTY_NOT_SELECTED_TOOL_TIP_TEXT, oldValue, notSelectedToolTipText);
    }

    public void setPopupBorder(Border popupBorder) {
        Border oldValue = this.popupBorder;
        this.popupBorder = popupBorder;
        firePropertyChange(PROPERTY_POPUP_BORDER, oldValue, popupBorder);
    }

    public void setPopupVisible(Boolean popupVisible) {
        Boolean oldValue = this.popupVisible;
        this.popupVisible = popupVisible;
        firePropertyChange(PROPERTY_POPUP_VISIBLE, oldValue, popupVisible);
    }

    public void setRenderer(LocaleListCellRenderer renderer) {
        LocaleListCellRenderer oldValue = this.renderer;
        this.renderer = renderer;
        firePropertyChange(PROPERTY_RENDERER, oldValue, renderer);
    }

    public void setSelectedLocale(Locale selectedLocale) {
        Locale oldValue = this.selectedLocale;
        this.selectedLocale = selectedLocale;
        firePropertyChange(PROPERTY_SELECTED_LOCALE, oldValue, selectedLocale);
    }

    public void setSelectedToolTipText(String selectedToolTipText) {
        String oldValue = this.selectedToolTipText;
        this.selectedToolTipText = selectedToolTipText;
        firePropertyChange(PROPERTY_SELECTED_TOOL_TIP_TEXT, oldValue, selectedToolTipText);
    }

    public void setShowIcon(Boolean showIcon) {
        Boolean oldValue = this.showIcon;
        this.showIcon = showIcon;
        firePropertyChange(PROPERTY_SHOW_ICON, oldValue, showIcon);
    }

    public void setShowPopupIcon(Boolean showPopupIcon) {
        Boolean oldValue = this.showPopupIcon;
        this.showPopupIcon = showPopupIcon;
        firePropertyChange(PROPERTY_SHOW_POPUP_ICON, oldValue, showPopupIcon);
    }

    public void setShowPopupText(Boolean showPopupText) {
        Boolean oldValue = this.showPopupText;
        this.showPopupText = showPopupText;
        firePropertyChange(PROPERTY_SHOW_POPUP_TEXT, oldValue, showPopupText);
    }

    public void setShowText(Boolean showText) {
        Boolean oldValue = this.showText;
        this.showText = showText;
        firePropertyChange(PROPERTY_SHOW_TEXT, oldValue, showText);
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Components creation methods ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected void addChildrenToContent() {
        if (!allComponentsCreated) {
            return;
        }
        add(renderer);
        add(button, BorderLayout.CENTER);
    }

    protected void addChildrenToPopup() {
        if (!allComponentsCreated) {
            return;
        }
        popup.add(popupLabel);
    }

    protected void createButton() {
        $objectMap.put("button", button = new JToggleButton());
        
        button.setName("button");
        button.setFocusPainted(false);
        button.setFocusable(true);
        button.addItemListener(JAXXUtil.getEventListener(ItemListener.class, "itemStateChanged", this, "doItemStateChanged__on__button"));
    }

    protected void createHandler() {
        $objectMap.put("handler", handler = new I18nEditorHandler(this));
    }

    protected void createIndexes() {
        $objectMap.put("indexes", indexes = new JAXXButtonGroup());
        
        indexes.addChangeListener(JAXXUtil.getEventListener(ChangeListener.class, "stateChanged", this, "doStateChanged__on__indexes"));
    }

    protected void createLocales() {
        $objectMap.put("locales", locales = null);
    }

    protected void createNotSelectedToolTipText() {
        $objectMap.put("notSelectedToolTipText", notSelectedToolTipText = null);
    }

    protected void createPopup() {
        $objectMap.put("popup", popup = new JPopupMenu());
        
        popup.setName("popup");
        popup.addPopupMenuListener(JAXXUtil.getEventListener(PopupMenuListener.class, "popupMenuCanceled", this, "doPopupMenuCanceled__on__popup"));
        popup.addPopupMenuListener(JAXXUtil.getEventListener(PopupMenuListener.class, "popupMenuWillBecomeInvisible", this, "doPopupMenuWillBecomeInvisible__on__popup"));
    }

    protected void createPopupBorder() {
        $objectMap.put("popupBorder", popupBorder = new TitledBorder(t("i18neditor.popup.title")));
    }

    protected void createPopupLabel() {
        $objectMap.put("popupLabel", popupLabel = new JLabel());
        
        popupLabel.setName("popupLabel");
        popupLabel.setEnabled(false);
        popupLabel.setText(t("i18neditor.empty.locales"));
    }

    protected void createPopupVisible() {
        $objectMap.put("popupVisible", popupVisible = Boolean.FALSE);
    }

    protected void createRenderer() {
        $objectMap.put("renderer", renderer = new LocaleListCellRenderer(showIcon , showText));
        
        renderer.setName("renderer");
    }

    protected void createSelectedLocale() {
        $objectMap.put("selectedLocale", selectedLocale = Locale.getDefault());
    }

    protected void createSelectedToolTipText() {
        $objectMap.put("selectedToolTipText", selectedToolTipText = null);
    }

    protected void createShowIcon() {
        $objectMap.put("showIcon", showIcon = Boolean.TRUE);
    }

    protected void createShowPopupIcon() {
        $objectMap.put("showPopupIcon", showPopupIcon = Boolean.TRUE);
    }

    protected void createShowPopupText() {
        $objectMap.put("showPopupText", showPopupText = Boolean.TRUE);
    }

    protected void createShowText() {
        $objectMap.put("showText", showText = Boolean.TRUE);
    }

    /*-----------------------------------------------------------------------*/
    /*------------------------ Internal jaxx methods ------------------------*/
    /*-----------------------------------------------------------------------*/

    private void $completeSetup() {
        allComponentsCreated = true;
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        addChildrenToContent();
        addChildrenToPopup();
        
        // apply 7 data bindings
        JAXXUtil.applyDataBinding(this, $bindings.keySet());
        
        $afterCompleteSetup();
    }

    private void $initialize() {
        if (allComponentsCreated) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        $objectMap.put("content", content);
        createHandler();
        createSelectedToolTipText();
        createNotSelectedToolTipText();
        createPopupBorder();
        createShowText();
        createShowIcon();
        createShowPopupText();
        createShowPopupIcon();
        createPopupVisible();
        createLocales();
        createSelectedLocale();
        createRenderer();
        createPopup();
        createPopupLabel();
        createButton();
        createIndexes();
        // inline creation of content
        setName("content");
        setLayout(new BorderLayout());
        content.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusGained", this, "doFocusGained__on__content"));
        content.addFocusListener(JAXXUtil.getEventListener(FocusListener.class, "focusLost", this, "doFocusLost__on__content"));
        
        // registers 7 data bindings
        $registerDefaultBindings();
        $completeSetup();
    }

    private void $registerDefaultBindings() {
        // register 7 data bindings
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_RENDERER_SHOW_ICON, true ,"showIcon") {
        
            @Override
            public void processDataBinding() {
                renderer.setShowIcon(isShowIcon());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_RENDERER_SHOW_TEXT, true ,"showText") {
        
            @Override
            public void processDataBinding() {
                renderer.setShowText(isShowText());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_POPUP_BORDER, true ,"popupBorder") {
        
            @Override
            public void processDataBinding() {
                popup.setBorder(getPopupBorder());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_ICON, true ,"selectedLocale") {
        
            @Override
            public void processDataBinding() {
                button.setIcon(renderer.getIcon(getSelectedLocale()));
            }
        });
        registerDataBinding(new DefaultJAXXBinding(this, BINDING_BUTTON_SELECTED, true) {
        
            @Override
            public void applyDataBinding() {
                if (popup != null) {
                    $bindingSources.put("popup", popup);
                    popup.addComponentListener( JAXXUtil.getEventListener(ComponentListener.class, this, "$pr$u0"));
                }
            }
        
            @Override
            public void processDataBinding() {
                if (popup != null) {
                    button.setSelected(popup.isVisible());
                }
            }
        
            @Override
            public void removeDataBinding() {
                if (popup != null) {
                    JPopupMenu $target = (JPopupMenu) $bindingSources.remove("popup");
                    if ($target != null) {
                        $target.removeComponentListener( JAXXUtil.getEventListener(ComponentListener.class, this, "$pr$u0"));
                    }
                }
            }
        
            public void $pr$u0(ComponentEvent event) {
                if (log.isDebugEnabled()) {
                    log.debug(event);
                }
                propertyChange(null);
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_TEXT, true ,"selectedLocale") {
        
            @Override
            public void processDataBinding() {
                button.setText(t(SwingUtil.getStringValue(renderer.getText(getSelectedLocale()))));
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BUTTON_TOOL_TIP_TEXT, true ,"selectedLocale") {
        
            @Override
            public void processDataBinding() {
                button.setToolTipText(t(getTip(getSelectedLocale())));
            }
        });
    }

}
