package org.nuiton.jaxx.widgets.datetime;

/*
 * #%L
 * JAXX :: Widgets DateTime
 * %%
 * Copyright (C) 2008 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import jaxx.runtime.spi.UIHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;

import javax.swing.JLabel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.plaf.basic.BasicSliderUI;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;
import java.util.Dictionary;
import java.util.GregorianCalendar;
import java.util.Hashtable;
import java.util.Map;

/**
 * Created on 9/9/14.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.12
 */
public class DateTimeEditorHandler implements UIHandler<DateTimeEditor> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DateTimeEditorHandler.class);

    protected DateTimeEditor ui;

    @Override
    public void beforeInit(DateTimeEditor ui) {

        DateTimeEditorModel model = new DateTimeEditorModel();
        ui.setContextValue(model);

        this.ui = ui;

    }

    @Override
    public void afterInit(DateTimeEditor ui) {

        ui.getMinuteEditor().setEditor(new JSpinner.DateEditor(ui.getMinuteEditor(), "mm"));
        ui.getHourEditor().setEditor(new JSpinner.DateEditor(ui.getHourEditor(), "HH"));

        // create slider labels
        Map<Integer, JLabel> labelTable = new Hashtable<Integer, JLabel>();
        for (int i = 0; i < 25; i += 2) {
            labelTable.put(i * 60, new JLabel(i + ""));
        }
        JSlider slider = ui.getSlider();
        slider.setLabelTable((Dictionary<?, ?>) labelTable);

        MouseAdapter m = new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                // set the value
                int value = getSliderValue(e);
                JSlider slider = (JSlider) e.getComponent();
                slider.setValueIsAdjusting(true);
                slider.setValue(value);
                slider.setValueIsAdjusting(false);
                showToolTip(e);
                e.consume();
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseEntered(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseMoved(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                showToolTip(e);
            }

            @Override
            public void mouseWheelMoved(MouseWheelEvent e) {
                JSlider slider = (JSlider) e.getComponent();

                // compute new value
                int nb = e.getWheelRotation();
                int value = slider.getValue() - nb;

                // set the value
                slider.setValueIsAdjusting(true);
                slider.setValue(value);
                slider.setValueIsAdjusting(false);
                e.consume();
            }

            int getSliderValue(MouseEvent e) {
                JSlider slider = (JSlider) e.getSource();
                int value = -1;
                if (slider.getUI() instanceof BasicSliderUI) {
                    BasicSliderUI ui = (BasicSliderUI) slider.getUI();
                    value = slider.getOrientation() == JSlider.HORIZONTAL
                            ? ui.valueForXPosition(e.getX())
                            : ui.valueForYPosition(e.getY());
                }
                return value;
            }

            void showToolTip(MouseEvent e) {

                int value = getSliderValue(e);
                if (value == -1) {
                    return;
                }
                int h = value / 60;
                int m = value % 60;

                String text = "";
                if (h < 10) {
                    text = "0";
                }
                text += h + " : ";
                if (m < 10) {
                    text += "0";
                }
                text += m;

                JSlider source = (JSlider) e.getSource();
                source.setToolTipText(text);

            }
        };
        slider.addMouseListener(m);
        slider.addMouseMotionListener(m);
        slider.addMouseWheelListener(m);

    }

    public void init(DateTimeEditor ui) {

        DateTimeEditorModel model = ui.getModel();

        Object bean = model.getBean();

        if (bean != null) {

            if (model.getPropertyDayDate() != null) {

                Method dayDateMutator = BeanUtil.getMutator(bean, model.getPropertyDayDate());
                Preconditions.checkNotNull(dayDateMutator, "could not find mutator for " + model.getPropertyDayDate());
                // When model day date changed, let's push it back in bean
                model.addPropertyChangeListener(
                        DateTimeEditorModel.PROPERTY_DAY_DATE,
                        new ModelPropertyChangeListener(model, dayDateMutator));

            }

            if (model.getPropertyTimeDate() != null) {

                Method timeDateMutator = BeanUtil.getMutator(bean, model.getPropertyTimeDate());
                Preconditions.checkNotNull(timeDateMutator, "could not find mutator for " + model.getPropertyTimeDate());

                // When model time date changed, let's push it back in bean
                model.addPropertyChangeListener(
                        DateTimeEditorModel.PROPERTY_TIME_DATE,
                        new ModelPropertyChangeListener(model, timeDateMutator));

            }

            if (model.getPropertyDate() != null) {

                Method fullDateMutator = BeanUtil.getMutator(bean, model.getPropertyDate());
                Preconditions.checkNotNull(fullDateMutator, "could not find mutator for " + model.getPropertyDate());

                // When model full date changed, let's push it back in bean
                model.addPropertyChangeListener(
                        DateTimeEditorModel.PROPERTY_DATE,
                        new ModelPropertyChangeListener(model, fullDateMutator));

            }

        }

    }

    protected final Calendar calendarMinute = new GregorianCalendar();

    protected final Calendar calendarHour = new GregorianCalendar();

    public Date getMinuteModelValue(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
        }
        calendarMinute.setTime(incomingDate);
        calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
        incomingDate = calendarMinute.getTime();
        return incomingDate;
    }

    public Date getHourModelValue(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
        }
        calendarHour.setTime(incomingDate);
        calendarHour.set(Calendar.MINUTE, 0);
        incomingDate = calendarHour.getTime();
        return incomingDate;
    }

    public void setHours(Date hourDate) {

        DateTimeEditorModel model = ui.getModel();

        Date oldTimeDate = model.getTimeDate();

        if (oldTimeDate == null) {
            return;
        }

        calendarHour.setTime(hourDate);
        int newHour = calendarHour.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarHour.get(Calendar.MINUTE);

        int oldHour = model.getHour(oldTimeDate);
        int oldMinute = model.getMinute(oldTimeDate);

        if (oldHour == newHour && oldMinute == newMinute) {

            // do nothing, same data
            if (log.isDebugEnabled()) {
                log.debug("Do not update time model , stay on same time = " + oldHour + ":" + oldMinute);
            }
            return;
        }

        // by default stay on same hour
        int hour = newHour;

        // by default, use the new minute data
        int minute = oldMinute;

        if (log.isDebugEnabled()) {
            log.debug("hh:mm (old from dateModel)   = " + oldHour + ":" + oldMinute);
            log.debug("hh:mm (new from hourModel) = " + newHour + ":" + newMinute);
        }

        Integer dayAdjust = null;

        if (newHour == 0 && oldHour == 23) {

            // add a day
            dayAdjust = +1;

        } else if (newHour == 23 && oldHour == 0) {

            // decrease a day
            dayAdjust = -1;

        }

        if (dayAdjust != null) {

            Date oldDayDate = model.getDayDate();

            calendarHour.setTime(oldDayDate);
            calendarHour.add(Calendar.DAY_OF_YEAR, dayAdjust);

            if (log.isDebugEnabled()) {
                log.debug("Update day to " + calendarHour.get(Calendar.DAY_OF_YEAR));
            }

            Date newDayDate = calendarHour.getTime();
            model.setDayDate(newDayDate);

        }

        // change time
        model.setTimeInMinutes(hour * 60 + minute);

    }

    public void setMinutes(Date minuteDate) {

        DateTimeEditorModel model = ui.getModel();

        Date oldTimeDate = model.getTimeDate();

        if (oldTimeDate == null) {
            return;
        }

        calendarMinute.setTime(minuteDate);
        int newHour = calendarMinute.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarMinute.get(Calendar.MINUTE);

        int oldHour = model.getHour(oldTimeDate);
        int oldMinute = model.getMinute(oldTimeDate);

        if (oldHour == newHour && oldMinute == newMinute) {

            // do nothing, same data
            if (log.isDebugEnabled()) {
                log.debug("Do not update time model , stay on same time = " + oldHour + ":" + oldMinute);
            }
            return;
        }

        // by default stay on same hour
        int hour = oldHour;

        // by default, use the new minute data
        int minute = newMinute;

        if (log.isDebugEnabled()) {
            log.debug("hh:mm (old from dateModel)   = " + oldHour + ":" + oldMinute);
            log.debug("hh:mm (new from minuteModel) = " + newHour + ":" + newMinute);
        }

        Integer dayAdjust = null;

        if (newMinute == 0) {

            // minute pass to zero (check if a new hour is required)
            if (newHour == 1) {

                if (oldHour == 23) {

                    // on next day
                    dayAdjust = +1;

                }
                hour = (oldHour + 1) % 24;

            }
        } else if (newMinute == 59) {

            // minute pass to 59 (check if a new hour is required)

            if (newHour == 23) {

                if (oldHour == 0) {

                    dayAdjust = -1;

                }

                // decrease hour
                hour = (oldHour - 1) % 24;

            }
        }

        if (dayAdjust != null) {

            Date oldDayDate = model.getDayDate();

            calendarHour.setTime(oldDayDate);
            calendarHour.add(Calendar.DAY_OF_YEAR, dayAdjust);

            if (log.isDebugEnabled()) {
                log.debug("Update day to " + calendarHour.get(Calendar.DAY_OF_YEAR));
            }

            Date newDayDate = calendarHour.getTime();
            model.setDayDate(newDayDate);

        }

        // date has changed
        if (log.isDebugEnabled()) {
            log.debug("Update time model to hh:mm = " + hour + ":" + minute);
        }

        model.setTimeInMinutes(hour * 60 + minute);
    }

    private class ModelPropertyChangeListener implements PropertyChangeListener {

        private final DateTimeEditorModel model;

        private final Method mutator;

        private ModelPropertyChangeListener(DateTimeEditorModel model, Method mutator) {
            this.model = model;
            this.mutator = mutator;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            if (!model.isValueIsAdjusting()) {

                Object newValue = evt.getNewValue();

                try {

//                    valueModelIsAdjusting = true;
//                    try {
                    mutator.invoke(model.getBean(), newValue);
//                    } finally {
//                        valueModelIsAdjusting = false;
//                    }

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }

            }

        }
    }

}
