/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.runtime.swing;

import javax.swing.AbstractListModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.ListModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class JAXXList extends JList {

    private static final long serialVersionUID = 1L;

    public class JAXXListModel extends AbstractListModel {

        private List<Item> items;
        private static final long serialVersionUID = -1598924187490122036L;

        public JAXXListModel(List<Item> items) {
            this.items = items;

            PropertyChangeListener listener = new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent e) {
                    if (e.getPropertyName().equals(Item.SELECTED_PROPERTY)) {
                        Item item = (Item) e.getSource();
                        int itemIndex = JAXXListModel.this.items.indexOf(item);
                        int[] oldSelection = getSelectedIndices();
                        int[] newSelection;
                        int index = -1;
                        for (int i = 0; i < oldSelection.length; i++) {
                            if (oldSelection[i] == itemIndex) {
                                index = i;
                                break;
                            }
                        }
                        if (item.isSelected()) {
                            if (index != -1) // it was already selected
                            {
                                return;
                            }
                            newSelection = new int[oldSelection.length + 1];
                            System.arraycopy(oldSelection, 0, newSelection, 0, oldSelection.length);
                            newSelection[newSelection.length - 1] = itemIndex;
                        } else {
                            if (index == -1) // it already wasn't selected
                            {
                                return;
                            }
                            newSelection = new int[oldSelection.length - 1];
                            System.arraycopy(oldSelection, 0, newSelection, 0, index);
                            System.arraycopy(oldSelection, index + 1, newSelection, index, oldSelection.length - 1 - index);
                        }
                        setSelectedIndices(newSelection);
                    } else {
                        for (int i = 0; i < getSize(); i++) {
                            if (getElementAt(i) == ((Item) e.getSource()).getValue()) {
                                fireContentsChanged(JAXXListModel.this, i, i);
                                if (isSelectedIndex(i)) {
                                    fireSelectionValueChanged(i, i, false);
                                }
                                return;
                            }
                        }
                    }
                }
            };
            for (Item item : items) {
                item.addPropertyChangeListener(listener);
            }
        }

        @Override
        public Object getElementAt(int i) {
            return items.get(i).getValue();
        }

        @Override
        public int getSize() {
            return items.size();
        }
    }

    public JAXXList() {
        setCellRenderer(new DefaultListCellRenderer() {

            private static final long serialVersionUID = 1L;

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                ListModel model = list.getModel();
                if (model instanceof JAXXListModel) {
                    Item item = ((JAXXListModel) model).items.get(index);
                    String label = item.getLabel();
                    if (label != null) {
                        value = label;
                    }
                }
                return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            }
        });

        addListSelectionListener(new ListSelectionListener() {

            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListModel model = getModel();
                if (model instanceof JAXXListModel) {
                    List<Item> items = ((JAXXListModel) model).items;
                    for (int i = items.size() - 1; i >= 0; i--) {
                        boolean selected = isSelectedIndex(i);
                        Item item = items.get(i);
                        if (selected != item.isSelected()) {
                            item.setSelected(selected);
                        }
                    }
                }
            }
        });
    }

    // this way we can keep it marked protected and still allow code in this file to call it
    @Override
    protected void fireSelectionValueChanged(int firstIndex, int lastIndex, boolean isAdjusting) {
        super.fireSelectionValueChanged(firstIndex, lastIndex, isAdjusting);
    }

    public void setSelectedValue(Object value) {
        super.setSelectedValue(value, true);
    }

    public void setItems(List<Item> items) {
        setModel(new JAXXListModel(items));
        List<Integer> selectedIndexList = new ArrayList<Integer>();
        for (int i = 0; i < items.size(); i++) {
            if (items.get(i).isSelected()) {
                selectedIndexList.add(i);
            }
        }
        int[] selectedIndices = new int[selectedIndexList.size()];
        for (int i = 0; i < selectedIndexList.size(); i++) {
            selectedIndices[i] = selectedIndexList.get(i);
        }
        setSelectedIndices(selectedIndices);
    }

    /**
     * Fill a list model with some datas, and select after all the given object
     *
     * @param data    data ot inject in combo
     * @param selects the objects to select in list after reflling his model
     */
    public void fillList(Collection<?> data, Collection<?> selects) {
        if (selects == null) {
            selects = java.util.Collections.EMPTY_LIST;
        }
        List<Item> items = new ArrayList<Item>();
        for (Object o : data) {
            boolean selected = false;
            for (Object select : selects) {
                if (selected = o.equals(select)) {
                    break;
                }
            }
            items.add(new Item(o.toString(), o.toString(), o, selected));
        }
        setItems(items);
    }

    /**
     * Fill a list model with some datas, and select after all the given object
     *
     * @param data   data ot inject in combo
     * @param select object to select in list after reflling his model
     */
    public void fillList(Collection<?> data, Object select) {
        List<Item> items = new ArrayList<Item>();
        for (Object o : data) {
            boolean selected = o.equals(select);
            items.add(new Item(o.toString(), o.toString(), o, selected));
        }
        setItems(items);
    }

    /**
     * Fill a list model with some datas, and select after all the given object
     *
     * @param data       data ot inject in combo
     * @param select     object to select in list after reflling his model
     * @param methodName method to invoke to display data's name
     */
    public void fillList(Collection<?> data, Object select, String methodName) {
        // prepare method to use
        Method m = null;

        List<Item> items = new ArrayList<Item>();
        for (Object o : data) {
            boolean selected = o.equals(select);
            if (m == null) {
                try {
                    m = o.getClass().getMethod(methodName);
                    m.setAccessible(true);
                } catch (NoSuchMethodException e) {
                    throw new IllegalArgumentException("could not find method " + methodName + " on " + o.getClass());
                }
            }
            try {
                items.add(new Item(o.toString(), (String) m.invoke(o), o, selected));
            } catch (SecurityException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (IllegalAccessException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (IllegalArgumentException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            }
        }
        setItems(items);
    }

    /**
     * Fill a list model with some datas, and select after all the given object
     *
     * @param data       data ot inject in combo
     * @param selects    the objects to select in list after reflling his model
     * @param methodName method to invoke to display data's name
     */
    public void fillList(Collection<?> data, Collection<?> selects, String methodName) {
        // prepare method to use
        Method m = null;

        List<Item> items = new ArrayList<Item>();
        for (Object o : data) {
            boolean selected = selects.contains(o);
            if (m == null) {
                try {
                    m = o.getClass().getMethod(methodName);
                    m.setAccessible(true);
                } catch (NoSuchMethodException e) {
                    throw new IllegalArgumentException("could not find method " + methodName + " on " + o.getClass());
                }
            }
            try {
                items.add(new Item(o.toString(), (String) m.invoke(o), o, selected));
            } catch (SecurityException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (IllegalAccessException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (IllegalArgumentException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            }
        }
        setItems(items);
    }

    /**
     * Set the selected Objects
     *
     * @param values    Objects must be selected in the list
     */
    public void setSelectedValues(Object[] values) {
        if (values != null){
            List<Integer> selectedIndices = new ArrayList<Integer>();
            ListModel model = getModel();
            for (int i = 0; i < model.getSize(); i++) {
                Object o = model.getElementAt(i);
                for (Object value : values) {
                    if (o.equals(value)) {
                        selectedIndices.add(i);
                        break;
                    }
                }
            }
            int[] ints = new int[selectedIndices.size()];
            for (int i = 0; i < ints.length; i++) {
                ints[i] = selectedIndices.get(i).intValue();
            }
            setSelectedIndices(ints);
        }
        else{
            // No selection if values is null
            setSelectedIndex(-1);
        }
    }
}
