/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: MethodDescriptor.java 2225 2011-02-19 20:15:00Z tchemit $
 * $HeadURL: https://nuiton.org/svn/jaxx/tags/jaxx-2.8.7/jaxx-compiler/src/main/java/jaxx/compiler/reflect/MethodDescriptor.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.reflect;

import jaxx.compiler.JAXXCompiler;

import java.util.Arrays;

/**
 * Mirrors the class <code>java.lang.ref.Method</code>.  JAXX uses <code>ClassDescriptor</code> instead of <code>Class</code>
 * almost everywhere so that it can handle circular dependencies (there can't be a <code>Class</code> object for an uncompiled
 * JAXX or Java source file, and a compiler must be allow references to symbols in uncompiled source files in order to handle
 * circular dependencies).
 */
public class MethodDescriptor extends MemberDescriptor {

    private String returnType;
    private String[] parameterTypes;

    public MethodDescriptor(String name, int modifiers, String returnType, String[] parameterTypes, ClassLoader classLoader) {
        super(name, modifiers, classLoader);
        this.returnType = returnType;
        this.parameterTypes = parameterTypes;
        if (JAXXCompiler.STRICT_CHECKS && Arrays.asList(parameterTypes).contains(null)) {
            throw new NullPointerException(name);
        }
    }

    public ClassDescriptor getReturnType() {
        try {
            //TC 20090228 : fix bug when no return type defined (constructor method)
            if (returnType == null) {
                return null;
            }
            return ClassDescriptorHelper.getClassDescriptor(returnType);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("could not find return type " + returnType, e);
        }
    }

    public ClassDescriptor[] getParameterTypes() {
        ClassDescriptor[] result = new ClassDescriptor[parameterTypes.length];
        try {
            for (int i = 0; i < result.length; i++) {
                if (parameterTypes[i] != null) {
                    result[i] = ClassDescriptorHelper.getClassDescriptor(parameterTypes[i], getClassLoader());
                }
            }
            return result;
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("could not find the parameter types " + Arrays.toString(parameterTypes), e);
        }
    }
}
