/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: I18nHelper.java 2783 2014-02-04 07:27:32Z tchemit $
 * $HeadURL: https://nuiton.org/svn/jaxx/tags/jaxx-2.8.2/jaxx-compiler/src/main/java/jaxx/compiler/I18nHelper.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler;

import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;

import java.util.Arrays;
import java.util.List;

/**
 * I18n methods to add {@link I18n#t(String, Object...)} method on some attributes.
 * <p/>
 * Make sure to set an i18nable compiler to have his {@link CompilerConfiguration#isI18nable()} returning true.
 *
 * @author tchemit <chemit@codelutin.com>
 */
public class I18nHelper {

    protected static final Log log = LogFactory.getLog(I18nHelper.class);

    public static final List<String> I18N_ATTRIBUTES = Arrays.asList("text", "title", "toolTipText");

    /**
     * Test if we have an active i18n attribute (says an i18n attribute on a i18neable compiler).
     *
     * @param attributeName name of attribute to test
     * @param compiler      current used compiler (contains options)
     * @return <code>true</code> if wa have an active i18n attribute, <code>false</code> otherwise.
     */
    public static boolean isI18nableAttribute(String attributeName, JAXXCompiler compiler) {
        return compiler.getConfiguration().isI18nable() && isI18nAttribute(attributeName);
    }

    /**
     * Test if we have an i18n attribute.
     *
     * @param attributeName name of attribute to test
     * @return <code>true</code> if wa have an active i18n attribute, <code>false</code> otherwise.
     */
    public static boolean isI18nAttribute(String attributeName) {
        return I18N_ATTRIBUTES.contains(attributeName);
    }

    /**
     * Add the i18n on a attribute.
     * <p/>
     * Note: <b>Be ware : </b> no test is done here to ensure we are on a i18neable attribute for an i18nable compiler.
     * <p/>
     * Make sure with the method {@link I18nHelper#isI18nableAttribute(String, JAXXCompiler)} returns
     * <code>true</code< before using this method.
     *
     * @param widgetId           the id of the widget
     * @param attributeName      the name of the attribute
     * @param attributeValueCode the value code of the attribute value
     * @param compiler           the current used compile
     * @return the surrender i18n call if attribute name is match the
     *         attributeValueCode otherwise
     */
    public static String addI18nInvocation(String widgetId, String attributeName, String attributeValueCode, JAXXCompiler compiler) {

        if (log.isDebugEnabled()) {
            log.debug(" try i18n support for [" + widgetId + ":" + attributeName + "] : " + attributeValueCode);
        }
        if (attributeValueCode.contains("t(") && attributeValueCode.contains(")") && !attributeValueCode.contains(",")) {
            compiler.reportWarning("\n\tjaxx supports i18n, no need to add explicit call to I18n.t for attribute '" + attributeName + "' in component '" + widgetId + "' : [" + attributeValueCode + "]");
        } else {
            compiler.addImport("static " + I18n.class.getName() + ".t");
            attributeValueCode = "t(" + attributeValueCode + ")";
        }

        return attributeValueCode;
    }

    /**
     * Box a i18n key with code to be used as a i18n mnemonic.
     * <p/>
     * Note: <b>Be ware : </b> no test is done here to ensure we are on a
     * i18neable attribute for an i18nable compiler.
     *
     * @param object             the object to use
     * @param attributeName      the name of the attribute
     * @param attributeValueCode the value code of the attribute value
     * @param compiler           the current used compile
     * @return the surrender i18n call if attribute name is match the
     *         attributeValueCode otherwise
     * @since 2.6.11
     */
    public static String addI18nMnemonicInvocation(CompiledObject object,
                                                   String attributeName,
                                                   String attributeValueCode,
                                                   JAXXCompiler compiler) {
        String stringValue = I18nHelper.addI18nInvocation(
                object.getId(),
                attributeName,
                attributeValueCode,
                compiler);
        compiler.addImport(SwingUtil.class.getName());
        String result = "{" + SwingUtil.class.getSimpleName() +
                        ".getFirstCharAt(" + stringValue + ",'Z')}";
        return result;
    }
}
