/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: DefaultCompiledObjectDecorator.java 2118 2010-10-26 17:44:57Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.3/jaxx-compiler/src/main/java/jaxx/compiler/decorators/DefaultCompiledObjectDecorator.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.decorators;

import jaxx.compiler.CompiledObject;
import jaxx.compiler.CompiledObjectDecorator;
import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.java.JavaMethod;
import jaxx.compiler.script.ScriptInitializer;
import jaxx.compiler.types.TypeManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.Modifier;
import java.util.Map.Entry;

/**
 * The default decorator to use on all compiled objects.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role-hint="default" role="jaxx.compiler.CompiledObjectDecorator"
 * @since 1.2
 */
public class DefaultCompiledObjectDecorator implements CompiledObjectDecorator {


    /** Logger */
    protected static final Log log =
            LogFactory.getLog(DefaultCompiledObjectDecorator.class);

    @Override
    public String getName() {
        return "default";
    }

    @Override
    public void finalizeCompiler(JAXXCompiler compiler,
                                 CompiledObject root,
                                 CompiledObject object,
                                 JavaFile javaFile,
                                 String packageName,
                                 String className,
                                 String fullClassName) {

        if (object instanceof ScriptInitializer) {

            // nothing to finalize
            return;
        }

        String fqn = JAXXCompiler.getCanonicalName(object);

        String id = object.getId();

        if (log.isDebugEnabled()) {
            log.debug("finalize " + id);
        }

        boolean override = object.isOverride();

        if (override) {

            if (object.isOverrideType()) {

                // add a specialized getter, only if type has changed

                String methodName = object.getGetterName();

                String body = "return (" + fqn + ") super." + methodName + "();";

                if (log.isDebugEnabled()) {
                    log.debug("Add specialized getter " + methodName + " : " + body);
                }
                JavaMethod getter = JavaFileGenerator.newMethod(
                        Modifier.PUBLIC,
                        fqn,
                        methodName,
                        body,
                        true
                );
                javaFile.addMethod(getter);
            }
        } else {

            int access = id.startsWith("$") ? Modifier.PRIVATE :
                         Modifier.PROTECTED;
            if (root.equals(object)) {
                JavaField field = JavaFileGenerator.newField(access,
                                                             className,
                                                             id,
                                                             false,
                                                             "this"
                );
                javaFile.addSimpleField(field);
            } else {

                JavaField field = JavaFileGenerator.newField(access,
                                                             fqn,
                                                             id,
                                                             override
                );
                javaFile.addField(field, object.isJavaBean());
            }
        }

        if (compiler.inlineCreation(object) || root.equals(object)) {

            // nothing more to do here
            return;
        }

        String code = getCreationCode(compiler, object);

        // tchemit 20100519 Do not add the method only if overriden and code is null

        if (code != null) {

            JavaMethod javaMethod = JavaFileGenerator.newMethod(
                    Modifier.PROTECTED,
                    "void",
                    object.getCreationMethodName(),
                    code,
                    override
            );
            javaFile.addMethod(javaMethod);
        }
    }

    @Override
    public String getCreationCode(JAXXCompiler compiler,
                                  CompiledObject object) throws CompilerException {
        if (object instanceof ScriptInitializer) {
            throw new IllegalStateException(
                    "A script initializer can not come in getCreationcode method!");
        }
        String eol = JAXXCompiler.getLineSeparator();

        StringBuffer result = new StringBuffer();
        StringBuilder init = new StringBuilder();

        if (compiler.getRootObject().equals(object) ||
            compiler.inlineCreation(object)) {
            result.append("// inline creation of ").append(object.getId());
        }

        if (object.isJavaBean() && object.getJavaBeanInitCode() != null) {
            init.append(object.getJavaBeanInitCode());
        } else if (object.getInitializer() != null) {
            init.append(object.getInitializer());
        }

        boolean addToObjectMap = true;
        String id = TypeManager.getJavaCode(object.getId());
        String constructorParams = object.getConstructorParams();

        if (object.isOverride()) {

            if (init.length() == 0 && constructorParams == null) {

                // no init code is given, no need to add to objectMap
                addToObjectMap = false;
            }
        }

        if (addToObjectMap && init.length() == 0) {

            // on special init, use constructor
            String canonicalName = JAXXCompiler.getCanonicalName(object);
            init.append("new ").append(canonicalName).append("(");

            if (constructorParams != null) {
                init.append(constructorParams);
            }
            init.append(")");
        }

        String superCall = "super." + object.getCreationMethodName() + "();";

        if (addToObjectMap) {
            result.append(eol);
            result.append("$objectMap.put(");
            result.append(id);
            result.append(", ");
            result.append(object.getId()).append(" = ");
            result.append(init);
            result.append(");");
            result.append(eol);
        } else {
            if (object.isOverride()) {

                // when override has no special init code, just use the super method

                result.append(superCall);

            }
        }

        String initCode = object.getInitializationCode(compiler);
        if (initCode != null && initCode.length() > 0) {
            result.append(eol).append(initCode);
        }

        // add client properties        
        addClientProperties(object, result, eol);

        String code = result.toString();

        if (!compiler.inlineCreation(object) &&
            object.isOverride() &&
            superCall.equals(code.trim())) {

            // special case : when override but do nothing more
            // method creation can be skipped
            return null;
        }
        return code;
    }

    @Override
    public String createCompleteSetupMethod(JAXXCompiler compiler,
                                            CompiledObject object,
                                            JavaFile javaFile) {
        StringBuffer code = new StringBuffer();
        String eol = JAXXCompiler.getLineSeparator();
        if (object.getId().startsWith("$")) {
            String additionCode = object.getAdditionCode();
            //TC-20091025 only generate the code if not empty
            if (!additionCode.isEmpty()) {
                code.append("// inline complete setup of ");
                code.append(object.getId());
                code.append(eol);
                code.append(additionCode);
            }
        } else {
            //TODO-TC-20091202 should always create the method to make api more consistent ?
            //TODO-TC-20091202 While generating, we deal with this case, it seems not sa natural
            //TODO-TC-20091202 to NOT having the setup method on each public property ?
//            code.append(object.getAdditionMethodName()).append("();").append(eol);
//            if (!additionCode.isEmpty()) {
//                additionCode = "if (!allComponentsCreated) {" + eol + "    return;" + eol + "}" + eol + additionCode;
//            }
//            javaFile.addMethod(JavaFileGenerator.newMethod(Modifier.PROTECTED, "void", object.getAdditionMethodName(), additionCode, false));
            String additionCode = object.getAdditionCode();
            if (additionCode.length() > 0) {
                code.append(object.getAdditionMethodName()).append("();").append(eol);
                additionCode = "if (!allComponentsCreated) {" + eol +
                               "    return;" + eol + "}" + eol + additionCode;
                javaFile.addMethod(JavaFileGenerator.newMethod(
                        Modifier.PROTECTED,
                        "void",
                        object.getAdditionMethodName(),
                        additionCode,
                        false)
                );
            }
        }
        String result = code.toString();
        return result;
    }

    @Override
    public boolean createInitializer(JAXXCompiler compiler,
                                     CompiledObject root,
                                     CompiledObject object,
                                     StringBuffer code,
                                     boolean lastWasMethodCall) {
        String eol = JAXXCompiler.getLineSeparator();

        if (object instanceof ScriptInitializer) {

            // initializer has special direct treatment : can not be in a method
            // just push code to compiler
            code.append(object.getInitializationCode(compiler));

            // nothing to initialize of a script
            return lastWasMethodCall;
        }

        if (root.equals(object)) {
            String rootCode = root.getInitializationCode(compiler);
            if (rootCode != null && rootCode.length() > 0) {
                code.append("// inline creation of ");
                code.append(object.getId());
                code.append(eol);
                code.append(rootCode);
                //TC-20091025 generate client properties at creation time (not at setup time)
                // in some case can save to create a setup method (when there is only client properties
                // to store)
                addClientProperties(object, code, eol);
                code.append(eol);
            }
        } else {
            if (!object.isOverride()) {
                if (compiler.inlineCreation(object)) {
                    if (lastWasMethodCall) {
                        lastWasMethodCall = false;
                    }
                    code.append(getCreationCode(compiler, object));
                } else {
                    code.append(object.getCreationMethodName()).append("();");
                    code.append(eol);
                    lastWasMethodCall = true;
                }
            }
        }
        return lastWasMethodCall;
    }

    protected void addClientProperties(CompiledObject object,
                                       StringBuffer code,
                                       String eol) {
        //TC-20090327 generate client properties
        if (object.hasClientProperties()) {
            // generate putClientProperty invocations
            for (Entry<String, String> entry :
                    object.getClientProperties().entrySet()) {
                code.append(object.getJavaCode());
                code.append(".putClientProperty(\"");
                code.append(entry.getKey());
                code.append("\", ");
                code.append(entry.getValue());
                code.append(");");
                code.append(eol);
            }
        }
    }
}
