/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: JAXXEngine.java 2118 2010-10-26 17:44:57Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.3/jaxx-compiler/src/main/java/jaxx/compiler/JAXXEngine.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler;

import jaxx.compiler.tasks.CompileFirstPassTask;
import jaxx.compiler.tasks.CompileSecondPassTask;
import jaxx.compiler.tasks.FinalizeTask;
import jaxx.compiler.tasks.GenerateTask;
import jaxx.compiler.tasks.InitTask;
import jaxx.compiler.tasks.JAXXEngineTask;
import jaxx.compiler.tasks.ProfileTask;
import jaxx.compiler.tasks.StyleSheetTask;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.StringUtil;

import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * The engine to compile jaxx files.
 * <p/>
 * The method {@link #run()} launch the compilation of files.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0.0 was previously JAXXCompilerLaunchor
 */
public class JAXXEngine {

    /** Logger */
    private static final Log log = LogFactory.getLog(JAXXEngine.class);

    /** configuration of the launchor and underlines compilers */
    protected final CompilerConfiguration configuration;

    /** original list of files to compile (says the detected modfied files) */
    protected final JAXXCompilerFile[] incomingFiles;

    /** Files to compile */
    protected final List<JAXXCompilerFile> compilingFiles;

    /** Warnings detected while running. */
    protected final List<String> warnings;

    /** Errors detected while running. */
    protected final List<String> errors;

    /** tasks to launch */
    protected JAXXEngineTask[] tasks;

    /** current pass of the engine */
    protected JAXXEngineTask currentTask;

    /** profile attached to the engine (can be null) */
    protected JAXXProfile profiler;

    protected JAXXEngine(CompilerConfiguration configuration,
                         File base,
                         String... relativePaths) {

        if (configuration == null) {

            // use a default configuration
            configuration = new DefaultCompilerConfiguration();
        }

        this.configuration = configuration;
        warnings = new ArrayList<String>();
        errors = new ArrayList<String>();
        compilingFiles = new ArrayList<JAXXCompilerFile>();

        // add all default files to compile
        for (String relativePath : relativePaths) {
            JAXXCompilerFile compilerFile =
                    new JAXXCompilerFile(base, new File(base, relativePath));
            addFileToCompile(compilerFile);
        }

        // fix once for all incoming files to compile
        incomingFiles = compilingFiles.toArray(
                new JAXXCompilerFile[compilingFiles.size()]);
    }

    public JAXXEngineTask[] getTasks() {
        if (tasks == null) {
            List<JAXXEngineTask> tasks = new ArrayList<JAXXEngineTask>();

            tasks.add(new InitTask());
            tasks.add(new CompileFirstPassTask());
            tasks.add(new CompileSecondPassTask());
            tasks.add(new StyleSheetTask());
            tasks.add(new FinalizeTask());
            tasks.add(new GenerateTask());

            if (getConfiguration().isProfile()) {
                tasks.add(new ProfileTask());
            }

            this.tasks = tasks.toArray(new JAXXEngineTask[tasks.size()]);
        }
        return tasks;
    }

    /** @return the errors of the engine */
    public List<String> getErrors() {
        return errors;
    }

    /** @return the warnings of the engine */
    public List<String> getWarnings() {
        return warnings;
    }

    public JAXXProfile getProfiler() {
        if (profiler == null && getConfiguration().isProfile()) {
            profiler = new JAXXProfile();
        }
        return profiler;
    }

    public CompilerConfiguration getConfiguration() {
        return configuration;
    }

    public JAXXCompilerFile[] getIncomingFiles() {
        return incomingFiles;
    }

//    public JAXXCompilerFile[] getCompilingFiles() {
//        return compilingFiles.toArray(
//                new JAXXCompilerFile[compilingFiles.size()]);
//    }

    public JAXXCompilerFile[] getFilesToCompile() {
        List<JAXXCompilerFile> files = new ArrayList<JAXXCompilerFile>();
        for (JAXXCompilerFile file : compilingFiles) {
            if (file.getCompiler() == null) {
                files.add(file);
            }
        }
        return files.toArray(new JAXXCompilerFile[files.size()]);
    }

    public JAXXCompilerFile[] getCompiledFiles() {
        List<JAXXCompilerFile> files = new ArrayList<JAXXCompilerFile>();
        for (JAXXCompilerFile file : compilingFiles) {
            if (file.getCompiler() != null) {
                files.add(file);
            }
        }
        // always send a copy to be safe.
        return files.toArray(new JAXXCompilerFile[files.size()]);
    }

    public boolean containsJaxxFileClassName(String className) {
        for (JAXXCompilerFile file : compilingFiles) {
            if (className.equals(file.getClassName())) {
                return true;
            }
        }
        return false;
    }

    public boolean isCompileFirstPassTask() {
        return currentTask != null &&
               CompileFirstPassTask.TASK_NAME.equals(currentTask.getName());
    }

    /** Resets all state in preparation for a new compilation session. */
    protected void reset() {
        for (JAXXCompilerFile compilerFile : compilingFiles) {
            compilerFile.clear();
        }
        compilingFiles.clear();
        if (profiler != null) {
            profiler.clear();
            profiler = null;
        }
        clearReports();
    }

    public void clearReports() {
        getWarnings().clear();
        getErrors().clear();
    }

    public String getVersion() {
        return "2.0.2";
    }

    /**
     * Obtain the jaxx compiler of the given class name.
     *
     * @param className the name of the class to use
     * @return the compiler instance which is processing the specified JAXX class.
     *         Each class is compiled by a different compiler instance.
     */
    public JAXXCompilerFile getJAXXCompilerFile(String className) {
        for (JAXXCompilerFile compilingFile : compilingFiles) {
            if (className.equals(compilingFile.getClassName())) {
                return compilingFile;
            }
        }
        return null;
    }

    /**
     * Obtain the jaxx compiler of the given class name.
     *
     * @param className the name of the class to use
     * @return the compiler instance which is processing the specified JAXX class.
     *         Each class is compiled by a different compiler instance.
     */
    public JAXXCompiler getJAXXCompiler(String className) {
        JAXXCompilerFile compilerFile = getJAXXCompilerFile(className);
        if (compilerFile == null) {
            return null;
        }
        return compilerFile.getCompiler();
    }

    /**
     * Obtain the symbo table for the given class name.
     *
     * @param className the name of the class to use
     * @return the symbol table for the specified JAXX class.
     *         Must be called during the second compiler pass.
     *         Returns <code>null</code> if no such symbol table could be found.
     */
    public SymbolTable getSymbolTable(String className) {
        JAXXCompiler compiler = getJAXXCompiler(className);
        if (compiler == null) {
            return null;
        }
        return compiler.getSymbolTable();
    }

    /**
     * Obtain the decorator of the given name.
     *
     * @param name the name of the decorator
     * @return the decorator found.
     * @throws IllegalArgumentException if decorator not found for the given name.
     */
    public CompiledObjectDecorator getDecorator(String name)
            throws IllegalArgumentException {
        Map<String, CompiledObjectDecorator> decorators =
                getConfiguration().getDecorators();
        CompiledObjectDecorator decorator = decorators.get(name);
        if (decorator == null) {
            throw new IllegalArgumentException(
                    "could not find decorator with key " + name +
                    " (known decorators : " + decorators.keySet() + ")");
        }
        return decorator;
    }

    /**
     * Obtain the decorator of the given type.
     *
     * @param type the type of the decorator (syas his fqn)
     * @return the decorator found
     */
    public CompiledObjectDecorator getDecorator(Class<?> type) {
        Map<String, CompiledObjectDecorator> decorators =
                getConfiguration().getDecorators();
        for (CompiledObjectDecorator decorator : decorators.values()) {
            if (decorator.getClass().equals(type)) {
                return decorator;
            }
        }
        return null;
    }

    /**
     * Add a warning to the engine.
     *
     * @param warning the warning to add
     */
    public void addWarning(String warning) {
        warnings.add(warning);
    }

    /**
     * Add an error to the engine.
     *
     * @param error the error to add
     */
    public void addError(String error) {
        errors.add(error);
    }

    /**
     * Compiled a set of files.
     *
     * @return {@code -1} if errors appears, the number of generated files
     *         otherwise.
     */
    public int run() {
        try {
            boolean success = true;

            for (JAXXEngineTask task : getTasks()) {
                if (!success) {
                    // stop as soon as a engine phase failed
                    break;
                }

                currentTask = task;
                long t0 = System.nanoTime();
                if (isVerbose()) {
                    log.info("Start task '" + task.getName() + "' on " +
                             compilingFiles.size() + " file(s)");
                }
                success = task.perform(this);
                if (isVerbose()) {
                    log.info("task '" + task.getName() + "' done in " +
                             StringUtil.convertTime(System.nanoTime() - t0)
                    );
                }
            }
            return success ? compilingFiles.size() : -1;

            //FIXME : deal better  the exception treatment...
        } catch (CompilerException e) {
            log.error(e.getMessage(), e);
            return -1;
        } catch (Throwable e) {
            log.error(e.getMessage(), e);
            return -1;
        } finally {
            //TC - 20081018 only reset when no error was detected
            if (configuration.isResetAfterCompile() && errors.isEmpty()) {
                reset();
            }
        }
    }

    /**
     * Adds a {@code file} to be compiled.
     *
     * @param file the {@link JAXXCompilerFile} to add.
     */
    public void addFileToCompile(JAXXCompilerFile file) {

        if (isVerbose()) {
            log.info("register jaxx file " + file.getJaxxFile());
        }
        compilingFiles.add(file);
    }

    /**
     * Adds a {@link JAXXCompilerFile} given the jaxx file and the
     * corresponding class fully qualified name.
     *
     * @param jaxxFile      the jaxx file location
     * @param jaxxClassName the fully qualified name of the jaxx class
     */
    public void addFileToCompile(File jaxxFile, String jaxxClassName) {

        if (log.isDebugEnabled()) {
            log.debug("file = " + jaxxFile + ", fqn = " + jaxxClassName);
        }

        JAXXCompilerFile file = new JAXXCompilerFile(jaxxFile, jaxxClassName);
        addFileToCompile(file);
    }

    /**
     * Create a new compiler and attach it to the given {@code jaxxFile}.
     *
     * @param jaxxFile the definition of jaxx file to compile
     * @return the new compiler
     * @throws Exception if any pb while creation of compiler
     */
    public JAXXCompiler newCompiler(JAXXCompilerFile jaxxFile) throws Exception {

        Class<?> compilerClass =
                getConfiguration().getCompilerClass();

        if (compilerClass == null) {
            throw new NullPointerException(
                    "Configuration compilerClass is null");
        }

        Constructor<?> cons = compilerClass.getConstructor(
                JAXXEngine.class,
                JAXXCompilerFile.class,
                List.class
        );

        JAXXCompiler jaxxCompiler = (JAXXCompiler) cons.newInstance(
                this,
                jaxxFile,
                Arrays.asList(
                        "java.awt.*",
                        "java.awt.event.*",
                        "java.io.*",
                        "java.lang.*",
                        "java.util.*",
                        "javax.swing.*",
                        "javax.swing.border.*",
                        "javax.swing.event.*",
                        "jaxx.runtime.*",
                        "jaxx.runtime.swing.*",
                        "static org.nuiton.i18n.I18n._",
                        "static jaxx.runtime.SwingUtil.createImageIcon"
                )
        );
        jaxxFile.setCompiler(jaxxCompiler);
        return jaxxCompiler;
    }

    public boolean isVerbose() {
        return getConfiguration().isVerbose();
    }

    /**
     * Add a profile time for the given compiler and key.
     * <p/>
     * Note: if {@link #profiler} is {@code null}, do nothing
     *
     * @param compiler the compiler to profile
     * @param key      the key of profiling
     */
    public void addProfileTime(JAXXCompiler compiler, String key) {
        JAXXProfile profiler = getProfiler();
        if (profiler != null) {
            profiler.addTime(compiler, key);
        }
    }
}
