/*
 * *##% 
 * I18n :: Maven Plugin
 * Copyright (C) 2007 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%* */

package org.nuiton.i18n.plugin.parser.impl;

import org.apache.maven.plugin.logging.Log;
import org.nuiton.i18n.plugin.parser.*;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.FileUpdaterHelper;
import org.nuiton.io.SortedProperties;
import org.nuiton.processor.filters.I18nFilter;

import java.io.*;

/**
 * Récupération des chaine a traduire depuis les fichiers java.
 *
 * @author julien
 * @goal parserJava
 * @phase generate-resources
 */
public class ParserJavaMojo extends AbstractI18nParserMojo {

    /**
     * Source entries (src+includes+excludes) .
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/*.java"
     */
    protected String defaultIncludes;

    /**
     * default src for an entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/java"
     */
    protected File defaultBasedir;

    /**
     * Repertoire sources des fichiers i18n.
     *
     * @parameter expression="${i18n.cp}" default-value="${basedir}/target/classes"
     * @required
     */
    protected File cp;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return FileUpdaterHelper.newJavaFileUpdater(entry.getBasedir(), cp);
    }

    @Override
    @Deprecated
    protected String getKeyModifierStart() {
        return "_\\(\\s*\"";
    }

    @Override
    @Deprecated
    protected String getKeyModifierEnd() {
        return "\"\\s*(\\)|,|\\+|$)";
    }

    @Override
    protected String getOutGetter() {
        return "java.getter";
    }

    @Override
    public FileParser newFileParser() {

        return new JavaFileParser(getLog(),
                                  encoding,
                                  oldParser,
                                  showTouchedFiles
        );
    }

    protected static class JavaFileParser extends AbstractFileParser {

        protected final I18nFilter filter = new I18nFilter();

        public JavaFileParser(Log log,
                              String encoding,
                              SortedProperties oldParser,
                              boolean showTouchedFiles) {
            super(log, encoding, oldParser, showTouchedFiles);
        }

        @Override
        public void parseFile(File file) throws IOException {
            String line = null;
            LineNumberReader lnr = new LineNumberReader(new InputStreamReader(
                    new FileInputStream(file)));
            try {
                while ((line = lnr.readLine()) != null) {
                    parseLine(file, line);
                }
            } catch (Exception e) {
                if (line != null) {
                    getLog().error(
                            "could not parse line (" + lnr.getLineNumber() + ") '"
                            + line + "' of file " + file);
                }
                throw new ParserException(e);
            } finally {
                lnr.close();
            }
        }

        @Override
        public void parseLine(File file, String line) throws IOException {

            String keysSet = filter.parse(line);

            if (keysSet.equals(I18nFilter.EMPTY_STRING)) {
                // no key detected on this line
                return;
            }

            // one key found in file, so file is marked as touched
            setTouched(true);
            // Found a set of i18n Strings, split it.
            String[] keys = keysSet.split("=");
            for (String key : keys) {
                registerKey(key);
            }
        }
    }
}
