/*
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.i18n.plugin.parser.impl;

import javax.xml.parsers.ParserConfigurationException;
import org.nuiton.i18n.plugin.parser.AbstractI18nParser;
import org.nuiton.i18n.plugin.parser.ParserEvent;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.xml.sax.EntityResolver;

/**
 * Récupération des chaines à traduire depuis les fichiers xml.
 *
 * @author julien
 */
public abstract class ParserXml extends AbstractI18nParser {

    /** Taille du buffer pour les lectures/écritures */
    protected static final int BUFFER_SIZE = 8 * 1024;
    /**
     * default src for an entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/uimodel"
     * @required
     */
    protected File defaultBasedir;
    protected String rules;
    protected XPathFactory factory;
    protected XPath xpath;
    protected DocumentBuilder builder;

    /**
     * Fonction d'extraction de la chaine
     *
     * @param i18nString le clef i18n
     * @return la chaine
     */
    public abstract String extract(String i18nString);

    /** @return le fichier des rules */
    protected abstract String getFileRules();

    /** @return le fichier des rules de base à toujours charger */
    protected abstract String getCoreFileRules();

    @Override
    public void init() {
        super.init();
        this.factory = XPathFactory.newInstance();
        this.rules = getRules(getFileRules());
        this.xpath = factory.newXPath();
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true); // never forget this!

        try {
            // never forget this!
            builder = documentBuilderFactory.newDocumentBuilder();

            EntityResolver resolver = getEntityResolver();

            if (resolver != null) {
                builder.setEntityResolver(resolver);
            }

        } catch (ParserConfigurationException ex) {
            throw new IllegalStateException("could not load DocumentBuilder for reason " + ex.getMessage(), ex);
        }
    }

    public EntityResolver getEntityResolver() {
        return null;
    }

    public InputSource getSystemId(String publicId) {
        return null;
    }

    @Override
    public void parseFile(File file) {
        NodeList list;
//        InputSource inputSource = new InputSource(file.getAbsolutePath()); // TODO: A deplacer pour les performances

        try {
            int size = result.size();

            // Recherche des clés à partir d'un xpath

            Document doc = builder.parse(file.getAbsolutePath());
//            Document doc = builder.parse(new FileInputStream(file.getAbsolutePath()),xworksResource.getFile());
            XPathExpression expression = xpath.compile(rules);
            list = (NodeList) expression.evaluate(doc, XPathConstants.NODESET);
//            list = (NodeList) expression.evaluate(inputSource, XPathConstants.NODESET);

            for (int index = 0; index < list.getLength(); index++) {
                Node node = list.item(index);
                parseLine(file, node.getTextContent());
            }
            if (safeMode) {
                // Détection de nouvelles clés, sauvegarde du fichier pour pouvoir le restaurer en cas de plantage
                if (size != result.size()) {
                    saveGetterFile();
                }
            }
        } catch (Exception e) {
            throw new ParserException(e);
        }
    }

    @Override
    public void parseLine(File file, String key) {
        key = extract(key);
        if (key != null) {
            touchFile = true;
            String keyModified = key;
            for (ParserEvent event : events) {
                event.eventChangeKey(key, !oldLanguage.containsKey(key));
                keyModified = event.eventGetRealKey();
            }

            if (oldParser.containsKey(key)) {
                result.put(keyModified, oldParser.get(key));
            } else {
                result.put(keyModified, key);
            }
        }
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    /**
     * Récupère le xpath à partir d'un fichier
     *
     * @param fileRules le nom du fichier contant les règles
     * @return le xpath à partir d'un fichier
     */
    private String getRules(String fileRules) {
        StringBuilder buffer = new StringBuilder();

        try {
            String readInputStream;

            // load core rules
            readInputStream = loadRulesFile(getCoreFileRules());
            if (!silent && verbose) {
                getLog().info("core rules : " + getCoreFileRules());
            }
            buffer.append(readInputStream);

            if (!fileRules.equals(getCoreFileRules())) {
                // add custom rules
                readInputStream = loadRulesFile(fileRules);
                if (!silent && verbose) {
                    getLog().info("custom rules : " + fileRules);
                }
                buffer.append(" | ").append(readInputStream);
            }
        } catch (IOException e) {
            throw new ParserException(e);
        }

        return buffer.toString();
    }

    private String loadRulesFile(String fileRules) throws IOException {
        File f = new File(fileRules);

        InputStream inputStream;
        if (f.exists()) {
            // load from a file
            try {
                inputStream = new FileInputStream(f);
            } catch (FileNotFoundException e) {
                throw new ParserException(e);
            }
        } else {
            // load from classpath
            ClassLoader classLoader = getClass().getClassLoader();
            inputStream = classLoader.getResourceAsStream(fileRules);
        }
        if (inputStream == null) {
            throw new ParserException("could not found file of rules : " + fileRules);
        }

        inputStream = new BufferedInputStream(inputStream);

        try {
            // Lecture
            String readInputStream;
            readInputStream = readInputStream(inputStream);
            return readInputStream;
        } catch (IOException e) {
            throw new ParserException(e);
        } finally {
            inputStream.close();
        }
    }

    /**
     * Permet la lecture d'un InputStream et Suppressions.
     *
     * @param in le flux entrant
     * @return le contenu du flux
     * @throws IOException si problème de lecture dans flux entrant
     */
    private String readInputStream(InputStream in) throws IOException {
        StringBuilder sb = new StringBuilder();
        byte[] buffer = new byte[BUFFER_SIZE];
        while (in.read(buffer, 0, BUFFER_SIZE) != -1) {
            String tmp = new String(buffer);
            sb.append(tmp);
        }
        in.close();
        // Suppression
        String txt = sb.toString().trim();
        txt = txt.replaceAll("#.*\n", ""); // suppression des commentaires
        txt = txt.replaceAll("\\s+", " | "); // contruction du xpath avec des ou
        txt = txt.replaceAll("(^ \\| )|( \\| $)", ""); // suppression des ou de début ee fin
        return txt;
    }
}
