/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.ActionContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.UserAccount;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * User session to put in servlet session.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenSession implements Serializable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollenSession.class);

    /** Key to store the {@link PollenSession} instance in the session's map. */
    protected static final String SESSION_PARAMETER = "pollenSession";

    public static final String SESSION_TOKEN_MESSAGES = "messages";

    public static final String SESSION_TOKEN_ERRORS = "errors";

    public static final String SESSION_TOKEN_WARNINGS = "warnings";

    private static final long serialVersionUID = -1L;

    public static PollenSession get(ServletRequest servletRequest) {
        HttpSession httpSession = ((HttpServletRequest) servletRequest).getSession(true);
        PollenSession pollenSession = (PollenSession) httpSession.getAttribute(SESSION_PARAMETER);
        if (pollenSession == null) {
            pollenSession = new PollenSession();
            httpSession.setAttribute(SESSION_PARAMETER, pollenSession);
        }
        return pollenSession;
    }

    public static PollenSession get(ActionContext actionContext) {

        // load it from session
        Map<String, Object> strutsSession =
                actionContext.getSession();
        PollenSession session = (PollenSession)
                strutsSession.get(SESSION_PARAMETER);

        if (session == null) {

            // create an empty session
            session = new PollenSession();

            // and store it in the struts session
            strutsSession.put(SESSION_PARAMETER, session);
        }
        return session;
    }


    /** User loggued account (if anonymous, then userAccount is null). */
    private UserAccount userAccount;

    /** To push dynamic data which should be consumed in a page scope. */
    private Map<String, Serializable> dynamicData;

    public UserAccount getUserAccount() {
        return userAccount;
    }

    public void setUserAccount(UserAccount userAccount) {
        this.userAccount = userAccount;
    }

    public <T extends Serializable> T getDynamicData(String token) {
        Serializable result = getDynamicData().get(token);
        return (T) result;
    }

    public <T extends Serializable> Set<T> getDynamicSetData(String token) {
        Serializable result = getDynamicData().get(token);
        return (Set<T>) result;
    }

    public <T extends Serializable> T consumeDynamicData(String token) {
        T result = getDynamicData(token);
        if (result != null) {
            removeDynamicData(token);
        }
        return result;
    }

    public <T extends Serializable> Set<T> consumeDynamicSetData(String token) {
        Set<T> result = getDynamicSetData(token);
        if (result != null) {
            removeDynamicData(token);
        }
        return result;
    }

    public <T extends Serializable> void putDynamicSetData(String token, Set<T> data) {
        getDynamicData().put(token, (Serializable) data);
        if (log.isDebugEnabled()) {
            log.debug("Dynamic attributes size : " + getDynamicData().size());
        }
    }

    public void putDynamicData(String token, Serializable data) {
        getDynamicData().put(token, data);
        if (log.isDebugEnabled()) {
            log.debug("Dynamic attributes size : " + getDynamicData().size());
        }
    }

    public void removeDynamicData(String token) {
        getDynamicData().remove(token);
        if (log.isDebugEnabled()) {
            log.debug("Dynamic attributes size : " + getDynamicData().size());
        }
    }

    public void clearDynamicData() {
        if (dynamicData != null) {
            dynamicData.clear();
            if (log.isDebugEnabled()) {
                log.debug("Dynamic attributes size : " + dynamicData.size());
            }
        }
    }

    protected Map<String, Serializable> getDynamicData() {
        if (dynamicData == null) {
            dynamicData = Maps.newHashMap();
        }
        return dynamicData;
    }

    public void addMessage(String messageScope, String message) {
        Set<String> messages = getDynamicSetData(messageScope);
        if (messages == null) {
            messages = Sets.newHashSet(message);
            putDynamicSetData(messageScope, messages);
        } else {
            messages.add(message);
        }
    }

    public void removeDynamicDataWithPrefix(String prefix) {
        Iterator<Map.Entry<String, Serializable>> itr =
                getDynamicData().entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry<String, Serializable> entry = itr.next();
            if (entry.getKey().startsWith(prefix)) {
                itr.remove();
            }
        }
    }
}
