/*
 * #%L
 * Pollen :: Services
 * $Id: PollenMigrationCallbackV1_4.java 3712 2012-09-30 12:57:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.4/pollen-services/src/main/java/org/chorem/pollen/entities/migration/PollenMigrationCallbackV1_4.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.entities.migration;

import com.opensymphony.xwork2.ActionContext;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenApplicationContext;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountImpl;
import org.chorem.pollen.business.persistence.PollImpl;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.business.persistence.VoteToChoiceImpl;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.services.impl.PollFeedService;
import org.chorem.pollen.services.impl.PollService;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaSQLQuery;
import org.nuiton.topia.migration.TopiaMigrationCallbackByClassNG;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Locale;

/**
 * Migration for version {@code 1.4}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4
 */
public class PollenMigrationCallbackV1_4 extends TopiaMigrationCallbackByClassNG.MigrationCallBackForVersion {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenMigrationCallbackV1_4.class);

    @Override
    public Version getVersion() {
        return VersionUtil.valueOf("1.4");
    }

    @Override
    protected void prepareMigrationScript(TopiaContextImplementor tx,
                                          List<String> queries,
                                          boolean showSql,
                                          boolean showProgression) throws TopiaException {

        // remove all votes of condorcet type with a null value (<=0)
        // see http://chorem.org/issues/574
        // see http://chorem.org/issues/576
        removeAllCondorcetNullVotes(tx, queries);

        // add a flag to not generate results each time (http://chorem.org/issues/135)
        addResultUptodateToPoll(queries);

        // generate missing feed files
        generateMissingPollFeedFiles(tx);

        // generate missing thumbs
        generateMissingImageThumbs(tx);

        // add unique on Poll.pollId and PollAccount.accountId (http://www.chorem.org/issues/583)
        addUniqueOnPollIds(queries);

        // Change to type 'timestamp' the topiacreatdate to have nice order (for vote and comment) (http://chorem.org/issues/173)
        addTimestampOnTopiaCreateDate(queries);
    }

    private void addTimestampOnTopiaCreateDate(List<String> queries) {
        queries.add("alter TABLE vote ALTER COLUMN topiacreatedate type timestamp;");
        queries.add("alter TABLE comment ALTER COLUMN topiacreatedate type timestamp;");
    }

    private void addUniqueOnPollIds(List<String> queries) {

        queries.add("CREATE UNIQUE INDEX idx_Pöll_pollId ON poll(pollId);");
    }

    private void generateMissingImageThumbs(TopiaContextImplementor tx) throws TopiaException {
        TopiaSQLQuery<Pair<String, String>> getAllImageChoicesQuery = new TopiaSQLQuery<Pair<String, String>>() {

            @Override
            protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement("SELECT p.pollId, c.name FROM choice c, poll p WHERE c.poll = p.topiaid AND p.choicetype=2;");
                return ps;
            }

            @Override
            protected Pair<String, String> prepareResult(ResultSet set) throws SQLException {
                String pollId = set.getString(1);
                String choiceName = set.getString(2);
                return Pair.of(pollId, choiceName);
            }
        };

        PollenServiceFactory serviceFactory = new PollenServiceFactory();

        PollenApplicationContext applicationContext = PollenApplicationContext.get(
                ActionContext.getContext());

        PollenConfiguration configuration =
                applicationContext.getConfiguration();


        PollenServiceContext sContext = DefaultPollenServiceContext.newContext(
                Locale.getDefault(),
                tx,
                configuration,
                serviceFactory,
                applicationContext.getVoteCountingFactory()
        );

        List<Pair<String, String>> choiceIds =
                getAllImageChoicesQuery.findMultipleResult(tx);

        PollService service = sContext.newService(PollService.class);

        for (Pair<String, String> choiceId : choiceIds) {

            String pollId = choiceId.getLeft();
            String choiceName = choiceId.getRight();

            File imageFile = service.getPollChoiceImageFile(
                    pollId, choiceName);

            if (!imageFile.exists()) {

                // image does not exists, can not generate thumb
                if (log.isWarnEnabled()) {
                    log.warn("Could not find image choice " + imageFile);
                }
                continue;
            }

            try {

                service.generateThumbIfNeeded(imageFile);
            } catch (IOException e) {
                throw new TopiaException("Could not create thumb for " +
                                         imageFile, e);
            }
        }
    }

    private void generateMissingPollFeedFiles(TopiaContextImplementor tx) throws TopiaException {
        TopiaSQLQuery<Poll> getAllPollIdsQuery = new TopiaSQLQuery<Poll>() {

            @Override
            protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement("SELECT p.pollId, p.title, p.description, c.votingId  FROM poll p, PollAccount c WHERE p.creator = c.topiaId;");
                return ps;
            }

            @Override
            protected Poll prepareResult(ResultSet set) throws SQLException {
                String pollId = set.getString(1);
                String pollTitle = set.getString(2);
                String pollDescription = set.getString(3);
                String votingId = set.getString(4);
                Poll poll = new PollImpl();
                poll.setPollId(pollId);
                poll.setTitle(pollTitle);
                poll.setDescription(pollDescription);
                PollAccount pollAccount = new PollAccountImpl();
                pollAccount.setVotingId(votingId);
                poll.setCreator(pollAccount);
                return poll;
            }
        };

        PollenServiceFactory serviceFactory = new PollenServiceFactory();

        PollenApplicationContext applicationContext = PollenApplicationContext.get(
                ActionContext.getContext());

        PollenConfiguration configuration =
                applicationContext.getConfiguration();


        PollenServiceContext sContext = DefaultPollenServiceContext.newContext(
                Locale.getDefault(),
                tx,
                configuration,
                serviceFactory,
                applicationContext.getVoteCountingFactory()
        );

        PollFeedService feedService = sContext.newService(PollFeedService.class);

        List<Poll> polls = getAllPollIdsQuery.findMultipleResult(tx);

        for (Poll poll : polls) {

            boolean feedExists = feedService.isFeedExists(poll);
            if (!feedExists) {

                // creates feed file

                if (log.isInfoEnabled()) {
                    log.info("Create missing file " +
                             feedService.getFeedLocation(poll));
                }

                feedService.onPollCreated(poll);
            }

        }
    }

    private void addResultUptodateToPoll(List<String> queries) {
        queries.add("ALTER TABLE poll ADD COLUMN resultuptodate boolean default false;");

    }

    private void removeAllCondorcetNullVotes(TopiaContextImplementor tx,
                                             List<String> queries) throws TopiaException {

        // get all votes from all polls with votecountingtype = 2 (condorcet)

        TopiaSQLQuery<String> getAllVotesQuery = new TopiaSQLQuery<String>() {

            @Override
            protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
                PreparedStatement ps = connection.prepareStatement("select topiaid from vote where poll in (select topiaid from poll where votecountingtype=2);");
                return ps;
            }

            @Override
            protected String prepareResult(ResultSet set) throws SQLException {
                return set.getString(1);
            }
        };
        List<String> voteIds = getAllVotesQuery.findMultipleResult(tx);

        for (String voteId : voteIds) {

            // get voteToChoice for this vote
            List<VoteToChoice> voteToChoices =
                    new GetVoteToChoiceSQLQuery(voteId).findMultipleResult(tx);

            for (VoteToChoice voteToChoice : voteToChoices) {

                if (voteToChoice != null && voteToChoice.getVoteValue() <= 0) {

                    // delete this vote

                    queries.add("DELETE FROM votetochoice where topiaid ='" +
                                voteToChoice.getTopiaId() + "';");
                }
            }
        }
    }

    private static class GetVoteToChoiceSQLQuery extends TopiaSQLQuery<VoteToChoice> {

        private final String voteId;

        public GetVoteToChoiceSQLQuery(String voteId) {
            this.voteId = voteId;
        }

        @Override
        protected PreparedStatement prepareQuery(Connection connection) throws SQLException {
            PreparedStatement ps = connection.prepareStatement(
                    "SELECT topiaid, votevalue FROM votetochoice WHERE vote = ?;");
            ps.setString(1, voteId);
            return ps;
        }

        @Override
        protected VoteToChoice prepareResult(ResultSet set) throws SQLException {
            VoteToChoice result = new VoteToChoiceImpl();
            result.setTopiaId(set.getString(1));
            result.setVoteValue(set.getInt(2));
            return result;
        }
    }
}
