/*
 * #%L
 * Pollen :: Services
 * $Id: PollResultsService.java 3556 2012-06-25 17:56:45Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollResultsService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenFunctions;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.Result;
import org.chorem.pollen.business.persistence.ResultDAO;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.PollenServiceFunctions;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.PollExportDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.services.ImportExportService;
import org.chorem.pollen.votecounting.services.VoteCountingService;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import static org.nuiton.i18n.I18n.l_;

/**
 * Deals with poll results.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollResultsService extends PollenServiceSupport {

    /** log. */
    private static final Log log = LogFactory.getLog(PollResultsService.class);

    /**
     * Récurpération des résultats d'un sondage sous forme de
     * {@link PollResultList}.
     * <p/>
     * Si les résultats ne sont pas à jour ({@link Poll#isResultUptodate()},
     * alors ils seront regénérés.
     *
     * @param poll le sondage à traiter
     * @return les résultats du sondage.
     */
    public PollResultList getResults(Poll poll) {

        Preconditions.checkNotNull(poll);


        VoteCountingResultDTO result;
        if (!poll.isResultUptodate()) {

            //must regenerate results

            result = generateResult(poll);

            commitTransaction("Could not generate poll result");
        } else {

            result = getResultDTO(poll);

        }

        boolean byGroup = poll.getPollType() == PollType.GROUP;

        int voteCountingType = poll.getVoteCountingType();

        // Conversion des résultats
        List<PollResult> list = Lists.transform(
                poll.getResult(),
                PollenServiceFunctions.newResultToBeanFunction(poll));

        // Filtrage des résultats que l'on veut retourner

        Iterator<PollResult> it = list.iterator();
        while (it.hasNext()) {
            PollResult currentResult = it.next();

            // Cas d'un sondage de type GROUP avec filtre
            if (byGroup) {

                // Filtre group : Suppression resultats non group
                if (!currentResult.isByGroup()) {
                    it.remove();
                }
            }
        }

        PollResultList resultList = new PollResultList();
        resultList.setPollResults(list);
        resultList.setVoteCountingResult(result);
        return resultList;
    }

    /**
     * Génère les résultats d'un sondage (= un dépouillement) et les stoque en
     * base.
     * <p/>
     * <strong>Note importante :</strong> Le commite n'est pas effectuée dans
     * cette méthode car d'autres services vont appeller cette méthode au sein
     * d'une transaction et on ne veut pas faire deux commits.
     *
     * @param poll le sondage à dépouiller
     * @since 1.4
     */
    public VoteCountingResultDTO generateResult(Poll poll) {

        Preconditions.checkNotNull(poll);

        if (log.isInfoEnabled()) {
            log.info("Will regenerate results for poll " + poll.getPollId());
        }

        VoteCountingResultDTO result = getResultDTO(poll);

        // clear result
        poll.clearResult();

        ResultDAO daoResult = getDAO(Result.class);

        for (ChoiceDTO choice : result.getChoices()) {

            Result eResult = create(daoResult);

            Choice eChoice = poll.getChoiceByTopiaId(choice.getIdChoice());

            eResult.setName(eChoice.getName());
            eResult.setByGroup(result.isByGroup());
//            eResult.setPoll(poll);
            eResult.setResultValue(String.valueOf(choice.getValue()));
            eResult.setVoteCountingType(poll.getVoteCountingType());

            poll.addResult(eResult);
        }

        poll.setResultUptodate(true);

        return result;
    }

    public String exportPolltoXml(String pollId) throws PollNotFoundException {

        // Recherche du sondage

        PollService pollService = newService(PollService.class);

        Poll poll = pollService.getExistingPollByPollId(pollId);

        PollDTO dto = PollenFunctions.POLL_TO_BEAN.apply(poll);

        VoteCountingService service = new VoteCountingService();

        List<VoteCountingResultDTO> results = Lists.newArrayList();

        VoteCountingResultDTO result = service.executeVoteCounting(dto);

        results.add(result);

        if (poll.getPollType() == PollType.GROUP) {
            VoteCountingResultDTO groupResult =
                    service.executeGroupCounting(dto);
            results.add(groupResult);
        }

        // Transformation du sondage
        PollExportDTO pollExport = new PollExportDTO();
        pollExport.setPollId(pollId);
        pollExport.setPoll(dto);
        pollExport.setVoteCountingResults(results);

        // Export du sondage
        ImportExportService serviceExport = new ImportExportService();
        String content = serviceExport.exportToXml(pollExport);

        return content;
    }

    public String getResultValue(Choice choice,
                                 Collection<PollResult> results) {

        String val = "";
        for (PollResult result : results) {
            if (result.getName().equals(choice.getName())) {
                val = removeTrailing0(result.getValue());
                break;
            }
        }
        return val;
    }

    public List<PollResult> createNumberVoteCountingResult(
            PollResultList resultListDTO,
            Multimap<String, String> choicesResults,
            Multimap<String, String> subtitles) {
        Preconditions.checkNotNull(resultListDTO);
        Preconditions.checkNotNull(choicesResults);
        Preconditions.checkNotNull(subtitles);

        List<PollResult> results = resultListDTO.getPollResults();

        /*
        *  liste des vrais choix (les choix pas cachés).
        *  cette variable sera affectée à la variable results
        */
        List<PollResult> results2 = Lists.newArrayList();

        // Ajout des résultats des choix cachés
        for (PollResult result : results) {
            if (result.isHidden()) {
                String name = result.getName();
                int indexOf = name.indexOf('#');
                String choice = name.substring(
                        NumberMethod.HIDDEN_PREFIX.length(), indexOf);

                String votingId = name.substring(indexOf + 1);
                String value = result.getValue();

                choicesResults.put(choice, votingId);
                choicesResults.put(choice, value);

            } else {
                results2.add(result);
            }
        }

        Locale locale = getLocale();

        // Récupération du choix correspondant au résultat
        VoteCountingResultDTO voteCountingResult = resultListDTO
                .getVoteCountingResult();

        for (String choiceName : choicesResults.keySet()) {
            Collection<String> values = choicesResults.get(choiceName);

            for (ChoiceDTO choice : voteCountingResult.getChoices()) {
                if (choiceName.equals(choice.getName())) {
                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-total"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getValue()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-average"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getAverage()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-blank-votes"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(choice.getNbBlankVotes()));

                    subtitles.put(
                            choiceName,
                            l_(locale, "pollen.common.numberVote-total-votes"));
                    subtitles.put(
                            choiceName,
                            String.valueOf(values.size() / 2));
                    break;
                }
            }
        }

        if (results2.size() == 1) {
            results2.remove(0);
        }
        return results2;
    }

    public List<PollResult> getTopRanking(List<PollResult> results) {
        List<PollResult> ranking = Lists.newArrayList(results);

        Collections.sort(ranking, new Comparator<PollResult>() {
            @Override
            public int compare(PollResult o1, PollResult o2) {
                Double result1 = Double.parseDouble(o1.getValue());
                Double result2 = Double.parseDouble(o2.getValue());
                int comp = 0; // résultat de la comparaison

                if (result1 > result2) {
                    comp = -1;
                } else if (result1 < result2) {
                    comp = 1;
                }
                if (comp == 0) {
                    comp = o1.getName().compareTo(o2.getName());
                }
                return comp;
            }
        });

        List<PollResult> topRanking = Lists.newArrayList();

        if (CollectionUtils.isNotEmpty(ranking)) {
            String winValue = ranking.get(0).getValue();
            for (PollResult r : ranking) {
                if (winValue.equals(r.getValue())) {
                    topRanking.add(r);
                }
            }
        }
        return topRanking;
    }

    /**
     * Supprime le 0 final d'un nombre à virgule. Le résultat peut-être un
     * double : 1,0 -> 1 et 1,2 -> 1,2.
     *
     * @param val le nombre
     * @return le nombre sans 0 final
     */
    protected String removeTrailing0(String val) {
        if (val.endsWith(".0")) {
            val = val.substring(0, val.indexOf('.'));
        }
        return val;
    }

    /**
     * Retourne une chaîne contenant les résultats du sondage.
     *
     * @param poll le sondage
     * @return les résultats sous forme de chaine de caractères
     */
    public String getResultsAsString(Poll poll) {

        DateFormat dateFormat = new SimpleDateFormat(getDateTimePattern());
        StringBuilder res = new StringBuilder("");
        Iterator<Result> it = poll.getResult().iterator();
        while (it.hasNext()) {
            Result result = it.next();
            if (poll.getChoiceType() == ChoiceType.DATE) {
                Date date = new Date(Long.parseLong(result.getName()));
                res.append(dateFormat.format(date));
            } else {
                res.append(result.getName());
            }
            res.append("=").append(removeTrailing0(result.getResultValue()));
            if (it.hasNext()) {
                res.append(", ");
            }
        }
        return res.toString();
    }

    protected VoteCountingResultDTO getResultDTO(Poll poll) {

        PollDTO dto = PollenFunctions.POLL_TO_BEAN.apply(poll);

        dto.setVoteCounting(poll.getVoteCountingType());

        VoteCountingService service = new VoteCountingService();

        VoteCountingResultDTO result = service.execute(dto);
        return result;
    }
}
