/*
 * #%L
 * Refcomp :: services
 * 
 * $Id: StorageServiceImpl.java 209 2011-07-22 15:28:45Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-1.4/refcomp-services/src/main/java/fr/inra/refcomp/services/StorageServiceImpl.java $
 * %%
 * Copyright (C) 2010 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.services;

import fr.inra.refcomp.entities.Agent;
import fr.inra.refcomp.entities.AgentSkill;
import fr.inra.refcomp.entities.AgentSkillImpl;
import fr.inra.refcomp.entities.EntitiesList;
import fr.inra.refcomp.entities.Cati;
import fr.inra.refcomp.entities.CatiImpl;
import fr.inra.refcomp.entities.Department;
import fr.inra.refcomp.entities.DepartmentImpl;
import fr.inra.refcomp.entities.Frequency;
import fr.inra.refcomp.entities.FrequencyImpl;
import fr.inra.refcomp.entities.Nature;
import fr.inra.refcomp.entities.Skill;
import fr.inra.refcomp.entities.SkillImpl;
import fr.inra.refcomp.entities.Unit;
import fr.inra.refcomp.entities.UnitDeptLink;
import fr.inra.refcomp.entities.UnitDeptLinkImpl;
import fr.inra.refcomp.entities.UnitImpl;
import fr.inra.refcomp.entities.User;
import fr.inra.refcomp.entities.UserImpl;
import java.io.Reader;
import java.io.Writer;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.StringUtil;
import org.nuiton.validator.NuitonValidator;
import org.nuiton.validator.NuitonValidatorFactory;
import org.nuiton.validator.NuitonValidatorResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyAuthorisation;
import org.nuiton.wikitty.entities.WikittyAuthorisationImpl;
import org.nuiton.wikitty.entities.WikittyGroup;
import org.nuiton.wikitty.entities.WikittyGroupImpl;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.WikittyUser;
import org.nuiton.wikitty.entities.WikittyUserImpl;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.dto.DTOHelper;
import org.nuiton.wikitty.search.operators.Like;
import org.nuiton.wikitty.services.WikittySecurityUtil;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.l_;

/**
 * User: couteau
 * Date: 30 nov. 2010
 */
public class StorageServiceImpl implements StorageService{

    /**
     * log.
     */
    private static final Log log = LogFactory.getLog(StorageServiceImpl.class);

    public static final String PRIVATE_GROUP = "private";

    WikittyProxy proxy;

    ApplicationConfig config;

    StorageServiceImpl(WikittyProxy proxy, ApplicationConfig config) {
        this.proxy = proxy;
        this.config = config;
    }

    @Override
    public WikittyProxy getProxy() {
        return proxy;
    }

    @Override
    public void setProxy(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    /**
     * Get all the catis.
     *
     * @return a list containing all the catis.
     */
    public List<Cati> getCatis() {

        //Create criteria : everything that got the cati extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Cati.EXT_CATI);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Cati.FQ_FIELD_CATI_NAME);

        //get results
        PagedResult<Cati> catiPagedResult = proxy.findAllByCriteria(Cati.class, criteria);

        List<Cati> catis = new ArrayList<Cati>();

        if (catiPagedResult != null) {
            for (Cati cati:catiPagedResult.getAll()){
                catis.add(DTOHelper.toDto(cati));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Return " + catis.size() + " catis in database");
        }
        return catis;
    }

    /**
     * Save a cati in database.
     *
     * @param cati the cati to save
     * @return the saved cati
     */
    public Cati saveCati(Locale locale, Cati cati)
            throws InvalidEntityException {


        NuitonValidator<Cati> validator =
                NuitonValidatorFactory.newValidator(Cati.class);

        NuitonValidatorResult result = validator.validate(cati);

        checkResults(locale, result);

        if (exist(cati)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.cati.alreadyExist",
                    cati.getName()));
        }

        Cati entity = DTOHelper.fromDto(proxy, Cati.class, cati);
        Cati dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    /**
     * Return a new cati created in database.
     *
     * @return the new cati
     */
    public Cati createCati() {
        Cati cati = new CatiImpl();
        Cati dto = DTOHelper.toDto(cati);

        if (log.isDebugEnabled()) {
            log.debug("Cati " + dto.getWikittyId() + " created");
        }

        return dto;
    }

    /**
     * Delete a cati
     */
    public void deleteCati(Cati cati) {
        proxy.delete(cati.getWikittyId());
    }

    /**
     * Get a cati from its id
     *
     * @param id the cati id
     * @return the cati
     */
    public Cati getCati(String id) {

        Cati cati = proxy.restore(Cati.class, id);

        return DTOHelper.toDto(cati);
    }

    /**
     * Get all the frequencies.
     *
     * @return a list containing all the frequencies.
     */
    public List<Frequency> getFrequencies() throws SecurityException {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Frequency.EXT_FREQUENCY);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Frequency.FQ_FIELD_FREQUENCY_NAME);

        //get results
        PagedResult<Frequency> frequenciesResult = proxy.findAllByCriteria(Frequency.class, criteria);

        List<Frequency> frequencies = new ArrayList<Frequency>();

        //Fill in frequencies with dtos
        if (frequenciesResult != null) {
            for (Frequency frequency:frequenciesResult.getAll()){
                frequencies.add(DTOHelper.toDto(frequency));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Return " + frequencies.size()
                    + " frequencies in database");
        }
        return frequencies;
    }

    /**
     * Save a frequency in database.
     *
     * @param frequency the frequency to save
     * @return the saved frequency
     */
    public Frequency saveFrequency(Locale locale, Frequency frequency)
            throws InvalidEntityException {


        NuitonValidator<Frequency> validator =
                NuitonValidatorFactory.newValidator(Frequency.class);

        NuitonValidatorResult result = validator.validate(frequency);

        checkResults(locale, result);

        if (exist(frequency)) {
            if (log.isDebugEnabled()) {
                log.debug("Frequency " + frequency.getName() + " already exist");
            }

            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.frequency.alreadyExist",
                    frequency.getName()));
        }

        Frequency returnFrequency = null;

        if (frequency != null ){
            Frequency toSaveFrequency = DTOHelper.fromDto(proxy, Frequency.class,frequency);

            Frequency storedFrequency = proxy.store(toSaveFrequency);

            if(log.isDebugEnabled()){
                log.debug("Frequency \"" +
                        storedFrequency.getWikittyId()+ "-" +
                        storedFrequency.getName()+ "\" saved");
            }

            returnFrequency = DTOHelper.toDto(storedFrequency);
        }

        return returnFrequency;
    }

    /**
     * Return a new frequency created in database.
     *
     * @return the new frequency
     */
    public Frequency createFrequency() {
        Frequency frequency = new FrequencyImpl();
        Frequency frequency1 = DTOHelper.toDto(frequency);

        if (log.isDebugEnabled()){
            log.debug("Frequency " + frequency1.getWikittyId() +" created");
        }

        return frequency1;
    }

    /**
     * Delete a frequency
     */
    public void deleteFrequency(Frequency frequency, boolean force)
            throws UsedException {

        int timeUsed = getTimeUsed(frequency);

        if ((force) || (timeUsed == 0)) {
            proxy.delete(frequency.getWikittyId());
        } else {
            throw new UsedException("Frequency used " + timeUsed + " times");
        }
    }

    protected int getTimeUsed(Frequency frequency) {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL)
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_FREQUENCY, frequency.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<AgentSkill> skillPagedResult = proxy.findAllByCriteria(AgentSkill.class, criteria);

        if (log.isDebugEnabled()){
            log.debug("Frequency " + frequency.getWikittyId() + " used " +
                    skillPagedResult.size() + " times");
        }

        return skillPagedResult.size();
    }

    /**
     * Get a frequency from its id
     *
     * @param id the frequency id
     * @return the frequency
     */
    public Frequency getFrequency(String id) {
        Frequency frequencyResult = proxy.restore(Frequency.class, id);
        return DTOHelper.toDto(frequencyResult);
    }

    /**
     * Get all the users.
     *
     * @return a list containing all the users.
     */
    public List<User> getUsers() {

        //Create criteria : everything that got frequency extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(User.FQ_FIELD_USER_LASTNAME);

        //get results
        PagedResult<User> userPagedResult = proxy.findAllByCriteria(User.class, criteria);

        List<User> users = new ArrayList<User>();

        if (userPagedResult != null) {
            for (User user:userPagedResult.getAll()){
                users.add(DTOHelper.toDto(user));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Return " + users.size() + "users in database");
        }
        return users;
    }

    /**
     * Save a user in database.
     *
     * @param user the user to save
     * @return the saved user
     */
    public User saveUser(Locale locale, String context, User user)
            throws InvalidEntityException {

        NuitonValidator<User> validator =
                NuitonValidatorFactory.newValidator(User.class, context);

        NuitonValidatorResult result = validator.validate(user);

        checkResults(locale, result);

        if (exist(user)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.user.alreadyExist",
                    user.getLogin()));
        }

        User entity = DTOHelper.fromDto(proxy, User.class, user);
        User dto = DTOHelper.toDto(proxy.store(entity));

        if (log.isDebugEnabled()){
            log.debug("User \"" + dto.getWikittyId() + " - "
                    + dto.getFirstName() + " " + dto.getLastName()
                    + "\" saved");
        }

        return dto;
    }

    /**
     * Return a new user created in database.
     *
     * @return the new user
     */
    public User createUser() {
        User user = new UserImpl();
        if(log.isDebugEnabled()){
            log.debug("User created : " + user.getWikittyId());
        }

        return DTOHelper.toDto(user);
    }

    /**
     * Delete a user
     */
    public void deleteUser(User user) {
        proxy.delete(user.getWikittyId());
    }

    /**
     * Get a user from its id
     *
     * @param id the user id
     * @return the user
     */
    public User getUser(String id) {
        User user = proxy.restore(User.class, id);
        User dto = DTOHelper.toDto(user);

        if (log.isDebugEnabled()){
            if (dto != null){
                log.debug("Restore User : " + dto.getWikittyId() + " - "
                        + dto.getFirstName() + " " + dto.getLastName());
            }
        }

        return dto;
    }

    @Override
    public Agent getAgent(String id) {

        User user = proxy.restore(User.class, id);
        
        Agent agent = null;
        if (user != null){
            Collection<String> extensions = user.getExtensionNames();
            for (String extension: extensions){
                if (extension.equals(Agent.EXT_AGENT)){
                    
                    Agent testAgent = proxy.restore(Agent.class, id);
                    if (testAgent.getAgent()){
                        agent = testAgent;
                    }
                }
            }
        }
        Agent dto = null;

        if (agent != null){
            dto = DTOHelper.toDto(agent);
        }

        if (log.isDebugEnabled()){
            if (dto != null) {
                log.debug("Restore agent : " + dto.getWikittyId() + " - "
                        + dto.getFirstName() + " " + dto.getLastName());
            }
        }

        return dto;
    }

    @Override
    public EntitiesList getAgentAndDependencies(String id) {
        
        System.out.println("id : " + id);
        
        Agent agent = getAgent(id);
        
        System.out.println("agent : " + agent);

        List<Agent> agents = new ArrayList<Agent>();
        agents.add(agent);

        EntitiesList result = createAgentList(agents);

        return result;
    }

    @Override
    public Agent setAgent(User user) {

        User userEntity = DTOHelper.fromDto(proxy, User.class, user);

        Agent agent = proxy.cast(userEntity, Agent.class);
        agent.setAgent(true);
        if (log.isDebugEnabled()) {
            log.debug("Transform user " + agent.getWikittyId() + " - "
                    + agent.getFirstName() + " " + agent.getLastName()
                    + " to agent");
        }
        return DTOHelper.toDto(agent);
    }

    @Override
    public void unsetAgent(String id) {

        Agent agent = proxy.restore(Agent.class, id);
        if (agent != null) {
            agent.setAgent(false);
        }

    }

    @Override
    public Agent saveAgent(Locale locale, String context, Agent agent)
            throws InvalidEntityException {

        NuitonValidator<Agent> validator =
                NuitonValidatorFactory.newValidator(Agent.class, context);

        NuitonValidatorResult result = validator.validate(agent);

        checkResults(locale, result);

        if (exist(agent)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.user.alreadyExist",
                    agent.getLogin()));
        }

        Agent entity = DTOHelper.fromDto(proxy, Agent.class, agent);

        Agent oldEntity = proxy.restore(Agent.class, entity.getWikittyId());

        Agent savedAgent = proxy.store(entity);

        WikittyGroup privateGroup = getPrivateGroup();

        if(savedAgent.getAgent()) {
            privateGroup.addMembers(savedAgent.getWikittyId());
        } else {
            privateGroup.removeMembers(savedAgent.getWikittyId());
        }

        proxy.store(privateGroup);

        //Send email if password change
        if (oldEntity == null ||
                oldEntity.getPassword() == null ||
                !oldEntity.getPassword().equals(entity.getPassword())) {
            sendPassword(locale, entity.getWikittyId());
        }

        Agent dto = DTOHelper.toDto(savedAgent);

        if (log.isDebugEnabled()){
            log.debug("Agent \"" + dto.getWikittyId() + " - "
                    + dto.getFirstName() + " " + dto.getLastName()
                    + "\" saved");
        }

        return dto;
    }

    @Override
    public Skill saveSkill(Locale locale, Skill skill)
            throws InvalidEntityException {

        NuitonValidator<Skill> validator =
                NuitonValidatorFactory.newValidator(Skill.class);

        NuitonValidatorResult result = validator.validate(skill);

        checkResults(locale, result);

        if (exist(skill)) {
            if(log.isDebugEnabled()){
                log.debug("Skill " + skill.getName() + " already exist");
            }

            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.skill.alreadyExist",
                    skill.getName()));
        }

        Skill entity = DTOHelper.fromDto(proxy, Skill.class, skill);

        entity = proxy.store(entity);

        Skill dto = DTOHelper.toDto(entity);

        if (log.isDebugEnabled()){
            log.debug("Saved skill : " + dto.getWikittyId() + " - "
                    + dto.getName());
        }

        return dto;
    }

    @Override
    public Skill createSkill(String parentId) {
        Skill skill = new SkillImpl();
        if ((parentId != null)&&(!"".equals(parentId))){
            skill.setParent(parentId);
        }
        Skill dto = DTOHelper.toDto(skill);

        if (log.isDebugEnabled()) {
            log.debug("Skill \"" + dto.getWikittyId() + " - " + dto.getName()
                    + "\" created");
        }

        return dto;
    }

    /**
     * Delete a skill present in database. Check if the skill is used before
     * deleting. If the skill is used, throw an exception.
     * @param skill the skill to delete
     */
    @Override
    public void deleteSkill(Skill skill) throws UsedException {

        //if not used, delete otherwise, throw an exception
        if (!isSkillUsed(skill)){
            List<Skill> children = getChildrenSkill(skill.getWikittyId());
            for (Skill child:children) {
                deleteSkill(child);
            }
            proxy.delete(skill.getWikittyId());
        } else {
            throw new UsedException("Skill still used by users");
        }
    }

    /**
     * Check if a skill is used by users in their skills description
     * @param skill the skill to check
     * @return true if it is used
     */
    protected boolean isSkillUsed(Skill skill) {

        boolean b = false;

        List<Skill> children = getChildrenSkill(skill.getWikittyId());


        for(Skill child:children) {
            b = b || isSkillUsed(child);
        }

        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL)
                .or()
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_DOMAINSKILL,
                        skill.getWikittyId())
                .eq(AgentSkill.FQ_FIELD_AGENTSKILL_TECHNICALSKILL,
                        skill.getWikittyId());

        Criteria criteria = search.criteria();

        PagedResult<Wikitty> agentSkillPagedResult = proxy.findAllByCriteria(criteria);

        int size = agentSkillPagedResult.size();

        if(log.isDebugEnabled()){
            log.debug("Number of time the Skill " + skill.getName() +
                    " is used : " + size);
        }
         b = b || size != 0;

        return b;
    }

    @Override
    public Skill getSkill(String id) {
        Skill skill = proxy.restore(Skill.class, id);
        if (log.isDebugEnabled()) {
            if (skill != null) {
                log.debug("Restore skill \"" + skill.getWikittyId() + " - "
                        + skill.getName() +"\"");
            }
        }
        Skill dto = null;
        if (skill != null){
            dto = DTOHelper.toDto(skill);
        }
        return dto;
    }

    protected Boolean exist(Skill skill){

        Search search;

        String name = skill.getName();

        String parent = skill.getParent();

        if (parent != null ){
            search = Search.query().eq(Element.ELT_EXTENSION, Skill.EXT_SKILL)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_NAME, name)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT, parent)
                    .neq(Element.ELT_ID, skill.getWikittyId());
        } else {
            search = Search.query().eq(Element.ELT_EXTENSION, Skill.EXT_SKILL)
                    .eq(Skill.FQ_FIELD_WIKITTYTREENODE_NAME, name)
                    .isNull(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT)
                    .neq(Element.ELT_ID, skill.getWikittyId());
        }

        Criteria criteria = search.criteria();

        //get results
        PagedResult<Skill> skills = proxy.findAllByCriteria(Skill.class,
                criteria);

        return (skills.size()>0);
    }

    /**
     * Check if a user is already present in database
     *
     * @param user the user to check
     * @return true is exist, false otherwise
     */
    protected boolean exist(User user) {

        Search search = Search.query()
                .eq(User.FQ_FIELD_WIKITTYUSER_LOGIN, user.getLogin())
                .neq(Element.ELT_ID, user.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<User> userPagedResult = proxy.findAllByCriteria(User.class, criteria);

        return (userPagedResult.size() > 0);
    }

    protected Boolean exist(Frequency frequency) {
        //Create criteria
        Search search = Search.query()
                .eq(Frequency.FQ_FIELD_FREQUENCY_NAME, frequency.getName())
                .neq(Element.ELT_ID, frequency.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Frequency> frequencyPagedResult = proxy.findAllByCriteria(Frequency.class, criteria);

        //Remove the frequency from the results
        List<Frequency> frequencies = new ArrayList<Frequency>(frequencyPagedResult.getAll());
        frequencies.remove(frequency);

        return (frequencies.size() > 0);
    }

    /**
     * Check if a cati is already present in database
     *
     * @param cati the cati to check
     * @return true is exist, false otherwise
     */
    protected boolean exist(Cati cati) {
        //Create criteria
        Search search = Search.query()
                .eq(Cati.FQ_FIELD_CATI_NAME, cati.getName())
                .neq(Element.ELT_ID, cati.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Cati> catiPagedResult = proxy.findAllByCriteria(Cati.class, criteria);

        return (catiPagedResult.size() > 0);
    }

    public Boolean login(String login, String password){
        try {
            proxy.login(login, password);
            log.info("User \"" + login + "\" logged in");
        } catch (Exception eee) {
            if (log.isDebugEnabled()) {
                log.debug("Error at login : " + eee);
            }
            return false;
        }
        return true;
    }

    public void logout(){
        proxy.logout();
        if (log.isInfoEnabled()){
            log.info("Logout");
        }
    }

    public List<Skill> getChildrenSkill(String id) {

        if (log.isDebugEnabled()){
            log.debug("Get children for skill : " + id);
        }

        //Create criteria : everything that got frequency extension
        Search search;
        if ((id != null)&& !"".equals(id)){
            search = Search.query()
                .eq(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT,id);
        } else {
            search = Search.query()
                    .isNull(Skill.FQ_FIELD_WIKITTYTREENODE_PARENT);
        }

        Criteria criteria = search.criteria();
        criteria.addSortAscending(Skill.FQ_FIELD_WIKITTYTREENODE_NAME);

        //get results
        PagedResult<Skill> skillPagedResult = proxy.findAllByCriteria(Skill.class, criteria);

        List<Skill> skills = new ArrayList<Skill>();

        if (skillPagedResult != null) {
            for (Skill skill:skillPagedResult.getAll()){
                skills.add(DTOHelper.toDto(skill));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Got " + skills.size() + " children");
        }
        return skills;
    }

    @Override
    public Department saveDepartment(Locale locale, Department department)
            throws InvalidEntityException {

        NuitonValidator<Department> validator =
                NuitonValidatorFactory.newValidator(Department.class);

        NuitonValidatorResult result = validator.validate(department);

        checkResults(locale, result);

        if (exist(department)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.department.alreadyExist",
                    department.getName()));
        }

        Department entity = DTOHelper.fromDto(proxy, Department.class, department);

        if (log.isDebugEnabled()) {
            log.debug("Save department : " + entity.getWikittyId() + " - "
                    + entity.getName());
        }

        Department dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    @Override
    public Department saveDepartment(Locale locale,
                                     Department department,
                                     List<String> units)
            throws InvalidEntityException {

        NuitonValidator<Department> validator =
                NuitonValidatorFactory.newValidator(Department.class);

        NuitonValidatorResult result = validator.validate(department);

        checkResults(locale, result);

        if (exist(department)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.department.alreadyExist",
                    department.getName()));
        }

        Department entity = DTOHelper.fromDto(proxy, Department.class, department);


        //get the actual links
        List<UnitDeptLink> links = getDepartmentLinks(entity);

        //clear the links
        entity.clearUnitDeptLink();

        //iterate on the departments and the actual links.
        for (String unit : units) {
            Boolean gotLink = false; //if already present link found
            for (Iterator<UnitDeptLink> iterator = links.iterator(); iterator.hasNext();) {
                UnitDeptLink link = iterator.next();
                if (unit.equals(link.getUnit()) && !gotLink) {
                    //unit found on links, remove it from the list of links
                    //stop looping when unit found once, this will remove
                    //useless extra links if found
                    //add the link to the department
                    entity.addUnitDeptLink(link.getWikittyId());
                    gotLink = true;
                    iterator.remove();
                    log.debug("found and add link : " + link);
                }
            }

            //if unit not already present, create a new link, store it,
            //add it to the department
            if (!gotLink) {
                UnitDeptLink newLink = new UnitDeptLinkImpl();
                newLink.setDepartment(department.getWikittyId());
                newLink.setUnit(unit);
                if (log.isDebugEnabled()) {
                    log.debug("Store link : " + newLink);
                }
                newLink = proxy.store(newLink);
                log.debug("add link : " + newLink);
                entity.addUnitDeptLink(newLink.getWikittyId());
            }
        }

        //Remove the links that did not get updated, they are not present anymore
        //This will remove double links and old links
        for (UnitDeptLink link : links) {
            log.debug("Remove link : " + link);
            proxy.delete(link.getWikittyId());
        }

        //restore department with all its links and return it
        Department savedDepartment = proxy.store(entity);

        if (log.isDebugEnabled()) {
            log.debug("Saved department : " + savedDepartment.getWikittyId()
                    + " - " + savedDepartment.getName());
        }

        return DTOHelper.toDto(savedDepartment);
    }

    public List<UnitDeptLink> getDepartmentLinks(Department entity) {

        //Create criteria : everything that got the unitDeptLink extension with
        //the department
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_DEPARTMENT, entity.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult = proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<UnitDeptLink> unitDeptLinks = new ArrayList<UnitDeptLink>();

        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink link : unitDeptLinkPagedResult.getAll()) {
                unitDeptLinks.add(DTOHelper.toDto(link));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Got " + unitDeptLinks.size() + " links to department "
                    + entity.getWikittyId() + " - " + entity.getName());
        }
        return unitDeptLinks;
    }

    @Override
    public Department createDepartment() {
        Department department = new DepartmentImpl();
        Department dto = DTOHelper.toDto(department);

        if (log.isDebugEnabled()) {
            log.debug("Skill \"" + dto.getWikittyId() + " - " + dto.getName() + "\" created");
        }

        return dto;
    }

    @Override
    public void deleteDepartment(Department department) {
        proxy.delete(department.getWikittyId());
    }

    @Override
    public Department getDepartment(String id) {
        Department department = proxy.restore(Department.class, id);
        return DTOHelper.toDto(department);
    }

    /**
     * Get all the departments.
     *
     * @return a list containing all the departments.
     */
    @Override
    public List<Department> getDepartments() {

        //Create criteria : everything that got the department extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Department.EXT_DEPARTMENT);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Department.FQ_FIELD_DEPARTMENT_NAME);

        //get results
        PagedResult<Department> departmentPagedResult = proxy.findAllByCriteria(Department.class, criteria);

        List<Department> departments = new ArrayList<Department>();

        if (departmentPagedResult != null) {
            for (Department department : departmentPagedResult.getAll()) {
                departments.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Departments list : " + departments);
        }
        return departments;
    }

    @Override
    public List<Department> getDepartments(String id) {

        //Create criteria : everything that got the unitdptlink extension and the unit id
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_UNIT, id);
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult =
                proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<Department> departments = new ArrayList<Department>();

        //for each link get the department and add it to the list
        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink unitDeptLink : unitDeptLinkPagedResult.getAll()) {

                Department department = proxy.restore(Department.class,
                        unitDeptLink.getDepartment());

                departments.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Departments list : " + departments);
        }
        
        return departments;
    }

    protected Boolean exist(Department department) {
        //Create criteria
        Search search = Search.query()
                .eq(Department.FQ_FIELD_DEPARTMENT_NAME, department.getName())
                .neq(Element.ELT_ID, department.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Department> departmentPagedResult = proxy.findAllByCriteria(Department.class, criteria);

        return (departmentPagedResult.size() > 0);
    }

    @Override
    public Unit saveUnit(Locale locale, Unit unit)
            throws InvalidEntityException {

        NuitonValidator<Unit> validator =
                NuitonValidatorFactory.newValidator(Unit.class);

        NuitonValidatorResult result = validator.validate(unit);

        checkResults(locale, result);

        if (exist(unit)) {
            throw new InvalidEntityException("Unit named " +
                    unit.getName() + " already exist in database");
        }

        Unit entity = DTOHelper.fromDto(proxy, Unit.class, unit);

        if (log.isDebugEnabled()) {
            log.debug("Try to save unit : " + entity);
        }

        Unit dto = DTOHelper.toDto(proxy.store(entity));

        return dto;
    }

    @Override
    public Unit saveUnit(Locale locale, Unit unit, List<String> departments)
            throws InvalidEntityException {

        NuitonValidator<Unit> validator =
                NuitonValidatorFactory.newValidator(Unit.class);

        NuitonValidatorResult result = validator.validate(unit);

        checkResults(locale, result);

        if (exist(unit)) {
            throw new InvalidEntityException(
                    l_(locale, "refcomp.admin.unit.alreadyExist",
                    unit.getName()));
        }

        Unit entity = DTOHelper.fromDto(proxy, Unit.class, unit);

        //get the actual links
        List<UnitDeptLink> links = getUnitLinks(entity);

        //clear the links
        entity.clearUnitDeptLink();

        //iterate on the departments and the actual links.
        for(String department:departments){
            Boolean gotLink = false; //if already present link found
            for (Iterator<UnitDeptLink> iterator = links.iterator(); iterator.hasNext();){
                UnitDeptLink link = iterator.next();
                if (department.equals(link.getDepartment()) && !gotLink){
                    //department found on links, remove it from the list of links
                    //stop looping when department found once, this will remove
                    //useless extra links if found
                    //add the link to the unit
                    entity.addUnitDeptLink(link.getWikittyId());
                    gotLink = true;
                    iterator.remove();
                    log.debug("found and add link : " + link);
                }
            }

            //if department not already present, create a new link, store it,
            //add it to the unit
            if (!gotLink){
                UnitDeptLink newLink = new UnitDeptLinkImpl();
                newLink.setDepartment(department);
                newLink.setUnit(unit.getWikittyId());
                if (log.isDebugEnabled()){
                    log.debug("Store link : " + newLink);
                }
                newLink = proxy.store(newLink);
                log.debug("add link : " + newLink);
                entity.addUnitDeptLink(newLink.getWikittyId());
            }
        }

        //Remove the links that did not get updated, they are not present anymore
        //This will remove double links and old links
        for(UnitDeptLink link:links){
            log.debug("Remove link : " + link);
            proxy.delete(link.getWikittyId());
        }

        //restore unit with all its links and return it
        Unit savedUnit = proxy.store(entity);

        if (log.isDebugEnabled()){
            log.debug("Stored unit : " + savedUnit);
        }

        return DTOHelper.toDto(savedUnit);
    }

    public List<UnitDeptLink> getUnitLinks(Unit entity) {

        //Create criteria : everything that got the unitDeptLink extension with
        //the unit
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_UNIT, entity.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult = proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<UnitDeptLink> unitDeptLinks = new ArrayList<UnitDeptLink>();

        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink link : unitDeptLinkPagedResult.getAll()) {
                unitDeptLinks.add(DTOHelper.toDto(link));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Links list : " + unitDeptLinks);
        }
        return unitDeptLinks;
    }

    @Override
    public Unit createUnit() {
        Unit unit = new UnitImpl();
        Unit dto = DTOHelper.toDto(unit);

        if (log.isDebugEnabled()) {
            log.debug("Unit " + dto + " created");
        }

        return dto;
    }

    @Override
    public void deleteUnit(Unit unit) {
        proxy.delete(unit.getWikittyId());
    }

    @Override
    public Unit getUnit(String id) {
        Unit unit = proxy.restore(Unit.class, id);
        return DTOHelper.toDto(unit);
    }

    @Override
    public List<Unit> getUnits(String id) {

        //Create criteria : everything that got the unitdptlink extension and the department id
        Search search = Search.query().eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK)
                .eq(UnitDeptLink.FQ_FIELD_UNITDEPTLINK_DEPARTMENT, id);
        Criteria criteria = search.criteria();

        //get results
        PagedResult<UnitDeptLink> unitDeptLinkPagedResult =
                proxy.findAllByCriteria(UnitDeptLink.class, criteria);

        List<Unit> units = new ArrayList<Unit>();

        //for each link get the department and add it to the list
        if (unitDeptLinkPagedResult != null) {
            for (UnitDeptLink unitDeptLink : unitDeptLinkPagedResult.getAll()) {

                Unit department = proxy.restore(Unit.class,
                        unitDeptLink.getUnit());

                units.add(DTOHelper.toDto(department));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Units list : " + units);
        }

        return units;
    }

    /**
     * Get all the departments.
     *
     * @return a list containing all the departments.
     */
    @Override
    public List<Unit> getUnits() {

        //Create criteria : everything that got the cati extension
        Search search = Search.query().eq(Element.ELT_EXTENSION, Unit.EXT_UNIT);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(Unit.FQ_FIELD_UNIT_NAME);

        //get results
        PagedResult<Unit> unitPagedResult = proxy.findAllByCriteria(Unit.class, criteria);

        List<Unit> units = new ArrayList<Unit>();

        if (unitPagedResult != null) {
            for (Unit unit : unitPagedResult.getAll()) {
                units.add(DTOHelper.toDto(unit));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Units list : " + units);
        }
        return units;
    }

    @Override
    public void saveEntity(BusinessEntity entity) {
        proxy.store(entity);
    }

    @Override
    public void deleteEntity(BusinessEntity entity) {
        proxy.delete(entity.getWikittyId());
    }

    @Override
    public String getAdminEmail() {
        RefcompConfiguration config = RefcompConfiguration.getInstance();
        return RefcompConfigurationHelper.getAdminEmail(config);
    }

    @Override
    public AgentSkill createAgentSkill(String agentId) {
        //force the impl for the authorisation.
        AgentSkillImpl skill = new AgentSkillImpl();
        skill.setAgent(agentId);
        //WikittyAuthorisation authorisation = new WikittyAuthorisationImpl(skill);
        //authorisation.setOwner(agentId);

        //skill = WikittyUtil.cast(authorisation, AgentSkillImpl.class);

        AgentSkill dto = DTOHelper.toDto(skill);

        if (log.isDebugEnabled()) {
            log.debug("AgentSkill " + dto + " created");
        }

        return dto;
    }

    @Override
    public void saveAgentSkill(Locale locale, AgentSkill skill)
            throws InvalidEntityException{


        NuitonValidator<AgentSkill> validator =
                NuitonValidatorFactory.newValidator(AgentSkill.class);

        NuitonValidatorResult result = validator.validate(skill);

        checkResults(locale, result);

        AgentSkill entity = DTOHelper.fromDto(proxy, AgentSkill.class, skill);
        proxy.store(entity);
    }

    protected WikittyGroup getPrivateGroup() {

        //Create criteria
        Search search = Search.query().eq(
                WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, PRIVATE_GROUP);
        Criteria criteria = search.criteria();

        //get results
        WikittyGroup group = proxy.findByCriteria(WikittyGroup.class, criteria);

        return group;
    }

    protected Boolean exist(Unit unit) {
        //Create criteria
        Search search = Search.query()
                .eq(Unit.FQ_FIELD_UNIT_NAME, unit.getName())
                .neq(Element.ELT_ID, unit.getWikittyId());
        Criteria criteria = search.criteria();

        //get results
        PagedResult<Unit> unitPagedResult = proxy.findAllByCriteria(Unit.class, criteria);

        return (unitPagedResult.size() > 0);
    }

    @Override
    public boolean isAdmin(){
        try {
            boolean isMember = proxy.isMember(WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);

            if (log.isDebugEnabled()){
                log.info("User admin : " + isMember);
            }

            return isMember;
        } catch (SecurityException eee) {
            return false;
        }
    }

    @Override
    public boolean isAdmin(String id) {
        try {
            boolean result = false;

            //Find the group from its name
            Search search = Search.query()
                    .eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                            WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME)
                    .contains(WikittyGroup.FQ_FIELD_WIKITTYGROUP_MEMBERS,
                            id);
            Criteria criteria = search.criteria();

            Wikitty group = proxy.findByCriteria(criteria);

            if (group != null) {
                log.debug("User " + id + "is admin");
                result = true;
            }

            return result;
        } catch (SecurityException eee) {
            log.debug("Security Exception while checking admin");
            return false;
        }
    }

    @Override
    public boolean isAgent() {
        try {
            WikittyUser user = proxy.getLoggedInUser();
            if (user != null) {
                Agent agent = proxy.restore(Agent.class, user.getWikittyId());
                if (agent == null) {
                    return false;
                } else {
                    return agent.getAgent();
                }
            } else {
                return false;
            }
        }  catch (SecurityException eee) {
            return false;
        }
    }

    @Override
    public Boolean isLoggedIn(){
        try {
            WikittyUser user = proxy.getLoggedInUser();
            if (user == null) {
                return false;
            }
        } catch (SecurityException eee) {
            return false;
        }
        log.info("User logged in");
        return true;
    }

    @Override
    public void adminCreationIfDoNotExist(){
        Search search = Search.query().
                eq(WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN,
                        RefcompConfigurationHelper.getAdminLogin(config));

        Criteria criteria = search.criteria();

        //get results
        PagedResult<WikittyUser> userPagedResult = proxy.findAllByCriteria(WikittyUser.class, criteria);

        if (userPagedResult.size() == 0 || userPagedResult.getFirst() == null){

            //Admin does not exist, create it
            WikittyUser admin = new WikittyUserImpl();
            admin.setLogin(RefcompConfigurationHelper.getAdminLogin(config));
            admin.setPassword(RefcompConfigurationHelper.getAdminPassword(config));
            admin = proxy.store(admin);

            WikittyGroup adminGroup = WikittySecurityUtil.createAppAdminGroup(admin);
            
            if (log.isDebugEnabled()){
                log.debug("Admin group : " + adminGroup);
            }

            proxy.store(adminGroup);

            if (log.isInfoEnabled()) {
                log.info("Admin account created");
            }
        } else {
            //attach admin to adminGroup if it does exist
            WikittyUser admin = userPagedResult.getFirst();

            //Find the group from its name
            Search searchGroup = Search.query()
                    .eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                            WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);
            Criteria criteriaGroup = searchGroup.criteria();
            WikittyGroup adminGroup = proxy.findByCriteria(WikittyGroup.class, criteriaGroup);

            if (adminGroup == null) {
                adminGroup = WikittySecurityUtil.createAppAdminGroup(admin);
            }else if ((adminGroup.getMembers() == null)||(!adminGroup.getMembers().contains(admin.getWikittyId()))) {
                adminGroup.addMembers(admin.getWikittyId());
            }
            proxy.store(adminGroup);
        }


    }

    @Override
    public void createPrivateGroupIfDoNotExist() {
        Search search = Search.query().
                eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, PRIVATE_GROUP);

        Criteria criteria = search.criteria();

        //get results
        PagedResult<WikittyGroup> groupPagedResult = proxy.findAllByCriteria(WikittyGroup.class, criteria);

        //get admin, add it to the members of the private grou
        Search adminSearch = Search.query().
                eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                        WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);
        Criteria adminCriteria = adminSearch.criteria();
        PagedResult<WikittyGroup> adminPagedResult = proxy.findAllByCriteria(WikittyGroup.class, adminCriteria);

        if (groupPagedResult.size() == 0) {

            WikittyGroup privateGroup = new WikittyGroupImpl();
            privateGroup.setName(PRIVATE_GROUP);

            if (adminPagedResult.size() > 0) {
                privateGroup.addAllMembers(adminPagedResult.getFirst().getMembers());
            }

            proxy.store(privateGroup);
            log.info("Private group created");
        }
    }

    @Override
    public void setPrivateSkills(List<String> ids) {

        List<WikittyAuthorisation> skills =
                proxy.restore(WikittyAuthorisation.class, ids);
        WikittyGroup privateGroup = getPrivateGroup();
        Set<String> privateSet = new HashSet<String>();
        privateSet.add(privateGroup.getWikittyId());

        for(WikittyAuthorisation skill:skills) {
            log.debug("Set skill private");
            skill.setReader(privateSet);
        }
        proxy.store(skills);
    }

    @Override
    public void setPublicSkills(List<String> ids) {

        List<WikittyAuthorisation> skills =
                proxy.restore(WikittyAuthorisation.class, ids);

        //set reader to null so that everybody can read
        for (WikittyAuthorisation skill : skills) {
            log.debug("Set skill public");
            skill.setReader(null);
        }
        proxy.store(skills);
    }

    @Override
    public void clear() {
        proxy.clear();
    }

    @Override
    public EntitiesList getSkills(String agentId){

        WikittyGroup privateGroup = getPrivateGroup();
        WikittyUser user = getLoggedInUser();

        EntitiesList result = null;

        if (agentId != null) {
            Search search = Search.query()
                    .eq(AgentSkill.FQ_FIELD_AGENTSKILL_AGENT, agentId);

            Set<String> members = privateGroup.getMembers();

            if (members != null && user != null && members.contains(user.getWikittyId())){
                //do nothing, criteria is ok
            } else {
                search.neq(
                        WikittyAuthorisation.FQ_FIELD_WIKITTYAUTHORISATION_READER,
                        privateGroup.getWikittyId());
            }

            Criteria criteria = search.criteria();

            PagedResult<AgentSkill> skills = proxy.findAllByCriteria(AgentSkill.class, criteria);

            Set<AgentSkill> skillSet = new HashSet<AgentSkill>();
            skillSet.addAll(skills.getAll());

            result = createAgentSkillList(skillSet);
        }

        return result;
    }

    @Override
    public AgentSkill addSkill(String agentId) {

        //create the new Skill and store it
        AgentSkill newSkill = new AgentSkillImpl();

        newSkill.setAgent(agentId);

        AgentSkill dto = DTOHelper.toDto(newSkill);

        return dto;
    }

    @Override
    public EntitiesList addSkill(String agentId, Skill skill, String frequencyId) {

        //create the new Skill and store it
        AgentSkill newSkill = new AgentSkillImpl();

        newSkill.setAgent(agentId);
        newSkill.setFrequency(frequencyId);
        newSkill.setTechnicalSkill(skill.getWikittyId());

        AgentSkill dto = DTOHelper.toDto(newSkill);

        Set<AgentSkill> skills = new HashSet<AgentSkill>();

        skills.add(dto);

        EntitiesList result = createAgentSkillList(skills);

        return result;
    }

    @Override
    public void removeSkill(String skillId) {
        //delete skill from database
        proxy.delete(skillId);
    }

    @Override
    public WikittyUser getLoggedInUser() {
        return proxy.getLoggedInUser();
    }

    @Override
    public EntitiesList saveMySkills(Locale locale, List<AgentSkill> skills)
            throws InvalidEntityException{

        NuitonValidator<AgentSkill> validator =
                NuitonValidatorFactory.newValidator(AgentSkill.class);

        Map<String, String> messages = new HashMap<String, String>();

        Set<String> skillsList = new HashSet<String>();

        //Validate each skill one by one and add messages to the map
        for (AgentSkill skill : skills) {
            NuitonValidatorResult result = validator.validate(skill);
            if (!result.isValid()) {
                if (log.isDebugEnabled()) {
                    log.debug("Entity not valid");
                }
                List<String> fields = result.getFieldsForError();
                for (String field : fields) {
                    List<String> errorMessages = result.getErrorMessages(field);
                    for (String message : errorMessages) {
                        messages.put(_(message), l_(locale, message));
                    }
                }
            } else {
                skillsList.add(skill.getTechnicalSkill());
                if (log.isDebugEnabled()) {
                    log.debug("Entity valid, will be saved");
                }
            }
        }

        if (skillsList.size() != skills.size()) {
            messages.put(_("refcomp.duplicate.skills"),
                    l_(locale, "refcomp.duplicate.skills"));
        }

        //Throw an exception if one skill is not valid
        if (!messages.isEmpty()) {
            String errorMessage = "";
            for (String message:messages.values()) {
                errorMessage += message + "\n";
            }
            throw new InvalidEntityException(errorMessage);
        }

        List<AgentSkill> skillsEntities = new ArrayList<AgentSkill>();

        String agentId = getLoggedInUser().getWikittyId();

        for (AgentSkill skill : skills) {
            AgentSkill entity = DTOHelper.fromDto(proxy, AgentSkill.class, skill);
            entity.setAgent(agentId);

            //check if got authorisation extension, if not add it
            if(!entity.getExtensionNames().contains(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION)) {
                WikittyAuthorisation authorisation = new WikittyAuthorisationImpl((AgentSkillImpl)entity);
                authorisation.setOwner(agentId);
            }

            skillsEntities.add(entity);
        }

        proxy.store(skillsEntities);

        return getSkills(agentId);
    }

    @Override
    public AgentSkill getAgentSkill(String skillId) {
        AgentSkill restoredSkill = proxy.restore(AgentSkill.class, skillId);
        AgentSkill dto = DTOHelper.toDto(restoredSkill);
        return dto;
    }

    @Override
    public EntitiesList searchUser(String search, String cati,
                                      String department, String unit,
                                      String frequency) {

        if (search == null || "".equals(search) || "null".equals(search)) {
            search = "*";
        }

        search = search.toLowerCase();

        if (log.isDebugEnabled()) {
            log.debug("Search for \"" + search +"\" in users");
        }

        String[] queryTerms = search.split(" ");

        Search userQuery = Search.query();

        for (String term:queryTerms){
            userQuery.keyword(term);
        }

        if (cati != null && !"null".equals(cati)) {
            log.debug("Filter search on cati " + cati);
            userQuery.eq(Agent.FQ_FIELD_AGENT_CATI, cati);
        }

        if (unit != null && !"null".equals(unit)) {
            log.debug("Filter search on unit " + unit);
            userQuery.contains(Agent.FQ_FIELD_AGENT_UNIT, unit);
        }

        if (department != null && !"null".equals(department)) {
            log.debug("Filter search on department " + department);
            userQuery.eq(Agent.FQ_FIELD_AGENT_DEPARTMENT, department);
        }

        Criteria userCriteria = userQuery.criteria();

        PagedResult<Agent> agentsResult = proxy.findAllByCriteria(Agent.class, userCriteria);

        List<Agent> agents = agentsResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug("Found " + agents.size() + " agents");
        }

        EntitiesList result = createAgentList(agents);

        return result;
    }

    @Override
    public EntitiesList searchSkill(String search, String cati,
                                      String department, String unit,
                                      String frequency) {

        if (search != null) {
            search = URLDecoder.decode(search);
        }

        if (log.isDebugEnabled()) {
            log.debug("Search for \"" + search +"\" in skills");
        }

        //necessary for security checks
        WikittyGroup privateGroup = getPrivateGroup();
        WikittyUser user = getLoggedInUser();
        Set<String> members = privateGroup.getMembers();


        // *******************************************************************
        // *                     FIRST PART                                  *
        // *******************************************************************
        // First get all the users that can be in the result set (filter on
        // cati/unit/department)
        Search userQuery = Search.query().eq(Element.ELT_EXTENSION,
                Agent.EXT_AGENT);

        //Cati filter
        if (cati != null && !"null".equals(cati)) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_CATI, cati);
        }

        //Unit filter
        if (unit != null && !"null".equals(unit)) {
            userQuery.contains(Agent.FQ_FIELD_AGENT_UNIT, unit);
        }

        //Department filter
        if (department != null && !"null".equals(department)) {
            userQuery.eq(Agent.FQ_FIELD_AGENT_DEPARTMENT, department);
        }

        //add right restrictions
        if (members != null && user != null &&
                members.contains(user.getWikittyId())) {
            //do nothing, criteria is ok
        } else {
            userQuery.neq(
                    WikittyAuthorisation.FQ_FIELD_WIKITTYAUTHORISATION_READER,
                    privateGroup.getWikittyId());
        }

        //Get users
        Criteria userCriteria = userQuery.criteria();
        PagedResult<Wikitty> agents = proxy.findAllByCriteria(userCriteria);
        Set<String> agentIds = new HashSet<String>();
        for (Wikitty agent : agents.getAll()) {
            agentIds.add(agent.getId());
        }

        if(log.isDebugEnabled()) {
            log.debug("Agents found : " + agentIds);
        }


        // *******************************************************************
        // *                     SECOND PART                                 *
        // *******************************************************************
        //Second, get all the skills that match the query
        if (search == null || "".equals(search) || "null".equals(search)){
            search="*";
        }

        search = search.toLowerCase();

        String[] queryTerms = StringUtil.split(search,  ",");

        //Lists of set of agents that can be returned by the request. Will be
        //merged at the end to determine which agents must be returned by the
        //search.
        Set<AgentSkill> skillsResult = new HashSet<AgentSkill>();

        for (String term : queryTerms) {
            // ****************************************************************
            // *                       THIRD PART                             *
            // ****************************************************************
            //Iterate on each search term

            Search skillQuery = Search.query().eq(Element.ELT_EXTENSION,
                    Skill.EXT_SKILL);

            //add right restrictions
            if (members != null && members.contains(user.getWikittyId())) {
                //do nothing, criteria is ok
            } else {
                skillQuery.neq(
                        WikittyAuthorisation.FQ_FIELD_WIKITTYAUTHORISATION_READER,
                        privateGroup.getWikittyId());
            }

            skillQuery.like(Skill.FQ_FIELD_WIKITTYTREENODE_NAME, term,
                    Like.SearchAs.AsText);

            Criteria skillCriteria = skillQuery.criteria();

            PagedResult<Wikitty> skills = proxy.findAllByCriteria(skillCriteria);
            List<String> skillIds = new ArrayList<String>();
            for (Wikitty skill : skills.getAll()) {
                skillIds.add(skill.getId());
            }
            
            if(log.isDebugEnabled()) {
                log.debug("Skills found for \"" + term + "\" : " + skillIds);
            }

            Set<String> resultsIds = new HashSet<String>();

            if (!agentIds.isEmpty() && !skillIds.isEmpty()) {
                //Then get all the AgentSkills that match the search term and filter
                Search agentSkillQuery = Search.query();

                if (frequency != null && !"null".equals(frequency)) {
                    agentSkillQuery.eq(AgentSkill.FQ_FIELD_AGENTSKILL_FREQUENCY, frequency);
                }

                agentSkillQuery.in(AgentSkill.FQ_FIELD_AGENTSKILL_AGENT, agentIds);
                agentSkillQuery.in(AgentSkill.FQ_FIELD_AGENTSKILL_TECHNICALSKILL, skillIds);

                //add right restrictions
                if (members != null && members.contains(user.getWikittyId())) {
                    //do nothing, criteria is ok
                } else {
                    agentSkillQuery.neq(
                            WikittyAuthorisation.FQ_FIELD_WIKITTYAUTHORISATION_READER,
                            privateGroup.getWikittyId());
                }

                Criteria skillsCriteria = agentSkillQuery.criteria();

                skillsResult.addAll(proxy.findAllByCriteria(
                        AgentSkill.class, skillsCriteria).getAll());
            }
        }

        EntitiesList result = createAgentSkillList(skillsResult);

        return result;
    }

    @Override
    public void setAdmin(String agentId, boolean admin) {
        Search searchGroup = Search.query().eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                WikittySecurityUtil.WIKITTY_APPADMIN_GROUP_NAME);
        Criteria groupCriteria = searchGroup.criteria();
        WikittyGroup adminGroup = proxy.findByCriteria(WikittyGroup.class,
                groupCriteria);
        if (admin) {
            if (log.isInfoEnabled()) {
                log.info("Add agent " + agentId + " to admin group");
            }
            adminGroup.addMembers(agentId);
            proxy.store(adminGroup);
        } else {
            if (log.isInfoEnabled()) {
                log.info("Remove agent " + agentId + " from admin group");
            }
            adminGroup.removeMembers(agentId);
            proxy.store(adminGroup);
        }
    }

    protected EntitiesList createAgentSkillList(Set<AgentSkill> skills) {
        EntitiesList result = new EntitiesList();

        //Run through the skills only once to transform to DTO and get dependencies
        for (AgentSkill skill : skills) {
            
            log.debug("add agent skill : " + skill);
            
            //transform skill to DTO
            result.addEntity(DTOHelper.toDto(skill));

            //Agent
            Agent agent = proxy.restore(Agent.class,
                    skill.getAgent());
            if (agent != null) {
                result.addDependency(agent.getWikittyId(),
                        DTOHelper.toDto(agent));
            }

            //Domain skill
            Skill domainSkill = proxy.restore(Skill.class,
                    skill.getDomainSkill());
            if (domainSkill != null) {
                result.addDependency(domainSkill.getWikittyId(),
                        DTOHelper.toDto(domainSkill));
            }

            //Technical skill
            Skill technicalSkill = proxy.restore(Skill.class,
                    skill.getTechnicalSkill());
            if (technicalSkill != null) {
                if (log.isDebugEnabled()){
                    log.debug("Add technicalSkill " +
                            technicalSkill.getWikittyId() + " - " +
                            technicalSkill.getName());
                }
                result.addDependency(technicalSkill.getWikittyId(),
                        DTOHelper.toDto(technicalSkill));
            }

            //Frequency
            Frequency frequency = proxy.restore(Frequency.class,
                    skill.getFrequency());
            if (frequency != null) {
                result.addDependency(frequency.getWikittyId(),
                        DTOHelper.toDto(frequency));
            }

            //Nature
            Nature nature = proxy.restore(Nature.class,
                    skill.getNature());
            if (nature != null) {
                result.addDependency(nature.getWikittyId(),
                        DTOHelper.toDto(nature));
            }

            //Rights
            WikittyAuthorisation authorisation = proxy.restore(
                    WikittyAuthorisation.class,
                    skill.getWikittyId());
            if (authorisation != null) {
                result.addDependency(authorisation.getWikittyId(),
                        DTOHelper.toDto(authorisation));
            }
        }

        return result;
    }

    protected EntitiesList createAgentList(List<Agent> agents) {
        EntitiesList result = new EntitiesList();

        //Run through the agents only once to transform to DTO and get dependencies
        for (Agent agent : agents) {
            
            System.out.println("agent : " + agent);

            if (agent != null) {

                //transform agent to DTO
                result.addEntity(DTOHelper.toDto(agent));

                //Department
                Department department = proxy.restore(Department.class,
                        agent.getDepartment());
                if (department != null) {
                    result.addDependency(department.getWikittyId(),
                            DTOHelper.toDto(department));
                }

                //Cati
                Cati cati= proxy.restore(Cati.class,
                        agent.getCati());
                if (cati != null) {
                    result.addDependency(cati.getWikittyId(),
                            DTOHelper.toDto(cati));
                }

                //Units
                List<String> unitsIds = new ArrayList<String>();
                if (agent.getUnit() != null) {
                    unitsIds.addAll(agent.getUnit());
                }

                List<Unit> units = proxy.restore(Unit.class, unitsIds);
                for (Unit unit:units) {
                    if (unit != null) {
                        result.addDependency(unit.getWikittyId(),
                                DTOHelper.toDto(unit));
                    }
                }
            }
        }

        return result;
    }

    @Override
    public void importCSV(Reader reader) {

        //import
        WikittyImportExportService ieService =
                ServiceFactory.getImportExportService(proxy);
        ieService.syncImport(WikittyImportExportService.FORMAT.CSV, reader);
    }

    @Override
    public void exportSkillsReferentiel(Writer writer) {
        try {
            //Get the entities to export
            Search search = Search.query()
                    .eq(Element.ELT_EXTENSION, Skill.EXT_SKILL);
            Criteria criteria = search.criteria();

            //export
            WikittyImportExportService ieService =
                    ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(
                    WikittyImportExportService.FORMAT.CSV, criteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    @Override
    public void exportEntitiesReferentiel(Writer writer) {
        try {
            //Get the entities to export
            Search search = Search.query().or()
                    .eq(Element.ELT_EXTENSION, Cati.EXT_CATI)
                    .eq(Element.ELT_EXTENSION, Department.EXT_DEPARTMENT)
                    .eq(Element.ELT_EXTENSION, Unit.EXT_UNIT)
                    .eq(Element.ELT_EXTENSION, UnitDeptLink.EXT_UNITDEPTLINK);
            Criteria criteria = search.criteria();

            //export
            WikittyImportExportService ieService =
                    ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(
                    WikittyImportExportService.FORMAT.CSV, criteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    @Override
    public void exportUserBase(Writer writer) {
        try {
            //Get the entities to export
            Search searchUsers = Search.query().or()
                    .eq(Element.ELT_EXTENSION, WikittyUser.EXT_WIKITTYUSER)
                    .eq(Element.ELT_EXTENSION, AgentSkill.EXT_AGENTSKILL);
            Criteria usersCriteria = searchUsers.criteria();

            //export
            WikittyImportExportService ieService = ServiceFactory.getImportExportService(proxy);
            ieService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, usersCriteria, writer);

        } catch (Exception eee) {
            log.debug("Cannot export : " + eee);
        }
    }

    /**
     * Get the path for a Skill
     * @param id the skill id
     */
    @Override
    public String getPath(String id) {
        Skill skill = proxy.restore(Skill.class, id);

        String parentID = skill.getParent();

        String path = "";

        if (parentID != null && !parentID.isEmpty()) {
            path = getPath(parentID);
        }

        path += "/" + skill.getName();

        return path;
    }

    /**
     * check the validation results and throw exception with appropriate
     * error messages.
     * @param locale the user locale for error messages
     * @param result the validation result
     * @throws InvalidEntityException if the validation gets errors
     */
    protected void checkResults(Locale locale, NuitonValidatorResult result)
            throws InvalidEntityException{
        if (!result.isValid()) {
            String errorMessage = "";
            List<String> fields = result.getFieldsForError();
            for (String field : fields) {
                List<String> messages = result.getErrorMessages(field);
                for (String message : messages) {
                    errorMessage += l_(locale, message) + "\n";
                }
            }
            throw new InvalidEntityException(errorMessage);
        } else {
            if (log.isDebugEnabled()){
                log.debug("Entity valid");
            }
        }
    }

    /**
     * Move a skill in the tree skill
     * @return true if move is ok, false if cannot move (moving inside itself)
     */
    @Override
    public boolean moveSkill(String destination, String node) {
        if (isChildren(node, destination, true)){
            return false;
        } else {
            Skill skill = proxy.restore (Skill.class, node);
            skill.setParent(destination);
            proxy.store(skill);
            return true;
        }
    }

    public boolean sendPassword(Locale locale, String userId) {

        User user = proxy.restore(User.class, userId);

        if (user != null) {

            Properties properties = config.getFlatOptions();

            Session session = Session.getDefaultInstance(properties, null);
            MimeMessage message = new MimeMessage(session);
            try {
                message.addRecipient(
                        Message.RecipientType.TO,
                        new InternetAddress(user.getEmail()));
                
                message.setFrom(new InternetAddress(getAdminEmail()));

                message.setSubject(l_(locale, "refcomp.admin.email.password.subject"));

                //Create message here
                String body = l_(locale, "refcomp.admin.email.password.body",
                        user.getFirstName(),
                        user.getLastName(),
                        user.getLogin(),
                        user.getPassword());

                message.setText(body);

                Transport.send(message);
            } catch (MessagingException ex) {
                log.debug("Error sending email", ex);
                return false;
            }
            log.info(String.format("Password mail sent to user %s %s - %s",
                    user.getFirstName(), user.getLastName(), userId));
            return true;
        } else {
            log.debug("Cannot send email to a non existing user");
            return false;
        }
    }

    @Override
    public Map<String, Skill> getRootSkill(List<String> skills) {
        Map<String, Skill> result = new HashMap<String, Skill>();
        for (String skill:skills) {
            result.put(skill,DTOHelper.toDto(getRootSkill(skill)));
        }

        return result;
    }

    @Override
    public List<Skill> getAllSkills() {
        //Create criteria : everything that got skill extension
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                Skill.EXT_SKILL);

        Criteria criteria = search.criteria();
        criteria.addSortAscending(Skill.FQ_FIELD_WIKITTYTREENODE_NAME);

        //get results
        PagedResult<Skill> skillPagedResult = proxy.findAllByCriteria(Skill.class, criteria);

        List<Skill> skills = new ArrayList<Skill>();

        if (skillPagedResult != null) {
            for (Skill skill : skillPagedResult.getAll()) {
                skills.add(DTOHelper.toDto(skill));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Return " + skills.size() + " skills");
        }
        return skills;
    }

    protected Skill getRootSkill(String skill) {
        Skill childrenNode = proxy.restore(Skill.class, skill);
        String parentId = childrenNode.getParent();
        if (parentId == null) {
            return childrenNode;
        } else {
            return getRootSkill(parentId);
        }

    }

    protected boolean isChildren(String parent, String children, boolean recursive) {
        WikittyTreeNode childrenNode = proxy.restore(WikittyTreeNode.class, children);
        String parentId = childrenNode.getParent();

        if (parentId == null) {
            return false;
        }

        if (parentId.equals(parent)) {
            return true;
        } else {
            if (!recursive) {
                return false;
            } else {
                return isChildren(parent, parentId, true);
            }
        }
    }
}
