package fr.ifremer.coselmar.converter;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import fr.ifremer.coselmar.beans.DocumentBean;
import fr.ifremer.coselmar.beans.QuestionBean;
import fr.ifremer.coselmar.beans.UserBean;
import fr.ifremer.coselmar.persistence.entity.CoselmarUser;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserGroup;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserImpl;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserRole;
import fr.ifremer.coselmar.persistence.entity.Document;
import fr.ifremer.coselmar.persistence.entity.Question;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.persistence.TopiaIdFactory;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class BeanEntityConverter {

    public static DocumentBean toBean(TopiaIdFactory idFactory, Document document) {
        String lightId = idFactory.getRandomPart(document.getTopiaId());
        Date depositeDate = document.getDepositDate();
        Date publicationDate = document.getPublicationDate();
        CoselmarUser documentOwner = document.getOwner();
        String owner = "N/A";
        String ownerId = "N/A";
        if (documentOwner != null) {
            String firstname = documentOwner.getFirstname();
            String lastname = documentOwner.getName();
            owner = StringUtils.defaultString(firstname) + StringUtils.defaultString(lastname);

            ownerId = idFactory.getRandomPart(documentOwner.getTopiaId());
        }


        return new DocumentBean(lightId,
            document.getName(),
            owner,
            ownerId,
            document.getPrivacy().name(),
            depositeDate,
            document.getKeywords(),

            document.getType(),
            document.getSummary(),
            document.getLanguage(),
            publicationDate,

            document.getAuthors(),
            document.getLicense(),
            document.getCopyright(),

            document.isWithFile(),
            document.getMimeType(),
            document.getExternalUrl(),
            document.getComment(),
            document.getFileName()
        );
    }

    public static UserBean toBean(String lightId ,CoselmarUser user) {
        return new UserBean(lightId,
            user.getFirstname(),
            user.getName(),
            user.getMail(),
            user.getPhoneNumber(),
            user.getRole().name(),
            user.getQualification(),
            user.getOrganization(),
            user.isActive());
    }

    public static CoselmarUser fromBean(UserBean userBean) {
        CoselmarUser user = new CoselmarUserImpl();

        user.setFirstname(userBean.getFirstName());
        user.setName(userBean.getName());
        user.setMail(userBean.getMail());
        String role = userBean.getRole();
        if (StringUtils.isNotBlank(role)) {
            user.setRole(CoselmarUserRole.valueOf(role));
        }
        user.setQualification(userBean.getQualification());
        user.setOrganization(userBean.getOrganization());
        user.setActive(userBean.isActive());
        user.setPhoneNumber(userBean.getPhoneNumber());

        return user;
    }

    public static QuestionBean toBean(TopiaIdFactory idFactory, Question question) {
        QuestionBean result = new QuestionBean();
        result.setId(idFactory.getRandomPart(question.getTopiaId()));

        result.setTitle(question.getTitle());
        result.setSummary(question.getSummary());
        result.setType(question.getType());
        result.setPrivacy(question.getPrivacy().name());
        result.setStatus(question.getStatus().name());

        Collection<String> theme = question.getTheme();
        if (theme != null && !theme.isEmpty()) {
            result.setThemes(new HashSet(theme));
        }

        Date submissionDate = question.getSubmissionDate();
        if (submissionDate != null){
            result.setSubmissionDate(new Date(submissionDate.getTime()));
        }

        Date deadline = question.getDeadline();
        if (deadline != null){
            result.setDeadline(new Date(deadline.getTime()));
        }

        Date closingDate = question.getClosingDate();
        if (closingDate != null){
            result.setClosingDate(new Date(closingDate.getTime()));
        }

        CoselmarUserGroup participants = question.getParticipants();
        if (participants != null && participants.getMembers() != null) {
            for (CoselmarUser participant : participants.getMembers()) {
                String lightId = idFactory.getRandomPart(participant.getTopiaId());
                UserBean participantBean = toBean(lightId, participant);
                result.addParticipant(participantBean);
            }
        }

        Set<CoselmarUser> supervisors = question.getSupervisors();
        if (supervisors != null && !supervisors.isEmpty()) {
            for (CoselmarUser supervisor : supervisors) {
                String lightId = idFactory.getRandomPart(supervisor.getTopiaId());
                UserBean supervisorBean = toBean(lightId, supervisor);
                result.addSupervisor(supervisorBean);
            }
        }

        Set<CoselmarUser> clients = question.getClients();
        if (clients != null && !clients.isEmpty()) {
            for (CoselmarUser client : clients) {
                String lightId = idFactory.getRandomPart(client.getTopiaId());
                UserBean clientBean = toBean(lightId, client);
                result.addClient(clientBean);
            }
        }

        Set<CoselmarUser> contributors = question.getContributors();
        if (contributors != null && !contributors.isEmpty()) {
            for (CoselmarUser contributor : contributors) {
                String lightId = idFactory.getRandomPart(contributor.getTopiaId());
                UserBean contributorBean = toBean(lightId, contributor);
                result.addContributor(contributorBean);
            }
        }

        Collection<Document> relatedDocuments = question.getRelatedDocuments();
        if (relatedDocuments != null && !relatedDocuments.isEmpty()) {
            for (Document relatedDocument : relatedDocuments) {
                DocumentBean documentBean = toBean(idFactory, relatedDocument);
                result.addRelatedDocument(documentBean);
            }
        }

        Collection<Question> parents = question.getParents();
        if (parents != null && !parents.isEmpty()) {
            for (Question parent : parents) {
                QuestionBean questionBean = toLightBean(idFactory, parent);
                result.addParent(questionBean);
            }
        }

        result.setConclusion(question.getConclusion());
        Collection<Document> closingDocuments = question.getClosingDocuments();
        if (closingDocuments != null && !closingDocuments.isEmpty()) {
            for (Document relatedDocument : closingDocuments) {
                String lightId = idFactory.getRandomPart(relatedDocument.getTopiaId());
                DocumentBean documentBean = toBean(idFactory, relatedDocument);
                result.addClosingDocument(documentBean);
            }
        }

        result.setRestricted(question.isUnavailable());
        return result;
    }

    public static QuestionBean toLightBean(TopiaIdFactory idFactory, Question question) {
        QuestionBean result = new QuestionBean();
        result.setId(idFactory.getRandomPart(question.getTopiaId()));

        result.setTitle(question.getTitle());
        result.setSummary(question.getSummary());
        result.setType(question.getType());
        result.setPrivacy(question.getPrivacy().name());
        result.setStatus(question.getStatus().name());

        Collection<String> theme = question.getTheme();
        if (theme != null && !theme.isEmpty()) {
            result.setThemes(new HashSet(theme));
        }

        Date submissionDate = question.getSubmissionDate();
        if (submissionDate != null){
            result.setSubmissionDate(new Date(submissionDate.getTime()));
        }

        Date deadline = question.getDeadline();
        if (deadline != null){
            result.setDeadline(new Date(deadline.getTime()));
        }

        Date closingDate = question.getClosingDate();
        if (closingDate != null){
            result.setClosingDate(new Date(closingDate.getTime()));
            result.setConclusion(question.getConclusion());
        }

        Collection<Document> closingDocuments = question.getClosingDocuments();
        if (closingDocuments != null && !closingDocuments.isEmpty()) {
            for (Document relatedDocument : closingDocuments) {
                String lightId = idFactory.getRandomPart(relatedDocument.getTopiaId());
                DocumentBean documentBean = toBean(idFactory, relatedDocument);
                result.addClosingDocument(documentBean);
            }
        }

        Collection<Question> parents = question.getParents();
        if (parents != null && !parents.isEmpty()) {
            for (Question parent : parents) {
                QuestionBean questionBean = toLightBean(idFactory, parent);
                result.addParent(questionBean);
            }
        }

        result.setRestricted(question.isUnavailable());
        return result;
    }

}
