/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.offer;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.ui.VradiMainUI;
import com.jurismarches.vradi.ui.VradiMainUIHandler;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.email.QueryMakerViewUI;
import com.jurismarches.vradi.ui.helpers.VradiComparators;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdesktop.swingx.event.TableColumnModelExtListener;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.wikitty.BusinessEntityWikitty;
import org.nuiton.wikitty.WikittyException;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.ModificationTagImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.search.SearchHandler;
import com.jurismarches.vradi.ui.search.SearchUI;

/**
 * @author letellier
 */
public class OfferListHandler {

    static private final Log log = LogFactory.getLog(OfferListHandler.class);

    static final public int TABPANEL_LABEL_MAX_CHAR_NB = 30;

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param rootHandler
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferListUI initUI(JAXXContext rootContext, VradiMainUIHandler rootHandler) {

        final JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this);

        final SearchHandler searchHandler = UIHelper.getHandler(context, SearchHandler.class);

        // show main ui
        final OfferListTableModel offerListTableModel = new OfferListTableModel();
        final OfferListColumnFactory offerListColumnFactory = new OfferListColumnFactory();
        
        context.add(offerListTableModel);
        context.add(offerListColumnFactory);

        final OfferListUI ui = new OfferListUI(context);
        
        // Saved in context
        VradiContext.OFFERT_LIST_UI_ENTRY_DEF.setContextValue(VradiContext.get(), ui);
        
        // add listeners
        final JXTable listTable = ui.getListTable();
        listTable.getColumnModel().addColumnModelListener(
                getOfferListTableColumnModelListener());
        
        final MouseListener offerListTableMouseListener = getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);
        
        final TableModelListener offerListTableModelListener = getOfferListTableModelListener(listTable);
        offerListTableModel.addTableModelListener(offerListTableModelListener);
        
        // listTable cellRenderer
        final OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();
        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);
        
        // listTable highlighter
        final Highlighter highlighter = HighlighterFactory.createAlternateStriping(
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        listTable.addHighlighter(highlighter);
    
        // Show search ui
        final SearchUI searchUI = searchHandler.initUI(rootContext, offerListTableModel);
        searchUI.setOfferListUI(true);

        ui.getSearchPanel().add(searchUI, BorderLayout.CENTER);

        return ui;
    }

    protected TableModelListener getOfferListTableModelListener(final JXTable listTable) {
        return new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                listTable.packAll();
            }
        };
    }

    protected TableColumnModelExtListener getOfferListTableColumnModelListener() {
        return new TableColumnModelExtListener() {
            @Override
            public void columnPropertyChange(
                    PropertyChangeEvent paramPropertyChangeEvent) {
                String propertyName = paramPropertyChangeEvent.getPropertyName();
                
                if ("visible".equals(propertyName)) {
                    TableColumnExt column = (TableColumnExt) paramPropertyChangeEvent.getSource();
					String fqFieldName = (String) column.getClientProperty("fqFieldName");

                    if (log.isTraceEnabled()) {
                        log.trace("clientprop: " + fqFieldName);
                        log.trace("identifier: " + column.getIdentifier());
                    }
                    
                    Boolean visible = (Boolean) paramPropertyChangeEvent.getNewValue();
                    if (visible) {
                        VradiHelper.addVradiListColumns(fqFieldName);
                    } else {
                        VradiHelper.removeVradiListColumns(fqFieldName);
                    }
                }
            }
            
            @Override
            public void columnAdded(TableColumnModelEvent e) {
            }
            @Override
            public void columnRemoved(TableColumnModelEvent e) {
            }
            @Override
            public void columnMoved(TableColumnModelEvent e) {
            }
            @Override
            public void columnMarginChanged(ChangeEvent e) {
            }
            @Override
            public void columnSelectionChanged(ListSelectionEvent e) {
            }
        };
    }

    /**
     * Implementation du click (droit/gauche) sur la table des offres résultats/
     * 
     * Comportement :
     * <ul>
     *  <li>double clic : ouvre l'offre
     *  <li>clic droit : changer le status, ajouter a une session de mail
     * </ul>
     * 
     * @return le mouse listener
     */
    public MouseListener getOfferListTableMouseListener() {
        // double click on listTable shows selected offer
        return new MouseAdapter() {
           
            @Override
            public void mouseClicked(MouseEvent e) {
                JTable listTable = (JTable)e.getSource();
                OfferListUI offerListUI = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(VradiContext.get());
                
                // double clic
                if (e.getButton() == MouseEvent.BUTTON1 &&
                        e.getClickCount() == 2) {
                    try {
                        offerListUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                        selectOffer(offerListUI, listTable);
                    } catch (Exception ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't open offer", ex);
                        }
                        ErrorDialogUI.showError(ex);
                    } finally {
                        offerListUI.setCursor(null);
                    }

                // clic droit
                } else if (e.getButton() == MouseEvent.BUTTON3) {
                    int[] selectedRows = offerListUI.getListTable().getSelectedRows();
                    int rowClicked = offerListUI.getListTable().rowAtPoint(e.getPoint());
                    boolean rowClickedSelected = false;
                    if (selectedRows == null || selectedRows.length == 0) {
                        offerListUI.getListTable().getSelectionModel().setSelectionInterval(rowClicked, rowClicked);
                        rowClickedSelected = true;
                    } else {
                        int i = 0;
                        while(i < selectedRows.length && !rowClickedSelected) {
                            rowClickedSelected = rowClicked == selectedRows[i++];
                        }
                    }
                    
                    if (rowClickedSelected) {
                        showOfferListContextMenu(offerListUI, e);
                    }
                }
            }
        };
    }

    /**
     * Le menu contextuel lors d'un clic droit sur la table.
     * 
     * @param offerListUI l'ui d'origine
     * @param e event du clic
     */
    protected void showOfferListContextMenu(final OfferListUI offerListUI, final MouseEvent e) {
        JPopupMenu menu = new JPopupMenu();
        JMenu changeStatus = new JMenu(_("vradi.offer.status.change"));

        List<Status> statuses = AdminHandler.getAllStatuses();
        for (Status status : statuses) {
            JMenuItem statusItem = new JMenuItem(_(status.getName()));
            statusItem.addActionListener(new StatusActionListener(offerListUI, status));
            changeStatus.add(statusItem);
        }

        menu.add(changeStatus);
        menu.add(new JSeparator());

        JMenu addSending = new JMenu(_("vradi.offer.email.add"));

        // Menu to add user or group sending
        JMenuItem addUserSendingItem = new JMenuItem(_("vradi.email.addUser"));
        addUserSendingItem.addActionListener(new UserAddSendingListener(offerListUI));

        JMenuItem addgroupSendingItem = new JMenuItem(_("vradi.email.addGroup"));
        addgroupSendingItem.addActionListener(new GroupAddSendingListener(offerListUI));

        addSending.add(addUserSendingItem);
        addSending.add(addgroupSendingItem);

        menu.add(addSending);
        menu.show(e.getComponent(), e.getX(), e.getY());
    }

    /**
     * Il est possible de faire un clic droit dans la liste des résutats
     * pour ajouter un appel d'offre directement dans la derniere session
     * ouverte.
     */
    protected abstract class AddSendingListener implements ActionListener {
        protected OfferListUI offerListUI;
        protected List<Form> formsToUpdate = new ArrayList<Form>();

        protected AddSendingListener(OfferListUI offerListUI) {
            this.offerListUI = offerListUI;

            OfferListUI localOfferListUI = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(offerListUI);
            int[] selectedRows = localOfferListUI.getListTable().getSelectedRows();

            for(int row : selectedRows) {
                JXTable listTable = localOfferListUI.getListTable();
                OfferListTableModel tableModel = localOfferListUI.getOfferListTableModel();
                row = listTable.convertRowIndexToModel(row);

                FormPagedResult formPage = tableModel.getFormPagedResult();
                List<Form> formsToShow = formPage.getFormsToShow();

                Form form = formsToShow.get(row);
                if (form != null) {
                    formsToUpdate.add(form);
                }
            }
        }

        protected AdminHandler getAdminHandler() {
            return UIHelper.getHandler(offerListUI, AdminHandler.class);
        }

        protected <E extends QueryMaker> void openUI(Class<E> clazz, String title, List<E> all) {

            try {

                Session currentSession = VradiService.getVradiStorageService().getLastOpenSession();

                if (currentSession == null) {
                    JOptionPane.showMessageDialog(offerListUI,
                            _("Can't found any opened mail session, please create a new session in mail management"),
                            _("Error"), JOptionPane.ERROR_MESSAGE);
                }
                else {
                    for (Form form : formsToUpdate) {
        
                        Status selectionne = VradiHelper.getSelectionneStatus();
                        if (!form.getStatus().equals(selectionne.getWikittyId())) {

                            int responce = JOptionPane.showConfirmDialog(
                                    getUI(offerListUI),
                                    _("vradi.email.confirmSelectionForm", form.getObjet()),
                                    _("vradi.email.confirmSelectionForm.title"),
                                    JOptionPane.YES_NO_OPTION);

                            if (responce == JOptionPane.YES_OPTION) {
                                changeStatus(getUI(offerListUI), form, selectionne);
                            } else {
                                return;
                            }
                        }
                    }

                    VradiMainUI mainUI = VradiContext.MAIN_UI_ENTRY_DEF.getContextValue(offerListUI);
                    final JDialog ui = new JDialog(mainUI, _(title));
                    ui.setLayout(new BorderLayout());

                    QueryMakerViewUI<E> makerViewUI = new QueryMakerViewUI<E>(offerListUI);
                    makerViewUI.setSession(currentSession);
                    makerViewUI.setBeanList(formsToUpdate);
                    makerViewUI.setClazz(clazz);

                    makerViewUI.init(all, new ArrayList<E>());

                    ui.add(makerViewUI, BorderLayout.CENTER);
                    JButton closeButton = new JButton(_("vradi.action.close"));
                    closeButton.addActionListener(new ActionListener() {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            ui.setVisible(false);
                            ui.dispose();
                        }
                    });
                    ui.add(closeButton, BorderLayout.SOUTH);
                    ui.setModal(true);
                    ui.setSize(800, 600);
                    ui.setLocationRelativeTo(mainUI);
                    ui.setVisible(true);
                }
            }
            catch (VradiException ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't get current opened session", ex);
                }
            }
        }
    }

    protected class GroupAddSendingListener extends AddSendingListener {

        public GroupAddSendingListener(OfferListUI offerListUI) {
            super(offerListUI);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            openUI(Group.class, "vradi.email.addGroup", getAdminHandler().getAllGroups());
        }
    }

    protected class UserAddSendingListener extends AddSendingListener {

        public UserAddSendingListener(OfferListUI offerListUI) {
            super(offerListUI);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            openUI(User.class, "vradi.email.addUser", getAdminHandler().getAllUsers());
        }
    }

    protected class StatusActionListener implements ActionListener {
        protected Status status;
        protected OfferListUI offerListUI;
        
        public StatusActionListener(OfferListUI offerListUI, Status status) {
            this.offerListUI = offerListUI;
            this.status = status;
        }
        
        @Override
        public void actionPerformed(ActionEvent actionEvent) {
            int confirm = JOptionPane.showConfirmDialog(offerListUI,
                    _("vradi.offerList.status.change.confirmation.message", _(status.getName())),
                    _("vradi.offerList.status.change.confirmation.title"),
                    JOptionPane.YES_NO_OPTION);
            
            if (confirm == JOptionPane.YES_OPTION) {
                offerListUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                try {
                    int[] selectedRows = offerListUI.getListTable().getSelectedRows();
            
                    for(int row : selectedRows) {
                        JXTable listTable = offerListUI.getListTable();
                        OfferListTableModel tableModel = offerListUI.getOfferListTableModel();
                        row = listTable.convertRowIndexToModel(row);
        
                        FormPagedResult formPageDTO = tableModel.getFormPagedResult();
                        List<Form> formsToShow = formPageDTO.getFormsToShow();
        
                        Form form = formsToShow.get(row);
                        if (form != null) {
                            changeStatus(offerListUI, form, status);
                        }
                    }

                    SearchHandler handler = offerListUI.getSearchHandler();
                    handler.executeQuery(offerListUI);
                    
                } catch (WikittyException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't change offer status", eee);
                    }
                    ErrorDialogUI.showError(eee);
                } finally {
                    offerListUI.setCursor(null);
                }
            }
        }
    }
    
    protected void changeStatus(JAXXContext context, Form form, Status status) {

        try {
            form = VradiService.getWikittyProxy().restore(Form.class, form.getWikittyId());
            String initialStatus = form.getStatus();
            String updateStatus = status.getWikittyId();

            if (initialStatus == null || !updateStatus.equals(initialStatus)) {
                if (!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATIONTAG)) {
                    BusinessEntityWikitty entityWikitty = (BusinessEntityWikitty)form;
                    entityWikitty.getWikitty().addExtension(ModificationTagImpl.extensionModificationTag);
                }

                VradiUser vradiUser = context.getContextValue(VradiUser.class);
                String modifier = (vradiUser != null ? vradiUser.getLogin() : null);

                if (log.isInfoEnabled()) {
                    log.info(String.format("Updating: statusModifier=%s on form: %s", modifier, form.getWikittyId()));
                    log.info(String.format("New Status: %s, Old Status: %s", initialStatus, updateStatus));
                }

                // FIXME EC20100910 remove direct access to wikitty
                form.setField(ModificationTag.EXT_MODIFICATIONTAG, ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIER,
                        modifier);
                form.setField(ModificationTag.EXT_MODIFICATIONTAG, ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIED,
                        new Date());
            }

            form.setStatus(updateStatus);
            VradiService.getWikittyProxy().store(form);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't change status", eee);
            }
        }
    }
    
    public void addEditPane(JAXXContext context, String formId) {
        final Form data = VradiService.getWikittyProxy().restore(Form.class, formId);
        addEditPane(context, data);
    }

    public void addEditPane(JAXXContext context, final Form form) {
        final OfferListUI ui = getUI(context);

        final String formId = form.getWikittyId();
        
        // Fill free space
        ui.setDividerLocation(0);
        ui.setResizeWeight(0);
		ui.setDividerSize(12);
		
        // Get tab
        JTabbedPane contentTabbedPane = ui.getEditTabs();

        // Exist ?
        if (contentTabbedPane.indexOfTab(formId) < 0) {

            OfferEditUI editUi;

            // Open new
            final OfferEditHandler handler = UIHelper.getHandler(context, OfferEditHandler.class);
            try {
                editUi = handler.initUI(context, form);
            } catch (Exception eee) {

                // Do nothing on fail
                log.error(eee);
                ErrorDialogUI.showError(eee);
                return;
            }

            // Do nothing on fail
            if (editUi == null) {
                return;
            }

            final OfferEditUI editUiFinal = editUi;
            contentTabbedPane.add(formId, editUiFinal);

            final VradiTabHeader panel = new VradiTabHeader();

            if (form.getObjet() != null) {
                String panelLabel = form.getObjet().length() < TABPANEL_LABEL_MAX_CHAR_NB ?
                        form.getObjet() : form.getObjet().substring(0, TABPANEL_LABEL_MAX_CHAR_NB) + "...";
                panel.getLabel().setText(panelLabel);
            }
            panel.setId(formId);

            // Table closeable
            contentTabbedPane.setTabComponentAt(contentTabbedPane.indexOfTab(formId), panel);
            panel.getCloseTab().addMouseListener(new javax.swing.event.MouseInputAdapter() {

                @Override
                public void mouseClicked(MouseEvent e) {
                    closePane(editUiFinal, panel.getId());
                }
            });

            // Detachable
            panel.getDetachTab().addMouseListener(new javax.swing.event.MouseInputAdapter() {

                @Override
                public void mouseClicked(MouseEvent e) {
                    detachPane(e);
                }
            });

            // Get all status
            List<Status> statuses = AdminHandler.getAllStatuses();

            // New modifiable list
            List<Status> orderedStatus = new ArrayList<Status>(statuses);

            // Sort by value
            Collections.sort(orderedStatus, VradiComparators.STATUSES_COMPARATOR);

            // Add all buttons
            JPanel buttonPanel = editUiFinal.getStatusButtonPanel();
            for (final Status status : orderedStatus) {
                JButton statusButton = new JButton(_(status.getName()));
                statusButton.setToolTipText(status.getDescription());
                if(status.getWikittyId().equals(form.getStatus())) {
                    Color newColor = new Color(110,150,200,
                            statusButton.getBackground().getAlpha());
                    statusButton.setBackground(newColor);
                }
                statusButton.addActionListener(new ActionListener(){

                    @Override
                    public void actionPerformed(ActionEvent actionEvent) {
                        handler.changeStatusAndSave(editUiFinal, form, status);
                        closePane(editUiFinal, panel.getId());

                        SearchHandler handler = UIHelper.getHandler(editUiFinal, SearchHandler.class);
                        handler.executeQuery(ui);
                    }
                });
                buttonPanel.add(statusButton);
            }
            // Cancel
            editUiFinal.getAnnuler().addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    closePane(editUiFinal, panel.getId());
                }
            });

            // Update
            /*editUiFinal.getValidate().addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent actionEvent) {
                    handler.save(editUiFinal, data);
                    closePane(editUiFinal, panel.getId());

                    SearchHandler handler = UIHelper.getHandler(editUiFinal, SearchHandler.class);
                    handler.executeQueries(ui);
                }
            });*/

        }
        
        // Select
        contentTabbedPane.setSelectedIndex(contentTabbedPane.indexOfTab(formId));
    }

    protected VradiTabHeader getTabHeader(MouseEvent e) {
        JButton button = (JButton) e.getSource();
        return (VradiTabHeader) button.getParent();
    }

    protected void closePane(JAXXContext context, String id) {
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(context);
        JTabbedPane contentTabbedPane = getContentPane(ui);
        int index = contentTabbedPane.indexOfTab(id);
        if (index > -1) {
            contentTabbedPane.remove(index);
        }
        allEditPanelRemoved(contentTabbedPane, ui);
    }

    // Detach editUI
    protected void detachPane(MouseEvent e) {
        // Get panel
        VradiTabHeader mytab = getTabHeader(e);
        String id = mytab.getId();
        String name = mytab.getLabel().getText();

        OfferListUI ui = mytab.getParentContainer(OfferListUI.class);
        JTabbedPane contentTabbedPane = getContentPane(ui);
        final OfferEditUI content = getOfferEdit(contentTabbedPane, id);

        // Create popups
        final JFrame frame = new JFrame(name);

        // Close
        /*content.getValidate().addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeDetachedPane(frame, content);
            }
        });*/

        // Cancel
        content.getAnnuler().addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeDetachedPane(frame, content);
            }
        });

        // Remove default listener
        frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

        // Add listener
        frame.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                closeDetachedPane(frame, content);
            }
        });

        frame.setLayout(new BorderLayout());
        frame.add(content, BorderLayout.CENTER);
        frame.setVisible(true);
        frame.setSize(new Dimension(800, 800));

        // Remove
        allEditPanelRemoved(contentTabbedPane, ui);
    }

    protected OfferEditUI getOfferEdit(JTabbedPane contentTabbedPane, String id) {
        return (OfferEditUI) contentTabbedPane.getComponentAt(contentTabbedPane.indexOfTab(id));
    }

    protected void closeDetachedPane(JFrame f, OfferEditUI ui) {
        f.dispose();
    }

    public void addEditPane(JAXXContext context, List<Form> datas) {
        for (Form data : datas) {
            addEditPane(context, data.getWikittyId());
        }
    }

    protected void allEditPanelRemoved(JTabbedPane contentTabbedPane, OfferListUI ui) {
        if (contentTabbedPane.getTabCount() == 0) {
            ui.setDividerLocation(Integer.MAX_VALUE);
            ui.setResizeWeight(1);
            ui.setDividerSize(0);
        }
    }

    public OfferListUI getUI(JAXXContext context) {
        if (context instanceof OfferListUI) {
            return (OfferListUI) context;
        }
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    protected JTabbedPane getContentPane(JAXXContext context) {
        OfferListUI ui = getUI(context);
        return ui.getEditTabs();
    }

    protected JTabbedPane getContentPane(OfferListUI ui) {
        return ui.getEditTabs();
    }

    protected void selectOffer(JAXXContext context, JTable listTable) {
        OfferListTableModel tableModel = (OfferListTableModel)listTable.getModel();
        int selectedRow = listTable.getSelectedRow();
        
        if (selectedRow >= 0 && selectedRow < tableModel.getRowCount()) {
            selectedRow = listTable.convertRowIndexToModel(selectedRow);
         
            FormPagedResult formPageDTO = tableModel.getFormPagedResult();
            List<Form> formsToShow = formPageDTO.getFormsToShow();
            
            Form data = formsToShow.get(selectedRow);
            if (data != null) {
                log.info("Show Form: " + data.getObjet());
                addEditPane(context, data.getWikittyId());
            }
        }
    }

}
