/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.email.helpers;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.tree.TreeModel;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.email.loadors.EmailFormNodeLoadors;
import com.jurismarches.vradi.ui.email.loadors.EmailGroupNodeLoadors;
import com.jurismarches.vradi.ui.email.loadors.EmailUsersNodeLoadors;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.nav.tree.NavTreeHelper;
import jaxx.runtime.swing.nav.tree.NavTreeNodeChildLoador;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.ui.email.EmailHandler;
import org.nuiton.wikitty.*;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

/**
 * @author letellier
 */
public class EmailNavigationTreeHelper extends NavTreeHelper<VradiTreeNode> implements WikittyServiceListener {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(EmailNavigationTreeHelper.class);

    protected List<Session> sessions;

    // All filter
    public enum EmailFilterEnum {
        // FILTER_BY_CLIENT(_("vradi.email.clientFilter")),
        FILTER_BY_USER(_("vradi.email.userFilter")),
        FILTER_BY_GROUP(_("vradi.email.groupFilter")),
        FILTER_BY_FORM(_("vradi.email.formFilter"));

        protected int value;
        protected String text;

        EmailFilterEnum(String text) {
            this.text = text;
        }

        @Override
        public String toString(){
            return text;
        }
    }

    public static String PATH_SEPARATOR = "/";

    // Filter by client by defaut
    protected EmailFilterEnum filter = EmailFilterEnum.FILTER_BY_USER;

    public EmailNavigationTreeHelper(VradiDataProvider dataProvider) {
        super();
        setDataProvider(dataProvider);
        
        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL);
    }

    @Override
    public VradiDataProvider getDataProvider() {
        return (VradiDataProvider)dataProvider;
    }

    public WikittyProxy getProxy() {
        return getDataProvider().getWikittyProxy();
    }

    // Get email handler
    protected EmailHandler getHandler(JAXXContext context) {
        return UIHelper.getHandler(context, EmailHandler.class);
    }

    public String getPathSeparator() {
        return PATH_SEPARATOR;
    }

    public boolean isByForm() {
        return filter.equals(EmailFilterEnum.FILTER_BY_FORM);
    }

    public boolean isByUser() {
        return filter.equals(EmailFilterEnum.FILTER_BY_USER);
    }

    /* public boolean isByClient() {
        return filter.equals(EmailFilterEnum.FILTER_BY_CLIENT);
    }*/

    public boolean isByGroup() {
        return filter.equals(EmailFilterEnum.FILTER_BY_GROUP);
    }

    /**
     * Return selected session active session.
     * 
     * @return session entity
     */
    public Session getSelectedActiveSession() {
        
        Session activeSession = null;
        VradiTreeNode selectedNode = getSelectedNode();

        if (selectedNode != null) {
            Session session = getParentSession(selectedNode);
    
            if (session != null && session.getStatus() != VradiConstants.SessionStatus.CANCELED.getValue() &&
                    session.getStatus() != VradiConstants.SessionStatus.SENT.getValue()) {
                activeSession = session;
            }
        }

        return activeSession;
    }

    public void updateTree(List<Session> sessions, EmailFilterEnum filter) {
        if (log.isDebugEnabled()) {
            log.debug("update data, nb sessions : " + sessions.size());
        }
        this.filter = filter;
        this.sessions = sessions;

        VradiTreeNode rootNode = new VradiTreeNode(
                String.class,
                "Root node",
                null,
                null
        );

        for (Session session : sessions) {
            rootNode.add(createSessionNode(session));
        }
        TreeModel model = createModel(rootNode);

        // Populate childs nodes
        rootNode.populateChilds(getBridge(), getDataProvider());
                                  
        getUI().setModel(model);
    }

    public VradiTreeNode createSessionNode(Session session) {
        if (log.isDebugEnabled()){
            log.debug("createSession node " + session.getSessionDate() + " num " + session.getNum());
        }
        String sessionId = session.getWikittyId();

        NavTreeNodeChildLoador<?, ?, VradiTreeNode> nodeLoador = null;

        if (isByForm()){
            nodeLoador = new EmailFormNodeLoadors();
        } else {
            if (isByUser()){
                nodeLoador = new EmailUsersNodeLoadors();
            /* } else if (isByClient()){
                nodeLoador = new EmailClientNodeLoadors();*/
            } else if (isByGroup()){
                nodeLoador = new EmailGroupNodeLoadors();
            }
        }

        return new VradiTreeNode(
                Session.class,
                sessionId,
                null,
                nodeLoador);

    }

    public List<Sending> getSendingToDisplay(VradiTreeNode node){
        Map<String, Sending> result = new HashMap<String, Sending>();
        Class type = node.getInternalClass();
        String id = node.getId();

        Session session = getParentSession(node);

        if (type.equals(Form.class)){

            // Get sending by form
            List<Sending> sendings = getSendingByForm(session, id);
            addSendingToMap(result, sendings);

        } else if (type.equals(Session.class)){

            // If no sending found return empty list
            Set<String> sendingIdsSet = session.getSending();
            if (sendingIdsSet == null) {
                return new ArrayList<Sending>();
            }
            
            List<String> sendingIds = new ArrayList<String>(sendingIdsSet);
            List<Sending> sendings = getProxy().restore(Sending.class, sendingIds);
            addSendingToMap(result, sendings);
            

        } else if (QueryMaker.class.isAssignableFrom(type)){
            Sending sending = EmailDataHelper.getSending(session, id);
            result.put(sending.getWikittyId(), sending);
        }
        
        return new ArrayList<Sending>(result.values());
    }

    protected void addSendingToMap(Map<String, Sending> result, List<Sending> sendings) {
        for (Sending sending : sendings){
            result.put(sending.getWikittyId(), sending);
        }
    }

    protected List<Sending> getSendingByForm(Session session, String formId) {
        Set<String> sendingIdsSet = session.getSending();
        List<Sending> result = new ArrayList<Sending>();
        if (sendingIdsSet == null) {
            return result;
        }
        List<Sending> sendings = getProxy().restore(Sending.class, new ArrayList<String>(sendingIdsSet));
        for (Sending sending : sendings) {
            Set<String> formIdsSet = sending.getForm();
            if (formIdsSet == null) {
                return result;
            }
            if (formIdsSet.contains(formId)) {
                result.add(sending);
            }
        }
        return result;
    }

    protected List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }

    public static Session getParentSession(VradiTreeNode node) {
        VradiTreeNode sessionNode = getParentSessionNode(node);
        if (sessionNode == null) {
            return null;
        }
        return VradiService.getWikittyProxy().restore(Session.class, sessionNode.getId());
    }

    public static VradiTreeNode getParentSessionNode(VradiTreeNode node) {
        if (node == null) {
            return null;
        }
        if (node.getInternalClass().equals(Session.class)) {
            return node;
        }
        VradiTreeNode parent = node.getParent();
        return getParentSessionNode(parent);
    }

    protected void refresh(VradiTreeNode parentNode) {

        if (parentNode == null) {
            return;
        }

        List<VradiTreeNode> selectedNodes = getSelectedNodes();
        parentNode.removeAllChildren();
        parentNode.populateChilds(getBridge(), getDataProvider());

        getBridge().nodeStructureChanged(parentNode);
        selectNodes(selectedNodes);
    }

    @SuppressWarnings({"ConstantConditions"})
    @Override
    public void putWikitty(WikittyServiceEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("[Email] put wikitty");
        }

        // map between id and extensions "name" (not extension ids)
        Map<String, Set<String>> idAndExtensions = event.getIdExtensions();
        for (String wikittyId : event.getIds()) {
            Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);

            // Keep selected
            VradiTreeNode selectedNode = getSelectedNode();

            // Session
            if (wikittyExtensions.contains(Session.EXT_SESSION)) {
                if (log.isDebugEnabled()) {
                    log.debug("[Session] put wikitty");
                }

                VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);

                // cas modification
                if (existingNode != null) {

                    if (log.isDebugEnabled()) {
                        log.debug("[Session] Modification case :  " + existingNode.getId());
                    }

                    refresh(existingNode);

                    return;
                }

                // Creation case
                else {

                    Session session = getProxy().restore(Session.class, wikittyId);

                    if (session == null) {
                        return;
                    }

                    if (log.isDebugEnabled()) {
                        log.debug("[Session] Creation case :  " + session.getWikittyId());
                    }

                    // Tri
                    getRootNode().add(createSessionNode(session));
                }
            }

            // Sending
            if (wikittyExtensions.contains(Sending.EXT_SENDING)) {
                if (log.isDebugEnabled()) {
                    log.debug("[Sending] put wikitty");
                }

                VradiTreeNode existingNode = findNode(getRootNode(), wikittyId);

                if (existingNode != null) {
                    VradiTreeNode toRefresh = getParentSessionNode(existingNode);
                    if (toRefresh != null) {
                        if (log.isDebugEnabled()) {
                            log.debug("[Sending] Modification case :  " + toRefresh.getId());
                        }

                        refresh(toRefresh);
                    }
                    return;
                }

                Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION)
                        .contains(Session.FQ_FIELD_SESSION_SENDING, wikittyId).criteria();

                Session session = getProxy().findByCriteria(Session.class, criteria);
                if (session == null) {
                    return;
                }
                VradiTreeNode toRefresh = findNode(getRootNode(), session.getWikittyId());
                if (toRefresh != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("[Sending] Creation case :  " + toRefresh.getId());
                    }
                    refresh(toRefresh);
                }
            }

            // Reselect
            if (selectedNode != null){
                UIHelper.selectNodeLater(this, selectedNode.getId());
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        // should not happen in vradi
    }

    @Override
    public void putExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service put extension event : " + event);
        }
        // TODO ???
    }

    @Override
    public void removeExtension(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove extension event : " + event);
        }
    }

    @Override
    public void clearExtension(WikittyServiceEvent event) {
        // should not happen in vradi
    }
}
