/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.admin;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import java.awt.Component;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;

import javax.swing.JPanel;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.decorator.Decorator;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.NavigationTreeHandler;
import jaxx.runtime.swing.navigation.NavigationTreeHandlerWithCardLayout;
import jaxx.runtime.swing.navigation.NavigationTreeHelper;
import jaxx.runtime.swing.navigation.NavigationTreeModel;
import jaxx.runtime.swing.navigation.NavigationTreeModelBuilder;
import jaxx.runtime.swing.navigation.NavigationTreeNode;
import jaxx.runtime.swing.navigation.NavigationTreeHandler.Strategy;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.ClientImpl;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.GroupImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.UserImpl;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.entities.XmlStreamImpl;
import com.jurismarches.vradi.ui.admin.content.AdminClientUI;
import com.jurismarches.vradi.ui.admin.content.AdminFormTypeUI;
import com.jurismarches.vradi.ui.admin.content.AdminGroupUI;
import com.jurismarches.vradi.ui.admin.content.AdminStatusUI;
import com.jurismarches.vradi.ui.admin.content.AdminUserUI;
import com.jurismarches.vradi.ui.admin.content.AdminXmlStreamUI;

/**
 * Navigation tree helper .
 *
 * @author sletellier
 * @see NavigationTreeHelper
 */
public abstract class VradiNavigationTreeHelper extends NavigationTreeHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(VradiNavigationTreeHelper.class);
    /**
     * where the clients are hold in context
     */
    public static final JAXXContextEntryDef<List<Client>> CLIENTS = SwingUtil.newListContextEntryDef(n_("vradi.adminClient.categoryClient"));
    /**
     * where the users are hold in context
     */
    public static final JAXXContextEntryDef<List<User>> USERS = SwingUtil.newListContextEntryDef(n_("vradi.adminUser.categoryUser"));
    /**
     * where the groups are hold in context
     */
    public static final JAXXContextEntryDef<List<Group>> GROUPS = SwingUtil.newListContextEntryDef(n_("vradi.adminGroup.categoryGroup"));
    /**
     * where the statuses are hold in context
     */
    public static final JAXXContextEntryDef<List<Status>> STATUSES = SwingUtil.newListContextEntryDef(n_("vradi.adminStatus.categoryStatus"));
    /**
     * where the forms are hold in context
     */
    public static final JAXXContextEntryDef<List<WikittyExtension>> FORMS = SwingUtil.newListContextEntryDef(n_("vradi.adminForm.categoryForm"));
    /**
     * where the xmlStreams are hold in context
     */
    public static final JAXXContextEntryDef<List<XmlStream>> XMLSTREAMS = SwingUtil.newListContextEntryDef(n_("vradi.adminXmlStream.categoryXmlStream"));
    /**
     * name of a new client
     */
    protected static String NEW_CLIENT_NAME = n_("vradi.adminClient.newClientName");
    /**
     * name of a new group
     */
    protected static String NEW_GROUP_NAME = n_("vradi.adminGroup.newGroupName");
    /**
     * name of a new user
     */
    protected static String NEW_USER_NAME = n_("vradi.adminUser.newUserName");
    /**
     * name of a new status
     */
    protected static String NEW_STATUS_NAME = n_("vradi.adminStatus.newStatusName");
    /**
     * TODO REMOVE ?
     */
    //    protected static String NEW_FORM_NAME = n_("vradi.adminForm.newFormName");
    /**
     * name of a new form
     */
    protected static String NEW_XMLSTREAM_NAME = n_("vradi.adminXmlStream.newStreamName");
    /**
     * tree model builder associated with the helper
     */
    protected final AdminTreeModelBuilder builder;

    /**
     * Create the tree model given model shared in builder context.
     *
     * @return the tree model
     */
    protected abstract NavigationTreeModel createTreeModel();

    public VradiNavigationTreeHelper(String prefix, JAXXContext context) {
        super(prefix);
        builder = new AdminTreeModelBuilder(context);
    }

    public JAXXContext getContext() {
        return builder.getModel().getContext();
    }

    public NavigationTreeNode addClientToSelected(JAXXContext context, Client client) {
        return builder.addClient(context, getSelectedNode(context), client);
    }

    public NavigationTreeNode addFormTypeToSelected(JAXXContext context, WikittyExtension form) {
        return builder.addForm(context, getSelectedNode(context), form);
    }

    public NavigationTreeNode addGroupToSelected(JAXXContext context, Group group) {
        return builder.addGroup(context, getSelectedNode(context), group);
    }

    public NavigationTreeNode addStatusToSelected(JAXXContext context, Status status) {
        return builder.addStatus(context, getSelectedNode(context), status);
    }

    public NavigationTreeNode addUserToSelected(JAXXContext context, User user) {
        return builder.addUser(context, getSelectedNode(context), user);
    }

    public NavigationTreeNode addXmlStreamToSelected(JAXXContext context, XmlStream xml) {
        return builder.addXmlStream(context, getSelectedNode(context), xml);
    }

    public NavigationTreeNode addClient(JAXXContext context, NavigationTreeNode selectedNode, Client client) {
        return builder.addClient(context, selectedNode, client);
    }

    public NavigationTreeNode addStatus(JAXXContext context, NavigationTreeNode selectedNode, Status status) {
        return builder.addStatus(context, selectedNode, status);
    }

    public NavigationTreeNode addFormType(JAXXContext context, NavigationTreeNode selectedNode, WikittyExtension form) {
        return builder.addForm(context, selectedNode, form);
    }

    public NavigationTreeNode addGroup(JAXXContext context, NavigationTreeNode selectedNode, Group group) {
        return builder.addGroup(context, selectedNode, group);
    }

    public NavigationTreeNode addUser(JAXXContext context, NavigationTreeNode selectedNode, User user) {
        return builder.addUser(context, selectedNode, user);
    }

    public NavigationTreeNode addXmlStream(JAXXContext context, NavigationTreeNode selectedNode, XmlStream xml) {
        return builder.addXmlStream(context, selectedNode, xml);
    }

    public NavigationTreeNode getParentClientNode(NavigationTreeNode current) {
        return builder.getParentClientNode(current);
    }

    public NavigationTreeNode getParentGroupNode(NavigationTreeNode current) {
        return builder.getParentGroupNode(current);
    }

    public NavigationTreeNode removeChildNode(NavigationTreeNode node) {
        return builder.removeChildNode(node);
    }

    /**
     * Essaye de recharger un noeud donné par son path complet.
     * <p/>
     * Si le noeud n'existe pas, on reselectionne le premier parent existant.
     *
     * @param context  the context ou trouve le modele
     * @param fullPath the path du noeud a reselectionne
     */
    public void reSelectNode(JAXXContext context, String fullPath) {
        if (log.isDebugEnabled()) {
            log.debug("try to reselect " + fullPath);
        }
        NavigationTreeModel model = getSafeTreeModel(context);
        String[] paths = fullPath.split(model.getPathSeparator());
        NavigationTreeNode node = model.getRoot();
        for (int i = 1, max = paths.length; i < max; i++) {
            String p = paths[i];
            if (log.isDebugEnabled()) {
                log.debug("find child with path " + p);
            }
            NavigationTreeNode node2 = node.getChild(p);
            if (node2 == null) {
                // noeud non trouve, on s'arrete la
                break;
            }
            node = node2;
        }
        if (log.isDebugEnabled()) {
            log.debug("reselect " + node.getFullPath());
        }
        if (node == model.getRoot()) {
            // on ne peut pas reselectionner le root
            if (node.getChildCount() > 0) {
                node = node.getChildAt(0);
            } else {
                node = null;
            }
        }
        selectNode(context, node);
    }

    public void reloadGroup(JAXXContext context, NavigationTreeNode groupNode, Group group) {
        groupNode.removeAllChildren();
        builder.fillGroupNode(group, groupNode);
        //builder.getModel().nodeChanged(groupNode, true);
    }

    @Override
    public final NavigationTreeModel createTreeModel(JAXXContext context) {
        return createTreeModel();
    }

    @Override
    public final NavigationTreeHandler createTreeHandler(JAXXObject context) {

        if (log.isDebugEnabled()) {
            log.debug("create handler");
        }

        NavigationTreeHandler handler;

        handler = new NavigationTreeHandlerWithCardLayout(
                getPrefix(),
                context,
                Strategy.PER_UI_TYPE) {

            private static final long serialVersionUID = 1L;

            @Override
            protected boolean closeUI(Component component) throws Exception {
                if (component != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Closing UI : " + component.getName());
                    }
                    AdminContentUI content = (AdminContentUI) component;

                    return content.getHandler().answerToSave(content);
                }
                return true;
            }

            @Override
            protected NavigationTreeModel getNavigationTreeModel() {
                return getSafeTreeModel(getContext());
            }

            @Override
            protected JPanel getContentContainer() {
                return getContext().getContent();
            }

            @Override
            protected CardLayout2 getContentLayout() {
                return getContext().getContentLayout();
            }

            @Override
            protected void treateError(Exception e) {
                ErrorDialogUI.showError(e);
            }

            @Override
            public AdminNavUI getContext() {
                return (AdminNavUI) this.context;
            }
        };
        // on ne peut selectionner qu'un seul noeud a la fois
        handler.setSelectionMode(NavigationTreeHandler.SINGLE_TREE_SELECTION);

        // save handler in ui context
        setTreeHandler(context, handler);

        // save tree in context
        setTree(context, ((AdminNavUI) context).getNav());

        return handler;
    }

    /**
     * @author letellier
     */
    protected class AdminTreeModelBuilder extends NavigationTreeModelBuilder {

        public AdminTreeModelBuilder(JAXXContext context) {
            super("/", context, null, null);
        }

        protected Decorator<?> getDecorator(Class<?> clazz) {
            return VradiContext.get().getDecorateurProvinder().getDecorator(clazz);
        }

        public NavigationTreeNode addClient(JAXXContext context, NavigationTreeNode node, Client client) {

            if (log.isDebugEnabled()) {
                log.debug("Selected node : " + node.getFullPath());
            }

            if (client == null) {
                client = new ClientImpl();
                client.setName(_(NEW_CLIENT_NAME));
                log.debug("Client id : " + client.getWikittyId());

                NavigationTreeNode groupNode = getParentGroupNode(node);

                if (groupNode != null) {
                    Group group = (Group) groupNode.getBean();
                    group.addClient(client.getWikittyId());
                }
            }
            addIfNecessary(CLIENTS.getContextValue(context), client);

            return addClient(node, client.getWikittyId());
        }

        public NavigationTreeNode addClient(NavigationTreeNode node, String id) {

            NavigationTreeNode result = build(getClientCategoryNode(node), getDecorator(Client.class), "..[@wikittyId=\"" + id + "\"]", id, AdminClientUI.class, null);

            if (!isGroupNavigationTree(result)) {
                build(
                        result,
                        _(USERS.getName()),
                        USERS,
                        "users",
                        AdminUserUI.class,
                        null);
            }
            return result;
        }

        public NavigationTreeNode addUser(JAXXContext context, NavigationTreeNode node, User user) {
            if (user == null) {
                user = new UserImpl();
                user.setName(_(NEW_USER_NAME));

                NavigationTreeNode groupNode = getParentGroupNode(node);

                if (groupNode == null) {
                    NavigationTreeNode clientNode = getParentClientNode(node);
                    Client client = (Client) clientNode.getBean();

                    client.addUser(user.getWikittyId());
                } else {
                    Group group = (Group) groupNode.getBean();
                    group.addUser(user.getWikittyId());
                }
            }

            addIfNecessary(USERS.getContextValue(context), user);

            return addUser(node, user.getWikittyId());
        }

        public NavigationTreeNode addUser(NavigationTreeNode node, String id) {
            return build(
                    getUserCategoryNode(node),
                    getDecorator(User.class),
                    "..[@wikittyId=\"" + id + "\"]",
                    id,
                    AdminUserUI.class,
                    null);

        }

        public NavigationTreeNode addGroup(JAXXContext context, NavigationTreeNode node, Group group) {
            if (group == null) {
                group = new GroupImpl();
                group.setName(_(NEW_GROUP_NAME));
            }

            addIfNecessary(GROUPS.getContextValue(context), group);

            return addGroup(node, group);
        }

        public NavigationTreeNode addGroup(NavigationTreeNode node, Group group) {

            NavigationTreeNode result = build(
                    getGroupCategoryNode(node),
                    getDecorator(Group.class),
                    "..[@wikittyId=\"" + group.getWikittyId() + "\"]",
                    group.getWikittyId(), AdminGroupUI.class,
                    null);

            fillGroupNode(group, result);

            return result;
        }


        public void fillGroupNode(Group g, NavigationTreeNode groupNode) {
            build(
                    groupNode,
                    _(USERS.getName()),
                    USERS,
                    "users",
                    AdminUserUI.class,
                    null);

            build(
                    groupNode,
                    _(CLIENTS.getName()),
                    CLIENTS,
                    "clients",
                    AdminClientUI.class,
                    null);

            Set<String> usersId = g.getUser();
            if (usersId != null) {
                NavigationTreeNode usersNode = groupNode.getChildAt(0);
                for (String userId : usersId) {
                    log.debug("add user " + userId + " to group " + g.getName());
                    addUser(usersNode, userId);
                }
            }

            Set<String> clientsId = g.getClient();
            if (clientsId != null) {
                NavigationTreeNode clientsNode = groupNode.getChildAt(1);
                for (String clientId : clientsId) {
                    log.debug("add client " + clientId + " to group " + g.getName());
                    addClient(clientsNode, clientId);
                }
            }
        }

        public NavigationTreeNode addStatus(JAXXContext context, NavigationTreeNode selectedNode, Status status) {
            if (status == null) {
                status = new StatusImpl();
                status.setName(_(NEW_STATUS_NAME));
            }
            
            addIfNecessary(STATUSES.getContextValue(context), status);
            return addStatus(selectedNode, status.getWikittyId());
        }

        public NavigationTreeNode addStatus(NavigationTreeNode selectedNode, String id) {
            NavigationTreeNode result = build(
                    getStatusCategoryNode(selectedNode),
                    getDecorator(Status.class),
                    "..[@wikittyId=\"" + id + "\"]",
                    id,
                    AdminStatusUI.class,
                    null);
            return result;
        }

        public NavigationTreeNode addForm(JAXXContext context, NavigationTreeNode selectedNode, WikittyExtension form) {
            addIfNecessary(FORMS.getContextValue(context), form);
            if (form != null) {
                return addForm(selectedNode, form.getId());
            }
            return null;
        }

        public NavigationTreeNode addForm(NavigationTreeNode selectedNode, String id) {
            return build(
                    getFormCategoryNode(selectedNode),
                    getDecorator(WikittyExtension.class),
                    "..[@id=\"" + id + "\"]",
                    id,
                    AdminFormTypeUI.class,
                    null);
        }

        public NavigationTreeNode addXmlStream(JAXXContext context, NavigationTreeNode selectedNode, XmlStream xml) {
            if (xml == null) {
                xml = new XmlStreamImpl();
                xml.setName(_(NEW_XMLSTREAM_NAME));
            }
            addIfNecessary(XMLSTREAMS.getContextValue(context), xml);
            return addXmlStream(selectedNode, xml.getWikittyId());
        }

        public NavigationTreeNode addXmlStream(NavigationTreeNode selectedNode, String id) {
            NavigationTreeNode result = build(
                    getXmlStreamCategoryNode(selectedNode),
                    getDecorator(XmlStream.class),
                    "..[@wikittyId=\"" + id + "\"]",
                    id,
                    AdminXmlStreamUI.class,
                    null);
            return result;
        }

        public NavigationTreeNode getParentClientNode(NavigationTreeNode current) {
            NavigationTreeNode node = getParentNode(current, Client.class);
            return node;
        }

        public NavigationTreeNode getParentGroupNode(NavigationTreeNode current) {
            NavigationTreeNode node = getParentNode(current, Group.class);
            return node;
        }

        protected NavigationTreeNode getCategoryNode(NavigationTreeNode current, String categoryName) {
            NavigationTreeNode result = getParentCategoryNode(current, categoryName);
            if (result == null){
                result = getChildCategoryNode(current, categoryName);
            }

            if (log.isDebugEnabled()) {
                if (result != null){
                    log.debug("Find category [" + categoryName + "] node " + result.getFullPath());
                } else {
                    log.debug("Cant find category [" + categoryName + "] node");
                }
            }
            return result;
        }

        protected NavigationTreeNode getChildCategoryNode(NavigationTreeNode current, String categoryName) {
            if (current.getNodePath().equals(categoryName)) {
                return current;
            } else {
                Enumeration<?> e = current.children();
                while (e.hasMoreElements()) {
                    NavigationTreeNode child = (NavigationTreeNode) e.nextElement();
                    return child == null ? null : getChildCategoryNode(child, categoryName);
                }
            }
            return null;
        }

        protected NavigationTreeNode getParentCategoryNode(NavigationTreeNode current, String categoryName) {
            if (current.getNodePath().equals(categoryName)) {
                return current;
            } else {
                return current.getParent() == null ? null : getCategoryNode(current.getParent(), categoryName);
            }
        }

        protected NavigationTreeNode getClientCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "clients");
        }

        protected NavigationTreeNode getUserCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "users");
        }

        protected NavigationTreeNode getGroupCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "groups");
        }

        protected NavigationTreeNode getStatusCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "statuses");
        }

        protected NavigationTreeNode getFormCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "forms");
        }

        protected NavigationTreeNode getXmlStreamCategoryNode(NavigationTreeNode current) {
            return getCategoryNode(current, "xmlStreams");
        }

        protected boolean isGroupNavigationTree(NavigationTreeNode current) {
            return current.getFullPath().matches("^\\$root\\/groups\\/.*$");
        }

        protected <T> void addIfNecessary(List<T> contextValue, T o) {
            if (!contextValue.contains(o)) {
                contextValue.add(o);
            }
        }
    }
}
