/*
 * #%L
 * Nuiton Web :: Nuiton Struts 2
 * 
 * $Id: TopiaTransactionInterceptor.java 87 2011-07-05 12:01:58Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-web/tags/nuiton-web-1.2/nuiton-struts2/src/main/java/org/nuiton/web/struts2/interceptor/TopiaTransactionInterceptor.java $
 * %%
 * Copyright (C) 2010 - 2011 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.web.struts2.interceptor;

import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;
import com.opensymphony.xwork2.util.TextParseUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.web.struts2.TopiaTransactionAware;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;


/**
 * <!-- START SNIPPET: description -->
 * <p/>
 * The aim of this Interceptor is to inject a freshly opened {@code transaction}
 * in a action which implements {@link TopiaTransactionAware} contract and
 * after result of action, to close the transaction.
 * <p/>
 * The interceptor is abstract and let user to implement the way how to open a
 * new transaction via the method {@link #beginTransaction()}.
 * <p/>
 * Note that the transaction pushed in the action is in fact a proxy of the
 * freshly opened transaction to forbid some method to be call on it. The list
 * of method to forbid can be customized using the interceptor parameter
 * {@link #excludeMethods}.
 * <p/>
 * Note also that the transaction is closed after all stack of interceptor
 * consumed, this means that the transaction will still be opened while
 * rendering the result, this is a particular interesting thing to avoid
 * pre-loading of entities due to lazy strategy of hibernate for example.
 * With this mecanism you can feel free to just obtain the obtain from database
 * via a DAO and then really load it in the rendering result.
 * <p/>
 * <p/>
 * This interceptor, as it provides connection to database should be in the
 * interceptor stack before any other interceptor requiring access to database.
 * For example, it is a common behaviour to do such calls in a prepare method,
 * so make sure to place this interceptor before the {@code prepare} interceptor.
 * <!-- END SNIPPET: description -->
 * <p/>
 * <p/> <u>Interceptor parameters:</u>
 * <p/>
 * <!-- START SNIPPET: parameters -->
 * <p/>
 * <ul>
 * <li>excludeMethods (optional) - Customized method names separated by coma to
 * forbid on the proxy of the transaction given to action. By default, if this
 * parameter is not filled, then we will use this one :
 * {@link #DEFAULT_EXCLUDE_METHODS}.</li>
 * </ul>
 * <p/>
 * <!-- END SNIPPET: parameters -->
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.2
 */
public abstract class TopiaTransactionInterceptor extends AbstractInterceptor {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TopiaTransactionInterceptor.class);

    private static final long serialVersionUID = 1L;

    public static final String[] DEFAULT_EXCLUDE_METHODS = {
            "beginTransaction",
            "closeContext",
            "clear"
    };

    /** names of methods to forbid access while using proxy. */
    protected Set<String> excludeMethods;

    public Set<String> getExcludeMethods() {
        return excludeMethods;
    }

    public void setExcludeMethods(String excludeMethods) {
        this.excludeMethods = TextParseUtil.commaDelimitedStringToSet(excludeMethods);
    }

    /**
     * Method to open a new transaction.
     *
     * @return the new freshly opened transaction
     * @throws TopiaException if any problem while opening a new transaction
     */
    protected abstract TopiaContext beginTransaction() throws TopiaException;

    @Override
    public void init() {
        super.init();

        if (getExcludeMethods() == null) {

            // use default exclude methods
            excludeMethods = new HashSet<String>(
                    Arrays.asList(DEFAULT_EXCLUDE_METHODS)
            );
        }
    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        TopiaTransactionAware transactionAware = null;

        Object action = invocation.getProxy().getAction();

        if (action instanceof TopiaTransactionAware) {
            transactionAware = (TopiaTransactionAware) action;
        }

        TopiaContext transaction = null;
        if (transactionAware != null) {

            // action need a transaction
            transaction = beginTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Open transaction " + transaction);
            }
            // creates a proxy on the transaction to push back in action
            TopiaContext proxy = (TopiaContext) Proxy.newProxyInstance(
                    getClass().getClassLoader(),
                    new Class<?>[]{TopiaContext.class,
                                   TopiaContextImplementor.class},
                    new TopiaTransactionProxyInvocationHandler(transaction)
            );

            // set the transaction in the action
            transactionAware.setTransaction(proxy);
        }
        try {
            return invocation.invoke();
        } finally {

            if (transactionAware != null) {

                // we are on a action with a internal topia transaction

                if (transaction != null && !transaction.isClosed()) {

                    if (log.isDebugEnabled()) {
                        log.debug("Close transaction " + transaction);
                    }
                    // close the opened transaction
                    transaction.closeContext();
                }
            }
        }
    }

    /**
     * Handler of a proxy on a {@link TopiaContext}.
     *
     * @see TopiaTransactionInterceptor#excludeMethods
     */
    class TopiaTransactionProxyInvocationHandler implements InvocationHandler {

        /** Target to use for the proxy. */
        protected final TopiaContext tx;

        TopiaTransactionProxyInvocationHandler(TopiaContext tx) {
            this.tx = tx;
        }

        @Override
        public Object invoke(Object proxy,
                             Method method,
                             Object[] args) throws Throwable {

            String methodName = method.getName();

            if (getExcludeMethods().contains(methodName)) {

                // not authorized
                throw new IllegalAccessException(
                        "Not allowed to access method " + methodName + " on " +
                        proxy);
            }

            // can invoke the method on the tx
            try {
                Object result = method.invoke(tx, args);
                return result;
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Could not execute method " + method.getName(), eee);
                }
                throw eee;
            }
        }
    }

}