package org.nuiton.topia.templates;

/*
 * #%L
 * ToPIA :: Templates
 * $Id: TopiaTagValues.java 3196 2014-06-27 15:18:16Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-7/topia-templates/src/main/java/org/nuiton/topia/templates/TopiaTagValues.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.sql.Blob;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.tagvalue.TagValueAble;
import org.nuiton.eugene.models.tagvalue.TagValueDefinition;
import org.nuiton.eugene.models.tagvalue.TagValueDefinitionProvider;
import org.nuiton.eugene.models.tagvalue.TagValues;
import org.nuiton.eugene.models.tagvalue.matcher.StartsWithTagNameMatcher;
import org.nuiton.topia.persistence.TopiaDaoSupplier;
import org.nuiton.topia.persistence.TopiaEntityContextable;

import com.google.common.collect.Maps;

/**
 * All extra tag values usable in topia generators.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.eugene.models.tagvalue.TagValueDefinitionProvider" role-hint="topia-templates"
 * @since 2.5
 */
public class TopiaTagValues extends TagValueDefinitionProvider {

    /**
     * Static singleton instance used to read defaultValue
     */
    protected static final TopiaTagValues INSTANCE = new TopiaTagValues();
    static {
        // XXX AThimel 18/06/14 I don't like writing such blocks, but I would like to leave INSTANCE final ... Need time to think about it
        INSTANCE.init();
    }

    /**
     * Tag pour le type de persistence.
     *
     * @see TopiaGeneratorUtil#getPersistenceType(ObjectModelClassifier)
     * @see #getPersistenceTypeTagValue(ObjectModelClassifier)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelClassifier.class},
                        documentation = "TODO")
    public static final String TAG_PERSISTENCE_TYPE = "persistenceType";

    /**
     * Tag pour que les entités etendent {@link TopiaEntityContextable} et
     * se fasse injecter le {@link TopiaDaoSupplier} par rapport aux autres
     * entités qui ne l'ont pas.
     *
     * @since 2.5.3
     */
    @TagValueDefinition(target = {ObjectModelClassifier.class}, documentation = "TODO")
    public static final String TAG_CONTEXTABLE = "contextable";

    /**
     * Tag pour le nom du champ / entité en BD.
     *
     * @see #getDbNameTagValue(ObjectModelElement)
     * @see TopiaGeneratorUtil#getDbName(ObjectModelElement)
     * @see TopiaGeneratorUtil#getReverseDbName(ObjectModelAttribute)
     */
    @TagValueDefinition(target = {ObjectModelElement.class},
                        documentation = "Sets the database name of an element of the model (a table or a column)")
    public static final String TAG_DB_NAME = "dbName";

    /**
     * Tag pour le nom du schema en BD.
     *
     * @see #getDbSchemaNameTagValue(ObjectModelClassifier, ObjectModel)
     * @see #getDbSchemaNameTagValue(ObjectModelAttribute, ObjectModel)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class, ObjectModelAttribute.class},
                        documentation = "Sets the database schema name")
    public static final String TAG_SCHEMA_NAME = "dbSchema";

    /**
     * Tag pour la taille du champ en BD.
     *
     * @see #getLengthTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the length of an attribute in database")
    public static final String TAG_LENGTH = "length";

    /**
     * Tag pour ajouter une annotation à un champ.
     *
     * @see #getAnnotationTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets an annotation of an attribute")
    public static final String TAG_ANNOTATION = "annotation";

    /**
     * Tag pour specfier le type d'acces a un champ.
     *
     * @see #getAccessTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the access type of an attribute (Hibernate mapping)")
    public static final String TAG_ACCESS = "access";

    /**
     * Tag pour ajouter un attribut dans une clef métier.
     *
     * @see #getNaturalIdTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets an attribute as part of a natural id (Hibernate Mapping)")
    public static final String TAG_NATURAL_ID = "naturalId";

    /**
     * Tag pour specifier si une clef metier est mutable.
     *
     * @see #getNaturalIdMutableTagValue(ObjectModelClassifier)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelClassifier.class},
                        documentation = "Sets if natural id of a entity is mutable (hibernate mapping)")
    public static final String TAG_NATURAL_ID_MUTABLE = "naturalIdMutable";

    /**
     * Tag pour permettre de choisir qui contrôle la relation N-N
     * bidirectionnelle. A utiliser sur les deux extremités de l'association.
     * Mettre inverse=false sur le rôle fils et inverse=true sur le rôle père.
     * Par défaut le inverse=true est placé sur le premier rôle trouvé dans
     * l'ordre alphabétique.
     *
     * @see #getInverseTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets which part of a N-N relation is master (inverse=true) and slave (inverse=false) (must be put on each side on a such relation) (Hibernate mapping)")
    public static final String TAG_INVERSE = "inverse";

    /**
     * Tag pour spécifier la caractère lazy d'une association multiple.
     *
     * @see #getLazyTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets if an association should be lazy (Hibernate mapping)")
    public static final String TAG_LAZY = "lazy";

    /**
     * Tag pour spécifier la caractère fetch d'une association multiple.
     *
     * @see #getFetchTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the fetch caracteristic of an attribute (Hibernate mapping)")
    public static final String TAG_FETCH = "fetch";

    /**
     * Tag pour spécifier la caractère order-by d'une association multiple.
     *
     * @see #getOrderByTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the order by propertie of an multiple association (Hibernate mapping)")
    public static final String TAG_ORDER_BY = "orderBy";

    /**
     * Tag pour spécifier la caractère not-null d'un attribut.
     *
     * @see #getNotNullTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets if an attribute must be not null (Hibernate mapping)")
    public static final String TAG_NOT_NULL = "notNull";

    /**
     * Tag à placer sur un l'attribut d'une entité. Cet attribut est de type
     * énumération : l'ajout de la tagValue indique qu'il faut utiliser le
     * {@code name} de l'énumération et non l'ordinal pour stocker la valeur en
     * base
     *
     * @see #hasUseEnumerationNameTagValue(ObjectModelAttribute, ObjectModelClassifier, ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class, ObjectModelAttribute.class},
                        documentation = "'true' if the value of this attribute of type Enumeration should be stored with its name (instead of using ordinal)")
    public static final String TAG_USE_ENUMERATION_NAME = "useEnumerationName";

    /**
     * Tag pour configurer l'interface du proxy sur autre chose que l'implementation par defaut.
     * <p/>
     * Par defaut :
     * null > generere le proxy sur l'interface de l'implementation
     * Autre valeur :
     * "none" > laisse la configuration par defaut d'hibernate
     *
     * @see #getPersistenceTypeTagValue(ObjectModelClassifier)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class},
                        documentation = "Configure the proxy interface on something else than the default implementation (null to use our default implementation, none to let hibernate deal it) (Hibernate mapping)")
    public static final String TAG_PROXY_INTERFACE = "hibernateProxyInterface";

    /**
     * Tag pour configurer la stategie d'heritage choisie.
     * <p/>
     * Par defaut : union-subclass : Table per class strategy
     * Autre valeur :
     * "subclass" > Single table per class hierarchy strategy
     * "joined-subclass" > Joined subclass strategy
     *
     * @see #getPersistenceTypeTagValue(ObjectModelClassifier)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class},
                        documentation = "Configure the proxy interface on something else than the default implementation (null to use our default implementation, none to let hibernate deal it) (Hibernate mapping)",
                        defaultValue = "union-subclass")
    public static final String TAG_INHERITANCE_STRATEGY = "inheritanceStrategy";

    /**
     * Tag to skip toString() methods generation on entities.
     *
     * @see #getNotGenerateToStringTagValue(ObjectModelClassifier, ObjectModel)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class},
                        defaultValue = "true",
                        documentation = "Tag to skip toString() methods generation on entities.")
    public static final String TAG_NOT_GENERATE_TO_STRING = "notGenerateToString";

    /**
     * Tag pour specifier de trier les attributs par nom lors de la generation.
     *
     * @see #getSortAttributeTagValue(ObjectModelClassifier, ObjectModel)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class},
                        documentation = "To sort attribute while generation")
    public static final String TAG_SORT_ATTRIBUTE = "sortAttribute";

    /**
     * Tag pour specfier si on doit générer la methode getOperator dans les daohelpers.
     *
     * @see #getGenerateOperatorForDAOHelperTagValue(ObjectModel)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class},
                        documentation = "To generate EntityOperation on generated DAOHelper")
    public static final String TAG_GENERATE_OPERATOR_FOR_DAO_HELPER = "generateOperatorForDAOHelper";

    /**
     * Tag pour spécifier le type d'une propriété dans le mapping hibernate.
     *
     * @see #getTypeTagValue(ObjectModelAttribute)
     * @since 2.5
     * @deprecated since 3.0, use now {@link #TAG_HIBERNATE_ATTRIBUTE_TYPE} which
     * permits to deal the same thing but at also model and classifier scope.
     */
    @Deprecated
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the hibernate type of an attribute (Hibernate mapping)")
    public static final String TAG_TYPE = "type";

    /**
     * Tag pour spécifier le type sql d'une propriété dans le mapping hibernate.
     *
     * @see #getSqlTypeTagValue(ObjectModelAttribute)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class},
                        documentation = "Sets the sql type of an attribute (Hibernate mapping)")
    public static final String TAG_SQL_TYPE = "sqlType";

    /**
     * To specify the abstract dao to use.
     * <p/>
     * If none given, will use the {@code org.nuiton.topia.persistence.TopiaDAOImpl}.
     * <p/>
     * Other value possible is {@code org.nuiton.topia.persistence.TopiaDAOLegacy}
     *
     * @see #getDaoImplementationTagValue(ObjectModel)
     * @since 2.5
     */
    @TagValueDefinition(target = {ObjectModel.class},
                        documentation = "Sets the fully qualified name of the DAO implementation to use in generated DAO (default is DAOImpl (base on TopiaQuery))")
    public static final String TAG_DAO_IMPLEMENTATION = "daoImplementation";

    /**
     * Stéréotype pour les attributs avec multiplicité nécessitant la création d'un index.
     *
     * @see #getIndexForeignKeysTagValue(ObjectModelAttribute, ObjectModel)
     * @since 2.6.5
     */
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelAttribute.class},
                        defaultValue = "true",
                        documentation = "Specifies if an nm-multiplicity attribute (or all nm-multiplicity attributes of a given model) needs an index in db (Hibernate mapping)")
    public static final String TAG_INDEX_FOREIGN_KEYS = "indexForeignKeys";

    /**
     * Tag to specify if we want to add an "id" property in DTO generated by
     * {@link EntityDTOTransformer}.
     *
     * @see EntityDTOTransformer
     * @see #shouldGenerateDTOTopiaIdTagValue(ObjectModelClassifier, ObjectModel)
     * @since 2.6.7
     * @deprecated since 3.0, will be removed at the same time that {@link EntityDTOTransformer}.
     */
    @Deprecated
    @TagValueDefinition(target = {ObjectModel.class, ObjectModelClassifier.class},
                        documentation = "Add a \"id\" property with its getter/setter on a DTO.")
    public static final String TAG_GENERATE_TOPIA_ID_IN_DTO = "generateDTOTopiaId";


    /**
     * Tag to specify a super class to use instead of
     * {@link org.nuiton.topia.persistence.internal.AbstractTopiaPersistenceContext}.
     * <p/>
     * <strong>Note:</strong> the class must implements
     * {@link org.nuiton.topia.persistence.internal.AbstractTopiaPersistenceContext}.
     *
     * @see org.nuiton.topia.persistence.internal.AbstractTopiaPersistenceContext
     * @see #getPersistenceContextSuperClassTagValue(org.nuiton.eugene.models.object.ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModel.class},
                        documentation = "Change the super class to use when generating PersistenceContext.")
    public static final String TAG_PERSISTENCE_CONTEXT_SUPER_CLASS = "persistenceContextSuperClass";

    /**
     * Tag to specify a super class to use instead of
     * {@link org.nuiton.topia.persistence.internal.AbstractTopiaApplicationContext}.
     * <p/>
     * <strong>Note:</strong> the class must implement
     * {@link org.nuiton.topia.persistence.internal.AbstractTopiaApplicationContext}.
     *
     * @see org.nuiton.topia.persistence.internal.AbstractTopiaApplicationContext
     * @see #getApplicationContextSuperClassTagValue(org.nuiton.eugene.models.object.ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModel.class},
                        documentation = "Change the super class to use when generating ApplicationContext.")
    public static final String TAG_APPLICATION_CONTEXT_SUPER_CLASS = "applicationContextSuperClass";

    /**
     * Tag to specify a super class to use instead of {@link org.nuiton.topia.persistence.internal.AbstractTopiaDao}.
     * <p/>
     * <strong>Note:</strong> the class must implements {@link org.nuiton.topia.persistence.TopiaDao}.
     *
     * @see org.nuiton.topia.persistence.internal.AbstractTopiaDao
     * @see #getDaoSuperClassTagValue(org.nuiton.eugene.models.object.ObjectModelClassifier, org.nuiton.eugene.models.object.ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModelClassifier.class, ObjectModel.class},
                        documentation = "Change the super class to use when generating dao.")
    public static final String TAG_DAO_SUPER_CLASS = "daoSuperClass";

    /**
     * Tag to specify a super class to use instead of {@link org.nuiton.topia.persistence.internal.AbstractTopiaEntity}.
     * <p/>
     * <strong>Note:</strong> the class must implements {@link org.nuiton.topia.persistence.TopiaEntity}.
     *
     * @see org.nuiton.topia.persistence.TopiaEntity
     * @see #getEntitySuperClassTagValue(org.nuiton.eugene.models.object.ObjectModelClassifier, org.nuiton.eugene.models.object.ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModelClassifier.class, ObjectModel.class},
                        documentation = "Change the super class to use when generating Entity.")
    public static final String TAG_ENTITY_SUPER_CLASS = "entitySuperClass";

    /**
     * Tag to change the type of an attribute in a hibernate mapping.
     * <p/>
     * This is a special tagValue that is dynamic.
     * <p/>
     * For example to change the type {@code String} into hibernate mapping type {@code text}, add this:
     * <pre>
     * model.tagValue.hibernateAttributeType.String=text
     * </pre>
     * <p/>
     * Before 3.0, you could do the same thing using:
     * <pre>
     * model.tagValue.String=text
     * </pre>
     * <p/>
     * The new way permits us to validate the usage of the tagValue, old way can't.
     *
     * @see #getHibernateAttributeType(ObjectModelAttribute, ObjectModelClassifier, ObjectModel)
     * @since 3.0
     */
    @TagValueDefinition(target = {ObjectModelAttribute.class, ObjectModelClassifier.class, ObjectModel.class},
                        documentation = "Change the super class to use when generating Entity.",
                        matcherClass = StartsWithTagNameMatcher.class)
    public static final String TAG_HIBERNATE_ATTRIBUTE_TYPE = "hibernateAttributeType";

    protected String findTagValue(String tagName, TagValueAble... elements) {
        String defaultValue = getDefaultValue(tagName);
        String result = TagValues.findTagValue(tagName, defaultValue, elements);
        return result;
    }

    protected boolean findBooleanTagValue(String tagName, TagValueAble... elements) {
        String defaultValue = getDefaultValue(tagName);
        boolean result = TagValues.findBooleanTagValue(tagName, defaultValue, elements);
        return result;
    }

    protected Boolean findNullableBooleanTagValue(String tagName, TagValueAble... elements) {
        String defaultValue = getDefaultValue(tagName);
        Boolean result = TagValues.findNullableBooleanTagValue(tagName, defaultValue, elements);
        return result;
    }

    /**
     * Obtain the value of the {@link #TAG_PERSISTENCE_TYPE} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_PERSISTENCE_TYPE
     * @since 2.5
     */
    public String getPersistenceTypeTagValue(ObjectModelClassifier classifier) {
        String value = findTagValue(TAG_PERSISTENCE_TYPE, classifier);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_INHERITANCE_STRATEGY} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_INHERITANCE_STRATEGY
     * @since 3.0
     */
    public String getInheritanceStrategyTagValue(ObjectModelClassifier classifier) {
        String value = findTagValue(TAG_INHERITANCE_STRATEGY, classifier);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_DB_NAME} tag value on the given classifier.
     *
     * @param element classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_DB_NAME
     * @since 2.5
     */
    public String getDbNameTagValue(ObjectModelElement element) {
        String value = findTagValue(TAG_DB_NAME, element);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_SCHEMA_NAME} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_SCHEMA_NAME
     * @since 2.5
     */
    public String getDbSchemaNameTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        String value = findTagValue(TAG_SCHEMA_NAME, classifier, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_SCHEMA_NAME} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @param model     model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_SCHEMA_NAME
     * @since 2.5
     */
    public String getDbSchemaNameTagValue(ObjectModelAttribute attribute, ObjectModel model) {
        String value = findTagValue(TAG_SCHEMA_NAME, attribute, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_LENGTH} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_LENGTH
     * @since 2.5
     */
    public String getLengthTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_LENGTH, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_ANNOTATION} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_ANNOTATION
     * @since 2.5
     */
    public String getAnnotationTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_ANNOTATION, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_ACCESS} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_ACCESS
     * @since 2.5
     */
    public String getAccessTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_ACCESS, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_NATURAL_ID} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_NATURAL_ID
     * @since 2.5
     */
    public boolean getNaturalIdTagValue(ObjectModelAttribute attribute) {
        boolean value = findBooleanTagValue(TAG_NATURAL_ID, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_NATURAL_ID_MUTABLE} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_NATURAL_ID_MUTABLE
     * @since 2.5
     */
    public boolean getNaturalIdMutableTagValue(ObjectModelClassifier classifier) {
        boolean value = findBooleanTagValue(TAG_NATURAL_ID_MUTABLE, classifier);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_CONTEXTABLE} tag value on the given classifier or model.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the boolean value of the found tag value or {@code false} if not found nor empty.
     * @see #TAG_CONTEXTABLE
     * @since 2.5
     */
    public boolean getContextableTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_CONTEXTABLE, classifier, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_INVERSE} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_INVERSE
     * @since 2.5
     */
    public String getInverseTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_INVERSE, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_LAZY} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_LAZY
     * @since 2.5
     */
    public String getLazyTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_LAZY, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_FETCH} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_FETCH
     * @since 2.5
     */
    public String getFetchTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_FETCH, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_ORDER_BY} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_ORDER_BY
     * @since 2.5
     */
    public String getOrderByTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_ORDER_BY, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_NOT_NULL} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_NOT_NULL
     * @since 2.5
     */
    public Boolean getNotNullTagValue(ObjectModelAttribute attribute) {
        Boolean value = findNullableBooleanTagValue(TAG_NOT_NULL, attribute);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_PROXY_INTERFACE} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_PROXY_INTERFACE
     * @since 2.5
     */
    public String getProxyInterfaceTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        String value = findTagValue(TAG_PROXY_INTERFACE, classifier, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_NOT_GENERATE_TO_STRING} tag value on the given class.
     *
     * @param clazz class to seek
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_NOT_GENERATE_TO_STRING
     * @since 2.5
     */
    public boolean getNotGenerateToStringTagValue(ObjectModelClassifier clazz, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_NOT_GENERATE_TO_STRING, clazz, model);
        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_SORT_ATTRIBUTE} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_SORT_ATTRIBUTE
     * @since 2.5
     */
    public boolean getSortAttributeTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_SORT_ATTRIBUTE, classifier, model);
        return value;
    }

    protected static final Map<String, String> HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES = Maps.newHashMap();
    static {
        // String
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(String.class.getSimpleName(), String.class.getName());

        // primitive types wrappers
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Boolean.class.getSimpleName(), Boolean.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Byte.class.getSimpleName(), Byte.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Character.class.getSimpleName(), Character.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Short.class.getSimpleName(), Short.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Integer.class.getSimpleName(), Integer.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Long.class.getSimpleName(), Long.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Float.class.getSimpleName(), Float.class.getName());
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Double.class.getSimpleName(), Double.class.getName());

        // some particular types
        HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.put(Blob.class.getSimpleName(), Blob.class.getName().toLowerCase());
    }

    /**
     * Obtain the value of the {@link #TAG_HIBERNATE_ATTRIBUTE_TYPE} tag value on the given attribute, classifier or model.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_HIBERNATE_ATTRIBUTE_TYPE
     * @since 3.0
     */
    public String getHibernateAttributeType(ObjectModelAttribute attribute, ObjectModelClassifier classifier, ObjectModel model) {
        // Look for "hibernateAttributeType" on attribute and classifier
        String tagValueName = TAG_HIBERNATE_ATTRIBUTE_TYPE;
        String value = TagValues.findTagValue(tagValueName, attribute, classifier);

        if (StringUtils.isEmpty(value)) {

            // If not found, try "hibernateAttributeType.<type>" on model
            tagValueName = TAG_HIBERNATE_ATTRIBUTE_TYPE + "." + attribute.getType();
            String defaultValue = HIBERNATE_ATTRIBUTE_TYPE_DEFAULT_VALUES.get(attribute.getType());
            value = TagValues.findTagValue(tagValueName, defaultValue, model);
        }

        return value;
    }

    /**
     * Obtain the value of the {@link #TAG_SQL_TYPE} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_TYPE
     * @since 2.5
     */
    public String getSqlTypeTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_SQL_TYPE, attribute);
        return value;
    }

    /**
     * Obtains the value of the {@link #TAG_DAO_IMPLEMENTATION} tag value on the given model.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_DAO_IMPLEMENTATION
     * @since 2.5
     */
    public String getDaoImplementationTagValue(ObjectModel model) {
        String value = findTagValue(TAG_DAO_IMPLEMENTATION, model);
        return value;
    }

    /**
     * Obtains the value of the  tag value {@link #TAG_INDEX_FOREIGN_KEYS} on the model or on the
     * given attribute.
     *
     * @param attribute attribute to test
     * @param model     model to test
     * @return none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_INDEX_FOREIGN_KEYS
     * @since 2.6.5
     */
    public boolean getIndexForeignKeysTagValue(ObjectModelAttribute attribute, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_INDEX_FOREIGN_KEYS, attribute, model);
        return value;
    }

    //TODO Javadoc
    public boolean hasUseEnumerationNameTagValue(ObjectModelAttribute attr, ObjectModelClassifier classifier, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_USE_ENUMERATION_NAME, attr, classifier, model);
        return value;
    }

    /**
     * Search if the TagValue {@link #TAG_GENERATE_TOPIA_ID_IN_DTO} has been activated in the model.
     *
     * @param classifier classifier to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_GENERATE_TOPIA_ID_IN_DTO
     * @since 2.6.7
     */
    public boolean shouldGenerateDTOTopiaIdTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_GENERATE_TOPIA_ID_IN_DTO, classifier, model);
        return value;
    }

    /**
     * Obtains the value of the {@link #TAG_PERSISTENCE_CONTEXT_SUPER_CLASS} tag value on the model.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_PERSISTENCE_CONTEXT_SUPER_CLASS
     * @since 3.0
     */
    public String getPersistenceContextSuperClassTagValue(ObjectModel model) {
        String value = findTagValue(TAG_PERSISTENCE_CONTEXT_SUPER_CLASS, model);
        return value;
    }

    /**
     * Obtains the value of the {@link #TAG_APPLICATION_CONTEXT_SUPER_CLASS} tag value on the model.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_APPLICATION_CONTEXT_SUPER_CLASS
     * @since 3.0
     */
    public String getApplicationContextSuperClassTagValue(ObjectModel model) {
        String value = findTagValue(TAG_APPLICATION_CONTEXT_SUPER_CLASS, model);
        return value;
    }

    /**
     * Obtains the value of the {@link #TAG_DAO_SUPER_CLASS} tag value on the given classifier or on the model.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_DAO_SUPER_CLASS
     * @since 3.0
     */
    public String getDaoSuperClassTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        String value = findTagValue(TAG_DAO_SUPER_CLASS, classifier, model);
        return value;
    }

    /**
     * Obtains the value of the {@link #TAG_ENTITY_SUPER_CLASS} tag value on the given classifier or on the model.
     *
     * @param model model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_ENTITY_SUPER_CLASS
     * @since 3.0
     */
    public String getEntitySuperClassTagValue(ObjectModelClassifier classifier, ObjectModel model) {
        String value = findTagValue(TAG_ENTITY_SUPER_CLASS, classifier, model);
        return value;
    }

    /**
     * Cherche si le tagvalue {@link #TAG_GENERATE_OPERATOR_FOR_DAO_HELPER} a été activé dans le model.
     *
     * @param model le modele utilisé
     * @return {@code true} si le tag value trouvé dans le modèle, {@code false}
     * sinon.
     * @since 2.5
     */
    public boolean getGenerateOperatorForDAOHelperTagValue(ObjectModel model) {
        boolean value = findBooleanTagValue(TAG_GENERATE_OPERATOR_FOR_DAO_HELPER, model);
        return value;
    }

    //---------------------------------------------------------------------------------------------------------------
    //--  DEPRECATED API TO REMOVE  ---------------------------------------------------------------------------------
    //---------------------------------------------------------------------------------------------------------------

    /**
     * Obtain the value of the {@link #TAG_TYPE} tag value on the given attribute.
     *
     * @param attribute attribute to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see #TAG_TYPE
     * @since 2.5
     * @deprecated since 3.0, use now {@link #getHibernateAttributeType(ObjectModelAttribute, ObjectModelClassifier, ObjectModel)}
     */
    @Deprecated
    public String getTypeTagValue(ObjectModelAttribute attribute) {
        String value = findTagValue(TAG_TYPE, attribute);
        return value;
    }
}
