/* *##% 
 * ToPIA :: Service Migration
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.topia.migration;

import java.io.File;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.topia.event.TopiaContextEvent;
import org.nuiton.topia.event.TopiaTransactionEvent;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.migration.callback.MigrationCallbackHandler;
import org.nuiton.util.Version;
import org.hibernate.cfg.Configuration;

/**
 * TopiaMigrationServiceImpl.java
 *
 * Classe principale du projet.
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon S�bastien
 * @author Tr�ve Vincent
 * @version $Revision: 1459 $
 *
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class TopiaMigrationServiceImpl extends MigrationServiceImpl implements TopiaMigrationService {

    // log
    private final static Log log = LogFactory.getLog(TopiaMigrationServiceImpl.class);
    /**
     * Nom des proprietes
     */
    static public final String MIGRATION_APPLICATION_VERSION = "topia.service.migration.version";
    static public final String MIGRATION_PREVIOUS_MAPPING_DIRECTORY = "topia.service.migration.mappingsdir";
    static public final String MIGRATION_MODEL_NAMES = "topia.service.migration.modelnames";
    static public final String MIGRATION_CALLBACKHANDLERS = "topia.service.migration.callbackhandlers";
    /**
     * C'est la meme, mais ca vire des dependances
     */
    static final protected String TOPIA_PERSISTENCE_DIRECTORIES = "topia.persistence.directories";
    static final protected String TOPIA_PERSISTENCE_CLASSES = "topia.persistence.classes";

    public TopiaMigrationServiceImpl() {
        super();
    }

    @Override
    public Class<?>[] getPersistenceClasses() {
        return null;
    }

    @Override
    public String getServiceName() {
        return TopiaMigrationService.SERVICE_NAME;
    }

    @Override
    public boolean preInit(TopiaContextImplementor context) {
        Properties config = context.getConfig();

        String mappingdir = config.getProperty(MIGRATION_PREVIOUS_MAPPING_DIRECTORY, null);
        String applicationversion = config.getProperty(MIGRATION_APPLICATION_VERSION, null);

        // Creer une nouvelle configuration a partir des fichiers speciaux
        // de TOPIA
        // il n'y a pas ici de hibernate.cfg.xml
        Configuration hibernateConfiguration = new Configuration();

        // ajout des repertoires contenant les mappings hibernate
        String[] dirs = config.getProperty(
                TOPIA_PERSISTENCE_DIRECTORIES, "").split(",");
        for (String dir : dirs) {
            dir = dir.trim();
            if (!"".equals(dir)) {
                hibernateConfiguration.addDirectory(new File(dir));
            }
        }

        // ajout des classes dites persistentes
        String listPersistenceClasses = config.getProperty(
                TOPIA_PERSISTENCE_CLASSES, "");

        String[] classes = listPersistenceClasses.split(",");
        for (String classname : classes) {
            classname = classname.trim();
            if (!"".equals(classname)) {
                Class<?> clazz;
                try {
                    clazz = Class.forName(classname);
                    hibernateConfiguration.addClass(clazz);
                } catch (ClassNotFoundException eee) {
                    log.error("Persistent class " + classname + " not found", eee);
                }
            }
        }

        Properties prop = new Properties();
        prop.putAll(hibernateConfiguration.getProperties());
        prop.putAll(config);
        /*TODO TOPIA_PERSISTENCE_PROPERTIES_FILE
        try {
        prop.putAll(TopiaUtil.getProperties(config.getProperty(
        TOPIA_PERSISTENCE_PROPERTIES_FILE)));
        } catch (TopiaNotFoundException e1) {
        logger.error(
        "Can't found property file",e1);
        }*/
        hibernateConfiguration.setProperties(prop);

        // reseigne la configuration qui sera utilisee
        this.setConfiguration(hibernateConfiguration);

        // test version null
        if (applicationversion == null) {
            log.error("'" + MIGRATION_APPLICATION_VERSION + "' not set.");
        } else {
            this.setApplicationVersion(applicationversion);
        }


        // ajout des callbackhandlers
        String callbackHandlers = config.getProperty(MIGRATION_CALLBACKHANDLERS, "");

        String[] tabCallbackHandlers = callbackHandlers.split(",");
        for (String callbackHandler : tabCallbackHandlers) {
            callbackHandler = callbackHandler.trim();
            if (!"".equals(callbackHandler)) {
                Class<?> clazz;
                try {
                    clazz = (Class<?>) Class.forName(callbackHandler);
                    MigrationCallbackHandler instance = (MigrationCallbackHandler) clazz.newInstance();
                    addMigrationCallbackHandler(instance);
                } catch (ClassNotFoundException e) {
                    log.error(
                            "CallbackHandler Class " + callbackHandler + " not found", e);
                } catch (InstantiationException e) {
                    log.error(
                            "CallbackHandler class " + callbackHandler + " cannot be instanciated", e);
                } catch (IllegalAccessException e) {
                    log.error(
                            "CallbackHandler class " + callbackHandler + " cannot be accessed", e);
                }
            }
        }

        // add topia context listener
        context.addTopiaContextListener(this);
        context.addTopiaTransactionVetoable(this);

        // test mappingdir null
        if (mappingdir == null) {
            log.error("'" + MIGRATION_PREVIOUS_MAPPING_DIRECTORY + "' not set.");
        } else {

            // test applicationversion null
            if (applicationversion != null) {
                // effectue la migration de tous les modeles
                String modelnamesList = config.getProperty(
                        MIGRATION_MODEL_NAMES, "");
                String[] modelnames = modelnamesList.split(",");

                for (String modelname : modelnames) {
                    // Don't use File.separator, don't work on windows
                    this.setMappingsDirectory(mappingdir + "/" + modelname);

                    boolean complete = false;

                    try {
                        // migration
                        complete = migrateSchema();
                    } catch (MigrationServiceException e) {
                        log.error("Can't migrate schema", e);
                    }

                    if (!complete) {
                        if (log.isDebugEnabled()) {
                            log.error("Database migration not complete");
                        }
                        throw new TopiaRuntimeException("Database migration not succesfully ended !");
                    }
                }
            }
        }

        return true;
    }

    @Override
    public boolean postInit(TopiaContextImplementor context) {
        return true;
    }

    @Override
    public void postCreateSchema(TopiaContextEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("postCreateSchema event called : put version in database");
        }

        TopiaContextImplementor context = (TopiaContextImplementor) event.getSource();

        String versionStr = context.getConfig().getProperty(MIGRATION_APPLICATION_VERSION, null);

        Version version = new Version(versionStr);
        putVersionInDatabase(context.getConfig(), version, true);
    }

    @Override
    public void postUpdateSchema(TopiaContextEvent event) {
    }

    @Override
    public void preCreateSchema(TopiaContextEvent event) {
    }

    @Override
    public void preUpdateSchema(TopiaContextEvent event) {
    }

    @Override
    public void preRestoreSchema(TopiaContextEvent event) {
    }

    @Override
    public void postRestoreSchema(TopiaContextEvent event) {

        if (log.isInfoEnabled()) {
            log.info("postRestoreSchema event detected, redo, schema migration");
        }
        try {
            migrateSchema();
        } catch (MigrationServiceException e) {
            if (log.isErrorEnabled()) {
                log.error("postRestoreSchema schema migration failed", e);
            }
        }
    }

    @Override
    public void beginTransaction(TopiaTransactionEvent event) {

        TopiaContextImplementor context = (TopiaContextImplementor) event.getSource();

        // add topia context listener
        context.addTopiaContextListener(this);

    }
}
