package org.nuiton.topia;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: AbstractTopiaApplicationContext.java 2843 2013-10-11 22:40:03Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-5/topia-persistence/src/main/java/org/nuiton/topia/AbstractTopiaApplicationContext.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.service.jdbc.connections.spi.ConnectionProvider;
import org.hibernate.service.spi.Stoppable;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.SchemaUpdate;
import org.nuiton.topia.event.TopiaEntitiesVetoable;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.event.TopiaSchemaListener;
import org.nuiton.topia.event.TopiaTransactionListener;
import org.nuiton.topia.event.TopiaTransactionVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaFiresSupport;
import org.nuiton.topia.framework.TopiaService;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.topia.persistence.DefaultTopiaIdFactory;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaIdFactory;

import java.beans.PropertyChangeListener;
import java.lang.ref.WeakReference;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * The application context is the main class in ToPIA usage. This class is a kind of equivalent of the RootTopiaContext.
 * It contains only high level methods and new contexts creation (transaction begin, ...). This class has to be extended
 * by the user, even if some default one could be automatically generated.
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 * @since 3.0
 */
public abstract class AbstractTopiaApplicationContext<K extends TopiaPersistenceContext> implements TopiaListenableSupport, TopiaServiceSupport {

    private static final Log log = LogFactory.getLog(AbstractTopiaApplicationContext.class);

    protected TopiaIdFactory topiaIdFactory;

    protected ImmutableMap<String, String> configuration;

    protected TopiaFiresSupport topiaFiresSupport = new TopiaFiresSupport();

    protected TopiaServiceSupport topiaServiceSupport;

    protected HibernateProvider hibernateProvider;

    protected boolean closed = false;

    protected List<WeakReference<TopiaPersistenceContext>> persistenceContexts = Lists.newArrayList(); // TODO AThimel 11/10/13 Which implementation ?

    public abstract K newPersistenceContext();

    public AbstractTopiaApplicationContext(Properties properties) {
        // TODO arnaud 11/10/13 vérifier le comportement
        this((Map) ImmutableMap.copyOf(properties));
    }

    public AbstractTopiaApplicationContext(Map<String, String> configuration) {
        Map<String, String> configurationCopy = Maps.newHashMap();
        configurationCopy.putAll(configuration);
        if ( ! configuration.containsKey(TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES)) {
            configurationCopy.put(TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES, getImplementationClassesAsString());
        }
        this.configuration = ImmutableMap.copyOf(configurationCopy);
        this.topiaServiceSupport = new TopiaServiceSupportImpl(this);
        // FIXME AThimel 11/10/13 This is not nice, need a 2-step init because of mutual dependency with hibernateProvider
        ((TopiaServiceSupportImpl)this.topiaServiceSupport).init();
    }

    protected abstract Class<? extends TopiaEntity>[] getImplementationClasses();

    protected String getImplementationClassesAsString() {
        StringBuilder buffer = new StringBuilder();
        for (Class<? extends TopiaEntity> aClass : getImplementationClasses()) {
            buffer.append(',').append(aClass.getName());
        }
        return buffer.substring(1);
    }

    protected void registerPersistenceContext(TopiaPersistenceContext persistenceContext) {
        persistenceContexts.add(new WeakReference<TopiaPersistenceContext>(persistenceContext));
    }

    protected HibernateProvider getHibernateProvider() {
        if (hibernateProvider == null) {
            hibernateProvider = new HibernateProvider(configuration, topiaServiceSupport);
        }
        return hibernateProvider;
    }

    protected TopiaListenableSupport getTopiaListenableSupport() {
        return topiaFiresSupport;
    }

    protected ImmutableMap<String, String> getConfiguration() {
        return configuration;
    }

    protected TopiaIdFactory getTopiaIdFactory() {
        if (topiaIdFactory == null) {
            String topiaIdFactoryClassName =
                    getConfiguration().get(TopiaContextFactory.CONFIG_PERSISTENCE_TOPIA_ID_FACTORY_CLASS_NAME);
            if (StringUtils.isEmpty(topiaIdFactoryClassName)) {
                topiaIdFactory = new DefaultTopiaIdFactory();
            } else {
                try {
                    Class topiaIdFactoryClass = Class.forName(topiaIdFactoryClassName);
                    Preconditions.checkState(TopiaIdFactory.class.isAssignableFrom(topiaIdFactoryClass),
                            topiaIdFactoryClassName + " is not a valid class name. The class must implements "
                                    + TopiaIdFactory.class.getSimpleName());

                    topiaIdFactory = (TopiaIdFactory) topiaIdFactoryClass.newInstance();
                } catch (ClassNotFoundException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                } catch (InstantiationException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                } catch (IllegalAccessException e) {
                    throw new TopiaException("Unable to create user specified TopiaIdFactory", e);
                }
            }

        }
        return topiaIdFactory;
    }

    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        topiaFiresSupport.addTopiaEntityListener(listener);
    }

    @Override
    public void addTopiaEntityListener(Class<? extends TopiaEntity> entityClass, TopiaEntityListener listener) {
        topiaFiresSupport.addTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        topiaFiresSupport.removeTopiaEntityListener(listener);
    }

    @Override
    public void removeTopiaEntityListener(Class<? extends TopiaEntity> entityClass, TopiaEntityListener listener) {
        topiaFiresSupport.removeTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaFiresSupport.addTopiaEntityVetoable(vetoable);
    }

    @Override
    public void addTopiaEntityVetoable(Class<? extends TopiaEntity> entityClass, TopiaEntityVetoable vetoable) {
        topiaFiresSupport.addTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntityVetoable(vetoable);
    }

    @Override
    public void removeTopiaEntityVetoable(Class<? extends TopiaEntity> entityClass, TopiaEntityVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void addTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        topiaFiresSupport.addTopiaEntitiesVetoable(vetoable);
    }

    @Override
    public void removeTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        topiaFiresSupport.removeTopiaEntitiesVetoable(vetoable);
    }

    @Override
    public void addTopiaTransactionListener(TopiaTransactionListener listener) {
        topiaFiresSupport.addTopiaTransactionListener(listener);
    }

    @Override
    public void removeTopiaTransactionListener(TopiaTransactionListener listener) {
        topiaFiresSupport.removeTopiaTransactionListener(listener);
    }

    @Override
    public void addTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        topiaFiresSupport.addTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void removeTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        topiaFiresSupport.removeTopiaTransactionVetoable(vetoable);
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        topiaFiresSupport.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        topiaFiresSupport.removePropertyChangeListener(listener);
    }

    @Override
    public void addTopiaSchemaListener(TopiaSchemaListener listener) {
        topiaFiresSupport.addTopiaSchemaListener(listener);
    }

    @Override
    public void removeTopiaSchemaListener(TopiaSchemaListener listener) {
        topiaFiresSupport.removeTopiaSchemaListener(listener);
    }

    @Override
    public <E extends TopiaService> boolean serviceEnabled(Class<E> interfaceService) {
        return topiaServiceSupport.serviceEnabled(interfaceService);
    }

    @Override
    public <E extends TopiaService> E getService(Class<E> interfaceService) throws TopiaNotFoundException {
        return topiaServiceSupport.getService(interfaceService);
    }

    @Override
    public Map<String, TopiaService> getServices() {
        return topiaServiceSupport.getServices();
    }

    public List<Class<?>> getPersistenceClasses() {
        return getHibernateProvider().getPersistentClasses();
    }

    public boolean isSchemaEmpty() {
        Configuration configuration = getHibernateProvider().getHibernateConfiguration();
        boolean result = TopiaUtil.isSchemaEmpty(configuration);
        return result;
    }

    public boolean isTableExists(Class<?> clazz) {
        Configuration configuration = getHibernateProvider().getHibernateConfiguration();
        boolean result = TopiaUtil.isSchemaExist(configuration, clazz.getName());
        return result;
    }

    public String getSchemaName() {
        // TODO AThimel 02/08/13 I absolutely don't know if it works
        return getConfiguration().get(TopiaContextFactory.CONFIG_DEFAULT_SCHEMA);
    }

    public void createSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }
            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreCreateSchema(this);
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).execute(showSchema, true, false, true);
            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePostCreateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not create schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    public void showCreateSchema() {
        try {
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).
                    execute(true, false, false, true);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not show create schema for reason: %s",
                            eee.getMessage()), eee);
        }

    }

    public void updateSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }

            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreUpdateSchema(this);
            new SchemaUpdate(getHibernateProvider().getHibernateConfiguration()).execute(showSchema,
                    true);
            // topiaFiresSupport.firePostUpdateSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not update schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    public void dropSchema() {
        try {
            boolean showSchema = false;
            if (log.isDebugEnabled()) {
                showSchema = true;
            }

            // TODO brendan 11/10/13 reable event
            // topiaFiresSupport.firePreDropSchema(this);
            new SchemaExport(getHibernateProvider().getHibernateConfiguration()).execute(showSchema, true, true, false);
//            topiaFiresSupport.firePostDropSchema(this);
        } catch (HibernateException eee) {
            throw new TopiaException(
                    String.format("Could not drop schema for reason: %s",
                            eee.getMessage()), eee);
        }
    }

    public void closeContext() throws TopiaException {
        // Throw exception if context is already closed
        Preconditions.checkState(!closed, "Context was already closed");

        // TODO AThimel 11/10/13 Implement close on persistence contexts
//        // Remove all children context
//        for (TopiaPersistenceContext child : getChildContext()) {
//            // Avoid to have exception from checkClosed method on child
//            if (!child.isClosed()) {
//                child.close();
//            }
//        }

        hibernateProvider.close();
        closed = true;

        if (log.isDebugEnabled()) {
            log.debug("TopiaApplicationContext closed");
        }
    }

    public boolean isClosed() {
        return closed;
    }

}
