/**
 * *##% Nuiton utilities library
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.i18n;

import java.net.URL;
import java.net.URLClassLoader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import org.nuiton.i18n.bundle.I18nBundle;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.i18n.bundle.I18nBundleFactory;
import org.nuiton.util.ClassLoaderUtil;
import org.nuiton.util.Resource;
import org.nuiton.util.StringUtil;

/**
 * Classe responsible of loading of I18n system.
 * <p/>
 * Contains the current used {@link #language} (can be null, if not set), and the list of already loaded {@link #languages}.
 * <p/>
 * <p/>
 * Note: <b>Init methods are package acces and should not be used alone, but within {@link I18n} class <code>init(XXX)</code> methods.</b>
 *
 * @author chemit
 */
public class I18nLoader {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(I18nLoader.class);
    /** le language actuellement utilise */
    protected Language language;
    /** le cache de languages deja charges */
    protected List<Language> languages;
    /** le cache des urls de recheche des bundles */
    protected static URL[] urls;
    /** le cache de bundles deja charges */
    protected I18nBundle[] bundles;
    /** la locale par defaut a utiliser */
    protected final Locale defaultLocale;
    /** le nom de l'unique bunlde a charger (mode unique) */
    protected final String uniqueBundleName;

    public I18nLoader(Locale defaultLocale) {
        this(defaultLocale, null);
    }

    public I18nLoader(Locale defaultLocale, String uniqueBundleName) {
        this.defaultLocale = defaultLocale;
        this.uniqueBundleName = uniqueBundleName;
    }

    /** @return current language loaded or null, if no language was load */
    public Language getLanguage() {
        return language;
    }

    /** @return le cache de language avec instanciation paresseuse */
    public List<Language> getLanguages() {
        if (languages == null) {
            languages = new ArrayList<Language>();
        }
        return languages;
    }

    public Locale getDefaultLocale() {
        return defaultLocale;
    }

    public boolean isEmpty() {
        checkInit();
        boolean isEmpty = I18nBundleFactory.isEmpty(bundles);
        return isEmpty;
    }

    /** @return array of all locales loaded */
    public Locale[] getLocales() {
        checkInit();
        Locale[] result = I18nBundleFactory.getLocales(bundles);
        return result;
    }

    public I18nBundle[] getBundles() {
        checkInit();
        return bundles;
    }

    public I18nBundle[] getBundles(Locale l) {
        checkInit();
        I18nBundle[] result = I18nBundleFactory.getBundles(l, bundles);
        return result;
    }

    public I18nBundleEntry[] getBundleEntries() {
        checkInit();
        I18nBundleEntry[] result = I18nBundleFactory.getBundleEntries(bundles);
        return result;
    }

    public I18nBundleEntry[] getBundleEntries(Locale l) {
        checkInit();
        I18nBundleEntry[] result = I18nBundleFactory.getBundleEntries(l, defaultLocale, bundles);
        return result;
    }

    void init() {

        if (isInit()) {
            // already init
            return;
        }

        // get all bundles urls
        if (urls == null || urls.length == 0) {

            // cache this expensive search

            if (uniqueBundleName != null) {
                // on recherche directement un bundle precis a aprtir
                // de son fichier de definition
                urls = I18nBundleFactory.getURLs(uniqueBundleName);
                if (urls == null) {
                    log.warn("coudl not find uniqueBundleName i18n " + uniqueBundleName);
                }
            }
            if (urls == null) {
                // on utilise le mecanisme de recherche des bundles dans toutes
                // les entrees du classloader
                urls = getURLs(Language.getLoader(), I18n.getExtraURL());
            }
        }

        long t0 = System.nanoTime();

        // detect bundles
        List<I18nBundle> bundleDetected = I18nBundleFactory.detectBundles(urls);

        // save bundles in cache
        this.bundles = bundleDetected.toArray(new I18nBundle[bundleDetected.size()]);

        log.info(bundleDetected.size() + " bundle(s) found, [" + getBundleEntries().length + " file(s)] in " + StringUtil.convertTime(System.nanoTime() - t0));
    }

    /**
     * Set a new language in loader, given a locale.
     *
     * @param locale        la locale du language requis
     * @param bundleManager bundle manager to used
     */
    synchronized void setLanguage(Locale locale) {
        init();
        if (log.isDebugEnabled()) {
            log.debug("locale: " + locale);
        }
        Language result = getLanguage(locale);
        if (result == null) {
            result = addLanguage(locale);
        } else {
            log.debug("using cached language : " + result);
        }
        language = result;
        //TC-20090702 the selected langue is the default locale, usefull for
        // objects dealing with the default locale (swing widgets,...)
        Locale.setDefault(locale);
    }

    /**
     * Close loader and release cache ofg language.
     * <p/>
     * Current language will be also clean.
     */
    void close() {
        if (languages != null) {
            log.info("nb languages loaded : " + languages.size());
            for (Language l : languages) {
                l.close();
            }
            languages.clear();
            languages = null;
        }
        if (urls != null) {
            urls = null;
        }
        if (bundles != null) {
            bundles = null;
        }
        language = null;
    }

    /**
     * @param locale la locale du language recherche
     * @return le language trouve dans le cache, ou null.
     */
    Language getLanguage(Locale locale) {

        if (!(languages == null || languages.isEmpty())) {
            for (Language l : languages) {
                if (locale.equals(l.getLocale())) {
                    return l;
                }
            }
        }
        return null;
    }

    Language addLanguage(Locale locale) {
        Language result;
        result = new Language(locale);
        long t0 = System.nanoTime();
        I18nBundleEntry[] entries = getBundleEntries(locale);
        result.load(entries);
        log.info(result + ", nbEntries: " + entries.length + ", nbSentences: " + result.size() + " in " + StringUtil.convertTime(System.nanoTime() - t0));
        getLanguages().add(result);
        return result;
    }

    boolean isInit() {
        return bundles != null;
    }

    void checkInit() {
        if (!isInit()) {
            throw new IllegalStateException("should call init method on " + I18nLoader.class);
        }
    }

    /**
     * Detecte les urls de toutes les entrees de bunbles sur tout un classLoader.
     *
     * Il s'agit du mode initialie de detection des entréés de bundles, i.e des
     * fichiers de traductions.
     *
     * <b>Note: </b> Cette methode devient couteuse des que le classLoader
     * contient de nombreuses entrées. Il est meiux d'utiliser le second type
     * de chargement qui n'utilise qu'un seul fichier de traduction unifié.
     *
     * @param loader le classloader a utiliser pour trouver les resources.
     * @return les urls des entrees de bundles
     */
    public static URL[] getURLs(URLClassLoader loader) {
        try {
            // on calcule toutes les urls utilisable dans le classloader donnee
            List<URL> urlToSeek = new ArrayList<URL>();
            urlToSeek.addAll(Arrays.asList(ClassLoaderUtil.getDeepURLs(loader)));

            // on va maintenant supprimer toutes les urls qui ne respectent pas
            // le pattern i18n : il faut que la resource contienne un repertoire i18n
            // ce simple test permet de restreindre la recherche des resources
            // i18n qui est tres couteuse
            int size = urlToSeek.size();
            for (Iterator<URL> it = urlToSeek.iterator(); it.hasNext();) {
                URL url = it.next();
                if (!Resource.containsDirectDirectory(url, I18nBundleFactory.DIRECTORY_SEARCH_BUNDLE_PATTERN)) {
                    if (log.isDebugEnabled()) {
                        log.debug("skip url with no " + I18nBundleFactory.DIRECTORY_SEARCH_BUNDLE_PATTERN + " directory : " + url);
                    }
                    it.remove();
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("detect " + urlToSeek.size() + " i18n capable url (out of " + size + ")");
            }
            // on effectue la recherche des urls des resources i18n (tous les
            // fichiers de traductions) sur toutes les urls precedemment calculees)
            List<URL> result = Resource.getURLs(I18nBundleFactory.SEARCH_BUNDLE_PATTERN, urlToSeek.toArray(new URL[urlToSeek.size()]));
            if (log.isDebugEnabled()) {
                for (URL url : result) {
                    log.debug(url.toString());
                }
            }
            return result.toArray(new URL[result.size()]);
        } catch (Exception eee) {
            log.warn("Unable to find urls for loader : " + loader + " for reason " + eee.getMessage(), eee);
            return new URL[0];
        }
    }

    /**
     * Recherche la liste des url de toutes les resources i18n, i.e les urls
     * des fichiers de traduction.
     *
     * @param loader   le classLoader où trouver les bundles
     * @param extraUrl des urls de resources i18n deja calcule, à ajouter au resultat sans traitement particulier
     * @return la liste des urls de bundle i18n pour la langue donné
     */
    public static URL[] getURLs(URLClassLoader loader, URL... extraUrl) {

        try {
            // on calcule toutes les urls utilisable dans le classloader donnee
            List<URL> urlToSeek = new ArrayList<URL>();
            urlToSeek.addAll(Arrays.asList(ClassLoaderUtil.getDeepURLs(loader)));
            // on ajoute les urls de resources i18n donnes
            if (extraUrl.length > 0) {
                urlToSeek.addAll(Arrays.asList(extraUrl));
            }
            // on va maintenant supprimer toutes les urls qui ne respectent pas
            // le pattern i18n : il faut que la resource contienne un repertoire i18n
            // ce simple test permet de restreindre la recherche des resources
            // i18n qui est tres couteuse
            int size = urlToSeek.size();
            for (Iterator<URL> it = urlToSeek.iterator(); it.hasNext();) {
                URL url = it.next();
                if (!Resource.containsDirectDirectory(url, I18nBundleFactory.DIRECTORY_SEARCH_BUNDLE_PATTERN)) {
                    if (log.isDebugEnabled()) {
                        log.debug("skip url with no " + I18nBundleFactory.DIRECTORY_SEARCH_BUNDLE_PATTERN + " directory : " + url);
                    }
                    it.remove();
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("detect " + urlToSeek.size() + " i18n capable url (out of " + size + ")");
            }
            // on effectue la recherche des urls des resources i18n (tous les
            // fichiers de traductions) sur toutes les urls precedemment calculees)
            List<URL> result = Resource.getURLs(I18nBundleFactory.SEARCH_BUNDLE_PATTERN, urlToSeek.toArray(new URL[urlToSeek.size()]));
            if (log.isDebugEnabled()) {
                for (URL url : result) {
                    log.debug(url.toString());
                }
            }
            return result.toArray(new URL[result.size()]);
        } catch (Exception eee) {
            log.warn("Unable to find urls for loader : " + loader + " for reason " + eee.getMessage(), eee);
            return new URL[0];
        }
    }
}
