/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: AbstractRedmineReport.java 157 2010-10-08 10:23:16Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.1.4/maven-jredmine-plugin/src/main/java/org/nuiton/jredmine/plugin/report/AbstractRedmineReport.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin.report;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.versioning.InvalidVersionSpecificationException;
import org.apache.maven.artifact.versioning.VersionRange;
import org.apache.maven.doxia.module.xhtml.decoration.render.RenderingContext;
import org.apache.maven.doxia.sink.Sink;
import org.apache.maven.doxia.site.decoration.Body;
import org.apache.maven.doxia.site.decoration.DecorationModel;
import org.apache.maven.doxia.site.decoration.Skin;
import org.apache.maven.doxia.siterenderer.Renderer;
import org.apache.maven.doxia.siterenderer.RendererException;
import org.apache.maven.doxia.siterenderer.SiteRenderingContext;
import org.apache.maven.doxia.siterenderer.sink.SiteRendererSink;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.reporting.MavenReport;
import org.apache.maven.reporting.MavenReportException;
import org.codehaus.plexus.i18n.I18N;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.plugin.AbstractRedmineMojo;
import org.nuiton.plugin.Plugin;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

/**
 * Abstract redmine report mojo.
 *
 * @author chemit
 * @requiresReports true
 * @since 1.0.0
 */
public abstract class AbstractRedmineReport extends AbstractRedmineMojo implements MavenReport, Plugin, RestClientConfiguration {

    /**
     * Flag to know if anonymùous connexion to redmine server is required.
     * <p/>
     * For this goal, the default value is {@code true}
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @parameter expression="${redmine.anonymous}" default-value="true"
     * @since 1.1.3
     */
    protected boolean anonymous;

    /**
     * Template strings per system that is used to discover the URL to use to display an issue report. Each key in this
     * map denotes the (case-sensitive) identifier of the issue tracking system and its value gives the URL template.
     * <p>
     * There are 2 template tokens you can use. <code>%URL%</code>: this is computed by getting the
     * <code>&lt;issueManagement&gt;/&lt;url&gt;</code> value from the POM, and removing the last '/'
     * and everything that comes after it. <code>%ISSUE%</code>: this is the issue number.
     * </p>
     *
     * @parameter expression="${redmine.issueLinkTemplate}" default-value="%URL%/issues/show/%ISSUE%"
     * @since 1.0.0
     */
    protected String issueLinkTemplate;

    /**
     * Template strings per system that is used to discover the URL to use to display an issue report. Each key in this
     * map denotes the (case-sensitive) identifier of the issue tracking system and its value gives the URL template.
     * <p>
     * There are 2 template tokens you can use. <code>%URL%</code>: this is computed by getting the
     * <code>&lt;issueManagement&gt;/&lt;url&gt;</code> value from the POM, and removing the last '/'
     * and everything that comes after it. <code>%VERSION%</code>: this is the issue number.
     * </p>
     *
     * @parameter expression="${redmine.versionLinkTemplate}" default-value="%URL%/versions/show/%VERSION%"
     * @since 1.0.0
     */
    protected String versionLinkTemplate;

    /**
     * Local Repository.
     *
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepository;

    /**
     * Report output directory. Note that this parameter is only relevant if the goal is run from the command line or
     * from the default build lifecycle. If the goal is run indirectly as part of a site generation, the output
     * directory configured in the Maven Site Plugin is used instead.
     *
     * @parameter default-value="${project.reporting.outputDirectory}"
     */
    protected File outputDirectory;

    /** @component */
    protected ArtifactResolver resolver;

    /** @component */
    protected ArtifactFactory factory;

    /**
     * Internationalization.
     *
     * @component
     */
    protected I18N i18n;

    /**
     * Doxia Site Renderer.
     *
     * @component
     */
    protected Renderer siteRenderer;

    private Sink sink;

    private File reportOutputDirectory;

    protected abstract void executeReport(Locale locale) throws MavenReportException;

    /** @return {@code true} if report should be skip */
    protected abstract boolean skipReport();

    public AbstractRedmineReport(boolean requireProject, boolean requireVersion, boolean requireUser) {
        super(requireProject, requireVersion, requireUser);
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }
    ///////////////////////////////////////////////////////////////////////////
    /// Plugin
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected void doAction() throws Exception {

        try {
            DecorationModel model = new DecorationModel();
            model.setBody(new Body());
            Map<String, String> attributes = new HashMap<String, String>();
            attributes.put("outputEncoding", encoding);
            Locale currentLocale = Locale.getDefault();
            SiteRenderingContext siteContext = siteRenderer.createContextForSkin(getSkinArtifactFile(), attributes,
                                                                                 model, getName(currentLocale), currentLocale);

            RenderingContext context = new RenderingContext(outputDirectory, getOutputName() + ".html");

            SiteRendererSink newSink = new SiteRendererSink(context);
            generate(newSink, currentLocale);

            createDirectoryIfNecessary(outputDirectory);

            Writer writer = new FileWriter(new File(outputDirectory, getOutputName() + ".html"));

            siteRenderer.generateDocument(writer, newSink, siteContext);

            siteRenderer.copyResources(siteContext, new File(project.getBasedir(), "src/site/resources"),
                                       outputDirectory);

        } catch (RendererException e) {
            throw new MojoExecutionException(
                    "An error has occurred in " + getName(Locale.ENGLISH) + " report generation.", e);
        } catch (IOException e) {
            throw new MojoExecutionException(
                    "An error has occurred in " + getName(Locale.ENGLISH) + " report generation.", e);
        } catch (MavenReportException e) {
            throw new MojoExecutionException(
                    "An error has occurred in " + getName(Locale.ENGLISH) + " report generation.", e);
        } finally {

            closeService();
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    /// MavenReport
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public boolean canGenerateReport() {
        boolean init;
        if (getLog().isDebugEnabled()) {
            getLog().debug("check can use report");
        }
        if (skipReport()) {
            getLog().info("User ask to skip report \"" + getName(Locale.ENGLISH) + "\".");
            return false;
        }
        if (session.getSettings().isOffline()) {
            // skip when offline
            getLog().info("Skipped \"" + getName(Locale.ENGLISH) + "\" report in offline mode.");
            return false;
        }
        try {
            init();
            init = true;
        } catch (Exception ex) {
            if (isVerbose()) {
                getLog().error("could not init report goal for reason " + ex.getMessage(), ex);
            } else {
                getLog().error("could not init report goal for reason " + ex.getMessage());
            }
            init = false;
        }
        if (!init) {
            getLog().info("Skipped \"" + getName(Locale.ENGLISH) + "\" report, goal could be initialized.");
        }
        if (getLog().isDebugEnabled()) {
            getLog().debug("check can use report : " + init);
        }
        return init;
    }

    @Override
    public void generate(org.codehaus.doxia.sink.Sink sink, Locale locale)
            throws MavenReportException {
        if (sink == null) {
            throw new MavenReportException("You must specify a sink.");
        }

        this.sink = sink;

        executeReport(locale);
    }

    @Override
    public String getName(Locale locale) {
        return getBundle(locale).getString("report.name");
    }

    @Override
    public String getDescription(Locale locale) {
        return getBundle(locale).getString("report.description");
    }

    @Override
    public String getCategoryName() {
        return CATEGORY_PROJECT_REPORTS;
    }

    @Override
    public File getReportOutputDirectory() {
        if (reportOutputDirectory == null) {
            reportOutputDirectory = new File(outputDirectory.getAbsolutePath());
        }
        return reportOutputDirectory;
    }

    @Override
    public void setReportOutputDirectory(File reportOutputDirectory) {
        this.reportOutputDirectory = reportOutputDirectory;
    }

    @Override
    public boolean isExternalReport() {
        return false;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// Others
    ///////////////////////////////////////////////////////////////////////////

    protected Sink getSink() {
        return sink;
    }

    protected ResourceBundle getBundle(Locale locale) {
        return ResourceBundle.getBundle("redmine-report", locale, getClass().getClassLoader());
    }

    protected File getSkinArtifactFile()
            throws MojoExecutionException {
        Skin skin = Skin.getDefaultSkin();

        String version = skin.getVersion();
        Artifact artifact;
        try {
            if (version == null) {
                version = Artifact.RELEASE_VERSION;
            }
            VersionRange versionSpec = VersionRange.createFromVersionSpec(version);
            artifact = factory.createDependencyArtifact(skin.getGroupId(), skin.getArtifactId(), versionSpec, "jar",
                                                        null, null);

            resolver.resolve(artifact, project.getRemoteArtifactRepositories(), localRepository);
        } catch (InvalidVersionSpecificationException e) {
            throw new MojoExecutionException("The skin version '" + version + "' is not valid: " + e.getMessage());
        } catch (ArtifactResolutionException e) {
            throw new MojoExecutionException("Unable to find skin", e);
        } catch (ArtifactNotFoundException e) {
            throw new MojoExecutionException("The skin does not exist: " + e.getMessage());
        }

        return artifact.getFile();
    }
}
