/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: LoginMojo.java 319 2012-07-19 06:50:02Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-maven-plugin/src/main/java/org/nuiton/jredmine/plugin/LoginMojo.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import org.apache.maven.model.IssueManagement;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;
import org.nuiton.helper.plugin.ShareServerSecretPlugin;
import org.nuiton.jredmine.RedmineService;
import org.nuiton.jredmine.RedmineServiceConfiguration;
import org.nuiton.jredmine.RedmineServiceImplementor;
import org.nuiton.plugin.AbstractPlugin;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;

import java.net.URL;

/**
 * Obtain login for the redmine server to use.
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @since 1.2.1
 */
@Mojo(name = "login", requiresOnline = true, requiresProject = true)
public class LoginMojo extends AbstractPlugin implements RedmineServiceConfiguration {

    /**
     * Dependance du projet.
     *
     * @since 1.2.1
     */
    @Component
    protected MavenProject project;

    /**
     * The real basedir redmine url.
     * <p/>
     * If no url is given, will use the issue management url.
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.url")
    protected URL url;

    /**
     * Un flag pour activer le mode verbeux.
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.verbose", defaultValue = "${maven.verbose}")
    protected boolean verbose;

    /**
     * Un flag pour verifier le login (effectue une connexion au serveur).
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.checkLogin", defaultValue = "false")
    protected boolean checkLogin;

    /**
     * Un flag pour faire échouer le build si la configuration n'est pas ok.
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.safe", defaultValue = "true")
    protected boolean safe;

    /**
     * Redmine server id to obtain login and password.
     * <p/>
     * The server must be defined in your settings.xml file in servers section.
     *
     * @since 1.2.1
     */
    @Parameter(property = "redmine.serverId", required = true)
    protected String serverId;

//    /**
//     * Un flag pour utiliser le clef API pluto que le login/password.
//     * <p/>
//     * Il faut alors que le serveur contienne en username le clef API (encrypté ou non).
//     *
//     * @since 2.0
//     */
//    @Parameter(property = "redmine.useApiKey", defaultValue = "false")
//    private boolean useApiKey;

    /**
     * Settings.
     *
     * @since 1.2.1
     */
    @Component
    protected Settings settings;

    /**
     * password decypher.
     *
     * @since 1.2.1
     */
    @Component(hint = "helper-maven-plugin")
    protected SecDispatcher sec;

    /**
     * Redmine service.
     *
     * @component
     * @since 1.2.1
     */
    @Component
    protected RedmineService service;

    /**
     * shared instance of delegate plugin to obtain login
     *
     * @since 1.2.1
     */
    protected static ShareServerSecretPlugin plugin;

    private static final String REDMINE_USERNAME = "redmine.username";

    private static final String REDMINE_PASSWORD = "redmine.password";

    private static final String REDMINE_API_KEY = "redmine.apiKey";

    ///////////////////////////////////////////////////////////////////////////
    /// Plugin
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected void init() throws Exception {
        if (plugin == null) {
            plugin = new ShareServerSecretPlugin();
            plugin.setServerId(serverId);
//            if (useApiKey) {
//                plugin.setUsernameOut(REDMINE_API_KEY);
//            } else {
                plugin.setUsernameOut(REDMINE_USERNAME);
                plugin.setPasswordOut(REDMINE_PASSWORD);
//            }
            plugin.setRunOnce(true);

            plugin.setProject(project);
            plugin.setLog(getLog());
            plugin.setVerbose(isVerbose());
            plugin.setSettings(settings);
            plugin.setSec(sec);

            plugin.init();
        }

        if (!checkLogin) {
            // no more thing to do
            return;
        }

        // check issue management

        IssueManagement issueManagement = project.getIssueManagement();

        if (issueManagement == null) {
            throw new MojoExecutionException("No Issue Management set.");
        } else if ((issueManagement.getUrl() == null) || (issueManagement.getUrl().trim().equals(""))) {
            throw new MojoExecutionException("No URL set in Issue Management.");
        } else if ((issueManagement.getSystem() != null) && !(issueManagement.getSystem().equalsIgnoreCase(AbstractRedmineMojo.REDMINE_SYSTEM))) {
            throw new MojoExecutionException("Redmine's Plugin only supports 'redmine' Issue Management system.");
        }

        // prepare Redmine service configuration

        URL url = getRestUrl();

        if (url == null || url.toString().isEmpty()) {

            // no redmine url specified, guess it from issueManagement

            url = new URL(issueManagement.getUrl());

            if (verbose) {
                getLog().info("use the url from issue management : " + url);
            }
        }

        // apply configuration

        setRestUrl(url);

    }

    @Override
    protected boolean checkSkip() {

        return plugin.checkSkip();
    }

    @Override
    protected void doAction() throws Exception {

        getLog().info("Obtain redmine login for serverId '" + serverId + "'");
        // when plugin is executed, then will repass in method checkSkip and then
        // will never execute the plugin :( Should instead open Plugin api to
        // pass to public the doAction method
        plugin.setRunOnce(false);
        plugin.execute();
        plugin.setRunOnce(true);

        if (!checkLogin) {
            // no more thing to do
            return;
        }

        // init Redmine service

        try {

            ((RedmineServiceImplementor) service).init(this);
            if (!((RedmineServiceImplementor) service).isInit()) {
                throw new MojoExecutionException("could not logged to redmine server");
            }

        } catch (Exception e) {
            if (safe) {
                throw e;
            }
//            if (verbose) {
            getLog().error("could not init Redmine service [" + getRestUrl() + "] with user '" + getRestUsername() + "'", e);
//            }
//            return false;
        }

        getLog().info("Connection to redmine server successfull.");
    }


    @Override
    protected void afterExecute() {
        if (service != null) {
            RedmineServiceImplementor i;
            i = ((RedmineServiceImplementor) service);

            if (i.isInit()) {
                try {
                    if (verbose) {
                        getLog().info("<<< Close redmine rest client...");
                    }
                    i.destroy();
                } catch (Exception ex) {
                    getLog().error("could not close redmine client for reason " + ex.getMessage(), ex);
                }
            }
        }
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RestClientConfiguration
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public String getRestPassword() {
        return project.getProperties().getProperty(REDMINE_PASSWORD);
    }

    @Override
    public URL getRestUrl() {
        return url;
    }

    @Override
    public String getRestUsername() {
        return project.getProperties().getProperty(REDMINE_USERNAME);
    }

    @Override
    public void setRestPassword(String restPassword) {
    }

    @Override
    public void setRestUrl(URL restUrl) {
        url = restUrl;
    }

    @Override
    public void setRestUsername(String restUsername) {
    }

    @Override
    public String getEncoding() {
        return null;
    }

    @Override
    public void setEncoding(String encoding) {
    }

    @Override
    public boolean isAnonymous() {
        return false;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
    }

    public boolean isSafe() {
        return safe;
    }

    public void setSafe(boolean safe) {
        this.safe = safe;
    }

    @Override
    public String getApiKey() {
        return project.getProperties().getProperty(REDMINE_API_KEY);
    }

    @Override
    public void setApiKey(String apiKey) {
    }
}
