/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: RedmineRestClient.java 319 2012-07-19 06:50:02Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-client/src/main/java/org/nuiton/jredmine/rest/RedmineRestClient.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.rest;

import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.StatusLine;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.io.rest.RequestFactory;
import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.io.rest.RestRequest;
import org.nuiton.io.rest.RestSession;
import org.nuiton.jredmine.RedmineServiceConfiguration;
import org.nuiton.jredmine.model.ModelHelper;

import java.io.IOException;

/**
 * Implementation of a {@link RestClient} to access a Redmine server via the
 * {@code redmine_rest} rails plugin.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.io.rest.RestClient" role-hint="redmine"
 * @since 1.0.0
 */
public class RedmineRestClient extends RestClient {

    private static final Log log = LogFactory.getLog(RedmineRestClient.class);

    private final RequestFactory requestFactory;

    public RedmineRestClient() {
        requestFactory = new RedmineRequestFactory();
        requestFactory.addDefaultRequests();
    }

    public RedmineRestClient(RestClientConfiguration configuration) {
        super(configuration);
        requestFactory = new RedmineRequestFactory();
        requestFactory.addDefaultRequests();
    }

    @Override
    public RequestFactory getRequestFactory() {
        return requestFactory;
    }

    @Override
    protected void open(RestSession session) throws IOException {

        ping(session);

        if (!configuration.isAnonymous()) {
            login(session);
        }
    }

    @Override
    protected void close(RestSession session) throws IOException {

        if (session != null) {
            try {
                RestRequest request = getRequest(ModelHelper.LOGOUT_REQUEST_NAME);
                session.setOpen(false);
                session.doRequest(request);
            } finally {
                session.close();
            }
        }
    }

    protected void ping(RestSession session) throws IOException {

        try {

            RestRequest request = getRequest(ModelHelper.PING_REQUEST_NAME);
            HttpMethod gm = session.doRequest(request);

            StatusLine sl = gm.getStatusLine();
            int statusCode = sl.getStatusCode();
            if (log.isDebugEnabled()) {
                log.debug("status code " + statusCode + " for " + gm.getPath());
            }

            if (statusCode != HttpStatus.SC_OK) {
                gm.releaseConnection();
                throw new IOException(
                        "Got error code <" + statusCode + ":" +
                        sl.getReasonPhrase() + "> on " + gm.getPath());
            }

            String content = gm.getResponseBodyAsString();

            boolean ok = "ping".equals(content);
            if (!ok) {
                throw new IOException(
                        "can not connect to " + configuration.getRestUrl());
            }
        } catch (IOException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new IOException(
                    "could not ping " + configuration.getRestUrl() +
                    " for reason " + ex.getMessage(), ex);
        }
    }

    protected void login(RestSession session) throws IOException {

        RestRequest request;
        String apiKey = getConfiguration().getApiKey();
        boolean useApiKey = StringUtils.isNotBlank(apiKey);
        if (useApiKey) {

            // use api key to login
            request = getRequest(ModelHelper.LOGIN_BY_API_KEY_REQUEST_NAME,
                                 apiKey);
        } else {
            // use normal login / password
            request = getRequest(ModelHelper.LOGIN_REQUEST_NAME,
                                 configuration.getRestUsername(),
                                 configuration.getRestPassword());
        }

        HttpMethod gm = session.doRequest(request);

        StatusLine sl = gm.getStatusLine();
        int statusCode = sl.getStatusCode();
        if (log.isDebugEnabled()) {
            log.debug("status code " + statusCode + " for " + gm.getPath());
        }

        if (statusCode != HttpStatus.SC_OK) {
            gm.releaseConnection();
            throw new IOException(
                    "Got error code <" + statusCode + ":" +
                    sl.getReasonPhrase() + "> on " + gm.getPath());
        }

        // ok session is logged in

    }

    @Override
    public RedmineServiceConfiguration getConfiguration() {
        return (RedmineServiceConfiguration) super.getConfiguration();
    }
}
