package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXInitialContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.navigation.NavigationTreeModel.NavigationTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JPanel;
import java.awt.Component;

/**
 * Simple {@link NavigationTreeSelectionAdapter} implementation with a {@link jaxx.runtime.swing.CardLayout2} to manage components to
 * associated with tree's nodes.
 * <p/>
 * For each node, the ui associated has a constraints in a cardlayout which is the node context path.
 * <p/>
 * A single container managed by the cardlayout is used to display the components associated with tree's nodes.
 *
 * @author chemit
 */
public abstract class NavigationTreeHandlerWithCardLayout extends NavigationTreeHandler {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(NavigationTreeHandlerWithCardLayout.class);

    /**
     * All components associated with a tree's node is displayed in a single container.
     *
     * @return the containter of components
     */
    protected abstract JPanel getContentContainer();

    /**
     * the cardlayout managing components associated with tree node. The constraints
     * of each component is the node contextPath.
     *
     * @return the layout used to display components associated with tree's nodes.
     */
    protected abstract CardLayout2 getContentLayout();

    public NavigationTreeHandlerWithCardLayout(Class<? extends JAXXObject> defaultUIClass, Class<? extends JAXXAction> defaultUIHandlerClass, JAXXObject context, Strategy strategy) {
        super(defaultUIClass, defaultUIHandlerClass, context, strategy);

        if (getContentContainer() == null) {
            throw new IllegalArgumentException("could not have a null 'contentContainer' in ui " + context);
        }
        if (getContentLayout() == null) {
            throw new IllegalArgumentException("could not have a null 'contentLayout' in ui " + context);
        }
    }

    @Override
    protected Component getCurrentUI() {
        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        return layout.getVisibleComponent(container);
    }

    @Override
    protected Component getUI(NavigationTreeNode node) {
        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        String path = getNodeConstraints(node);
        return layout.contains(path) ? layout.getComponent(container, path) : null;
    }

    @Override
    protected void openUI(Component newUI, NavigationTreeNode node) throws Exception {

        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        // switch layout
        layout.show(container, getNodeConstraints(node));
    }

    @Override
    protected boolean closeUI(Component component) throws Exception {
        // by default, we says that component was succesfull closed
        return true;
    }

    @Override
    protected JAXXContext createUIContext(NavigationTreeNode node) throws Exception {

        if (node.getJaxxActionClass() == null) {
            if (log.isWarnEnabled()) {
                log.warn("no action associated with ui " + node.getJaxxClass());
            }
            // no action associated, just
            return context;
        }

        JAXXAction action = getJAXXAction(node.getJaxxActionClass());

        // init context with
        JAXXInitialContext uiContext = action.init(this.context);
        return uiContext;
    }

    @Override
    protected Component createUI(NavigationTreeNode node) throws Exception {

        JAXXContext uiContext = createUIContext(node);

        JAXXObject newUI = node.getJaxxClass().getConstructor(JAXXContext.class).newInstance(uiContext);

        if (log.isDebugEnabled()) {
            log.debug("instanciate new ui " + newUI);
        }

        getContentContainer().add((Component) newUI, getNodeConstraints(node));
        return (Component) newUI;
    }
}

