package jaxx.runtime.swing;

import java.awt.Component;
import jaxx.runtime.JXPathDecorator;
import jaxx.runtime.MultiJXPathDecorator;
import jaxx.runtime.Util;
import jaxx.runtime.SwingUtil;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;

import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JToggleButton;
import javax.swing.SwingUtilities;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import java.awt.Dimension;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.beans.Introspector;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.List;
import javax.swing.JComboBox;
import jaxx.runtime.Decorator;
import org.jdesktop.swingx.autocomplete.AutoCompletePropertyChangeListener;

/**
 * Le handler d'un {@link EntityComboBox}.
 * <p/>
 * Note: ce handler n'est pas staeless et n'est donc pas partageable entre plusieurs ui.
 *
 * @param <O> le type des objet contenus dans le modèle du composant.
 * 
 * @author chemit
 * @see EntityComboBox
 */
public class EntityComboBoxHandler<O> implements PropertyChangeListener {

    public static final Log log = LogFactory.getLog(EntityComboBoxHandler.class);
    public static final String SELECTED_ITEM_PROPERTY = "selectedItem";
    public static final String INDEX_PROPERTY = "index";
    public static final String AUTO_COMPLETE_PROPERTY = "autoComplete";
    public static final String DATA_PROPERTY = "data";
    /** ui if the handler */
    protected EntityComboBox ui;
    /** the mutator method on the property of boxed bean in the ui */
    protected Method mutator;
    /** the original document of the combbo box editor (keep it to make possible undecorate) */
    protected Document originalDocument;
    /** the convertor used to auto-complete */
    protected ObjectToStringConverter convertor;
    /** the decorator of data */
    protected MultiJXPathDecorator<O> decorator;
    protected boolean init;

    public EntityComboBoxHandler(EntityComboBox ui) {
        this.ui = ui;
    }
    protected final FocusListener EDITOR_TEXT_COMP0NENT_FOCUSLISTENER = new FocusListener() {

        @Override
        public void focusGained(FocusEvent e) {
            if (log.isDebugEnabled()) {
                log.debug("close popup from " + e);
            }
            ui.getPopup().setVisible(false);
        }

        @Override
        public void focusLost(FocusEvent e) {
        }
    };

    /**
     * Initialise le handler de l'ui
     *
     * @param decorator le decorateur a utiliser
     * @param data      la liste des données a gérer
     */
    public void init(JXPathDecorator<O> decorator, List<O> data) {

        if (init) {
            throw new IllegalStateException("can not init the handler twice");
        }
        init = true;
        if (decorator == null) {
            throw new NullPointerException("can not have a null decorator as parameter");
        }

        JAXXButtonGroup indexes = ui.getIndexes();

        MultiJXPathDecorator<O> d;
        if (decorator instanceof MultiJXPathDecorator<?>) {
            // should clone decorator ?
            d = (MultiJXPathDecorator<O>) decorator;
        } else {
            d = MultiJXPathDecorator.newDecorator(decorator.getInternalClass(), decorator.getInitialExpression(), " - ");
        }
        this.decorator = d;

        // init combobox renderer base on given decorator
        ui.getCombobox().setRenderer(Util.newDecoratedListCellRenderer(d));

        this.convertor = newDecoratedObjectToStringConverter(d);

        // keep a trace of original document (to make possible reverse autom-complete) 
        JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
        this.originalDocument = editorComponent.getDocument();

        // build popup
        preparePopup(d);

        ui.autoComplete = true;

        ui.addPropertyChangeListener(this);

        // set datas
        ui.setData(data);

        // select sort button
        indexes.setSelectedButton(ui.getIndex());
    }

    /** Toggle the popup visible state. */
    public void togglePopup() {
        boolean newValue = !ui.getPopup().isVisible();

        if (log.isTraceEnabled()) {
            log.trace(newValue);
        }

        if (!newValue) {
            if (ui.getPopup() != null) {
                ui.getPopup().setVisible(false);
            }
            return;
        }
        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                ui.getPopup().pack();
                Dimension dim = ui.getPopup().getPreferredSize();
                JToggleButton invoker = ui.getChangeDecorator();
                ui.getPopup().show(invoker, (int) (invoker.getPreferredSize().getWidth() - dim.getWidth()), invoker.getHeight());
            }
        });
    }

    /**
     * Modifie l'état autoComplete de l'ui.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setAutoComplete(Boolean oldValue, Boolean newValue) {
        oldValue = oldValue != null && oldValue;
        newValue = newValue != null && newValue;
        if (oldValue == newValue) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("autocomplete state : <" + oldValue + " to " + newValue + ">");
        }
        if (!newValue) {
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.removeFocusListener(EDITOR_TEXT_COMP0NENT_FOCUSLISTENER);
            undecorate(ui.getCombobox(), originalDocument);
        } else {
            decorate(ui.getCombobox(), convertor);
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.addFocusListener(EDITOR_TEXT_COMP0NENT_FOCUSLISTENER);
        }
    }

    /**
     * Modifie l'index du décorateur
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    @SuppressWarnings({"unchecked"})
    protected void setIndex(Integer oldValue, Integer newValue) {
        if (newValue.equals(oldValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("check state : <" + oldValue + " to " + newValue + ">");
        }

        // change decorator context
        decorator.setContextIndex(newValue);

        // keep selected item
        Object previousSelectedItem = ui.getSelectedItem();
        Boolean wasAutoComplete = ui.isAutoComplete();

        if (wasAutoComplete) {
            ui.setAutoComplete(false);
        }

        // remove autocomplete
        if (previousSelectedItem != null) {
            ui.getCombobox().setSelectedItem(null);
            ui.selectedItem = null;
        }


        try {
            // Sort data with the decorator jxpath tokens.
            JXPathDecorator.sort(decorator, ui.getData(), newValue);
        } catch (Exception e) {
            log.warn(e.getMessage(), e);
            //System.out.println("newValue :: "+decorator+" : "+newValue);
            //System.out.println("datas :: "+ui.getData());
        }

        // reload the model
        SwingUtil.fillComboBox(ui.getCombobox(), ui.getData(), null);

        if (wasAutoComplete) {
            ui.setAutoComplete(true);
        }

        if (previousSelectedItem != null) {
            ui.setSelectedItem(previousSelectedItem);
        }

        ui.getCombobox().requestFocus();
    }

    /**
     * Modifie la valeur sélectionnée dans la liste déroulante.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setSelectedItem(Object oldValue, Object newValue) {
        if (ui.getBean() == null) {
            return;
        }

        if (newValue == null) {
            if (ui.getCombobox().getSelectedItem() == null) {
                return;
            }
            ui.getCombobox().setSelectedItem(null);

            if (ui.isAutoComplete()) {
                ui.setAutoComplete(false);
                ui.setAutoComplete(true);
            }

            if (oldValue == null) {
                return;
            }
        }
        if (log.isDebugEnabled()) {
            log.debug(ui.getProperty() + " on " + ui.getBean().getClass() + " :: " + oldValue + " to " + newValue);
        }

        try {
            Method mut = getMutator();
            if (mut != null) {
                mut.invoke(ui.getBean(), newValue);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /** @return le document de l'éditeur avant complétion. */
    public Document getOriginalDocument() {
        return originalDocument;
    }

    public MultiJXPathDecorator<O> getDecorator() {
        return decorator;
    }

    /**
     * Creation de l'ui pour modifier le décorateur.
     *
     * @param decorator le decorateur a utiliser
     */
    protected void preparePopup(MultiJXPathDecorator<?> decorator) {
        String selectedTip = ui.getSelectedToolTipText();
        if (selectedTip == null) {
            // use default selected tip text
            selectedTip = EntityComboBox.DEFAULT_SELECTED_TOOLTIP;
        }
        String notSelectedTip = ui.getNotSelectedToolTipText();
        if (notSelectedTip == null) {
            // use default selected tip text
            notSelectedTip = EntityComboBox.DEFAULT_NOT_SELECTED_TOOLTIP;
        }
        JPopupMenu popup = ui.getPopup();

        //Container container = ui.getIndexesContainer();
        for (int i = 0, max = decorator.getNbContext(); i < max; i++) {
            String property = ui.getI18nPrefix() + decorator.getProperty(i);
//            String property = "observe.common." + decorator.getProperty(i);
            String propertyI18n = _(property);
            JRadioButtonMenuItem button = new JRadioButtonMenuItem(propertyI18n);
            button.putClientProperty(JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY, ui.getIndexes());
            button.putClientProperty(JAXXButtonGroup.VALUE_CLIENT_PROPERTY, i);
            popup.add(button);
            if (selectedTip != null) {
                button.putClientProperty(JAXXButtonGroup.SELECTED_TIP_CLIENT_PROPERTY, _(selectedTip, propertyI18n));
            }
            if (notSelectedTip != null) {
                button.putClientProperty(JAXXButtonGroup.NOT_SELECTED_TIP_CLIENT_PROPERTY, _(notSelectedTip, propertyI18n));
            }
            button.setSelected(false);
            ui.getIndexes().add(button);
        }
        String title = ui.getPopupTitleText();
        if (title == null) {
            // use default popup title
            title = EntityComboBox.DEFAULT_POPUP_LABEL;

            Class<?> internalClass = decorator.getInternalClass();
            String beanI18nKey;
            if (internalClass == null) {
                beanI18nKey = n_("entitycombobox.unknown.type");
            } else {
                beanI18nKey = ui.getI18nPrefix() + Introspector.decapitalize(internalClass.getSimpleName());
                //beanI18nKey = "observe.common." + Introspector.decapitalize(internalClass.getSimpleName());
            }
            String beanI18n = _(beanI18nKey);
            title = _(title, beanI18n);
        } else {
            title = _(title);
        }
        ui.getPopupLabel().setText(title);
        ui.getPopup().setLabel(title);
        ui.getPopup().invalidate();
    }

    public Class<?> getTargetClass() {
        Method m = getMutator();
        return m == null ? null : m.getParameterTypes()[0];
    }

    /** @return le mutateur a utiliser pour modifier le bean associé. */
    protected Method getMutator() {
        if (mutator == null) {
            Object bean = ui.getBean();
            if (bean == null) {
                throw new NullPointerException("could not find bean in " + ui);
            }
            String property = ui.getProperty();
            if (property == null) {
                throw new NullPointerException("could not find property in " + ui);
            }

            try {
                PropertyDescriptor descriptor = PropertyUtils.getPropertyDescriptor(bean, property);
                if (descriptor != null) {
                    mutator = descriptor.getWriteMethod();
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return mutator;
    }

    /**
     * Encapsule un {@link Decorator} dans un {@link ObjectToStringConverter}.
     *
     * @param decorator le decorateur a encapsuler.
     * @return le converter encapsule dans un {@link ObjectToStringConverter}
     */
    public static ObjectToStringConverter newDecoratedObjectToStringConverter(final Decorator<?> decorator) {

        return new ObjectToStringConverter() {

            @Override
            public String getPreferredStringForItem(Object item) {
                return item instanceof String ? (String) item : (item == null ? "" : decorator.toString(item));
            }
        };
    }

    /**
     * Ajout l'auto-complétion sur une liste déroulante, en utilisant le
     * converteur donné pour afficher les données.
     *
     * @param combo la combo à décorer
     * @param convertor le converter utilisé pour afficher les données.
     */
    public static void decorate(JComboBox combo, ObjectToStringConverter convertor) {

        org.jdesktop.swingx.autocomplete.AutoCompleteDecorator.decorate(combo, convertor);
    }

    /**
     * Désactive l'aut-complétion sur une liste déroulante, en y repositionnant
     * le modèle du document d'édition d'avant auto-complétion.
     *
     * @param combo la liste déroulante à décorer
     * @param originalDocument le document original de l'édtieur de la
     *        liste déroulante.
     */
    public static void undecorate(JComboBox combo, Document originalDocument) {

        // has not to be editable
        combo.setEditable(false);

        // configure the text component=editor component
        Component c = combo.getEditor().getEditorComponent();
        JTextComponent editorComponent = (JTextComponent) c;
        editorComponent.setDocument(originalDocument);
        editorComponent.setText(null);
        //remove old property change listener
        for (PropertyChangeListener l : c.getPropertyChangeListeners("editor")) {
            if (l instanceof AutoCompletePropertyChangeListener) {
                c.removePropertyChangeListener("editor", l);
            }
        }
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        String propertyName = evt.getPropertyName();

        if (SELECTED_ITEM_PROPERTY.equals(propertyName)) {
            setSelectedItem(evt.getOldValue(), evt.getNewValue());
            return;
        }

        if (INDEX_PROPERTY.equals(propertyName)) {
            setIndex((Integer) evt.getOldValue(), (Integer) evt.getNewValue());
            return;
        }
        if (AUTO_COMPLETE_PROPERTY.equals(propertyName)) {
            setAutoComplete((Boolean) evt.getOldValue(), (Boolean) evt.getNewValue());
            return;
        }
        if (DATA_PROPERTY.equals(propertyName)) {
            // list has changed, force reload of index
            setIndex(-1, ui.getIndex());
        }

    }
}
