/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.services.ServiceList;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollAction;
import org.chorem.pollen.ui.data.PollStep;
import org.chorem.pollen.ui.services.Configuration;

/**
 * Classe de la page de modification d'un sondage.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: PollModification.java 2705 2009-08-13 16:17:33Z nrannou $
 */
@IncludeStylesheet("context:css/pollCreation.css")
public class PollModification {

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Paramètres de la page */
    private String param;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /** Étape courante du formulaire */
    @Persist
    private PollStep step;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SuppressWarnings("unused")
    @SessionState
    private UserDTO user;
    @SuppressWarnings("unused")
    @Property
    private boolean userExists;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    @Property
    private String siteURL;

    /** Composants DateTimeField pour le début des ajouts de choix */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginChoiceDate;

    /** Composants DateTimeField pour le début et la fin du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginDate;
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField endDate;

    /** Format des dates */
    @SuppressWarnings("unused")
    @Persist
    @Property
    private DateFormat dateFormat;

    /** Locale courante */
    @Inject
    private Locale currentLocale;

    /**
     * Sondage
     */
    @Property
    @Persist
    private PollDTO poll;

    /**
     * Créateur du sondage
     */
    @Property
    @Persist
    private PollAccountDTO creator;

    /** Indique si l'utilisateur est authorisé */
    @Property
    private boolean userAllowed = false;

    /** Date de début du sondage (utilisée pour la validation). */
    private Date beginDateValidation;

    @InjectComponent
    private Zone pollCreationZone;

    @InjectPage
    private ConfirmPoll confirmPoll;

    /** variable utilisée pour l'affichage de la notificationCheckBox */
    @Property
    @Persist
    private boolean notificationCheckBox;

    /** Règle de notification */
    @Property
    @Persist
    private PreventRuleDTO newRule;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des votants
     */
    @Property
    private VotingListDTO votingList;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des votants
     */
    @SuppressWarnings("unused")
    @Property
    private PollAccountDTO votingListPerson;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @Property
    private ChoiceDTO choice;

    @InjectPage
    private ModificationValidation modificationValidation;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceList serviceList;

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape suivante de la
     * modification de sondage.
     */
    Object onSuccessFromPollCreationForm() {
        switch (step) {
        case POLL:
            step = PollStep.OPTIONS;
            break;
        case OPTIONS:
            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }

            if (isFreePoll()) {
                step = PollStep.CHOICES;
            } else {
                step = PollStep.LISTS;
            }
            break;
        case LISTS:
            step = PollStep.CHOICES;
            break;
        case CHOICES:

            // Règles de notification
            PreventRuleDTO oldRule = null;
            for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
                if (rule.getId().equals(newRule.getId())) {
                    oldRule = rule;
                }
            }
            if (oldRule != null) { // remplacement de oldRule par newRule
                poll.getPreventRuleDTOs().remove(oldRule);
                if (notificationCheckBox) {
                    poll.getPreventRuleDTOs().add(newRule);
                }
            } else { // création d'une nouvelle règle
                if (notificationCheckBox) {
                    poll.getPreventRuleDTOs().add(newRule);
                }
            }

            // sauvegarde des comptes modifiés
            List<PollAccountDTO> modifiedAccounts = getModifiedAccounts();

            // mise à jour des listes de votants du sondage
            servicePollAccount.updatePollAccount(creator);
            for (VotingListDTO votingList : poll.getVotingListDTOs()) {
                for (PollAccountDTO account : votingList.getPollAccountDTOs()) {
                    servicePollAccount.updatePollAccount(account);
                }
                serviceList.updateVotingList(votingList);
            }

            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }

            for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                choice.setValidate(true);
            }
            servicePoll.updatePoll(poll);

            // envoi de mails aux comptes modifiés
            sendMailNotification(modifiedAccounts);

            step = PollStep.POLL;
            modificationValidation.setPoll(poll);
            return modificationValidation;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée que le formulaire soit valide ou non. Il est nécessaire
     * de la redéfinir pour qu'en cas d'erreur de validation, la zone soit tout
     * de même mise à jour pour afficher l'erreur.
     */
    Object onSubmitFromPollCreationForm() {
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape précédente de la
     * création de sondage.
     */
    Object onPrevious() {
        switch (step) {
        case OPTIONS:
            step = PollStep.POLL;
            break;
        case LISTS:
            step = PollStep.OPTIONS;
            break;
        case CHOICES:
            if (isFreePoll()) {
                step = PollStep.OPTIONS;
            } else {
                step = PollStep.LISTS;
            }
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * endDate.
     *
     * @throws ValidationException
     */
    void onValidateFromEndDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.before(beginDateValidation)) {
            throw new ValidationException(messages.get("endDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginChoiceDate.
     *
     * @throws ValidationException
     */
    void onValidateFromBeginChoiceDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.after(beginDateValidation)) {
            throw new ValidationException(messages
                    .get("beginChoiceDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation des
     * listes de votants.
     *
     * @throws ValidationException
     */
    void onValidateFormFromPollCreationForm() throws ValidationException {

        // Validation des votants
        if (step == PollStep.LISTS) {
            List<VotingListDTO> votingLists = poll.getVotingListDTOs();
            int nbListEqual = 0;
            int nbEqual = 0;
            int nbNotNull = 0;

            // Repérage des doublons (listes)
            for (VotingListDTO list1 : votingLists) {
                for (VotingListDTO list2 : votingLists) {
                    if (list1.getName().equals(list2.getName())) {
                        nbListEqual++;
                    }
                }
            }

            if (nbListEqual > votingLists.size()) {
                throw new ValidationException(messages.get("lists-validate"));
            }

            // Repérage des doublons (votants)
            for (VotingListDTO list1 : votingLists) {
                int nbLocalNotNull = 0;
                for (PollAccountDTO account1 : list1.getPollAccountDTOs()) {
                    if (account1.getVotingId() != null
                            && account1.getVotingId() != "") {
                        nbNotNull++;
                        nbLocalNotNull++;

                        // comparaison avec les autres votants
                        for (VotingListDTO list2 : votingLists) {
                            for (PollAccountDTO account2 : list2
                                    .getPollAccountDTOs()) {
                                if (account2.getVotingId() != null
                                        && account1.getVotingId() != "") {
                                    if (account1.getVotingId().equals(
                                            account2.getVotingId())) {
                                        nbEqual++;
                                    }
                                }
                            }
                        }
                    }
                }
                if (nbLocalNotNull == 0) {
                    throw new ValidationException(messages
                            .get("noList-validate"));
                }
            }

            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("list-validate"));
            }
        }
    }

    Object onActionFromCloseButton() {
        String backPage = "poll/modification";
        confirmPoll.setBackPage(backPage);
        confirmPoll.setAction(PollAction.CLOSE);
        confirmPoll.setPoll(poll);
        return confirmPoll;
    }

    public String getChoiceDateDisplay() {
        return poll.isChoiceAddAllowed() ? "display: block;" : "display: none;";
    }

    public String getNotificationDisplay() {
        return notificationCheckBox ? "display: block;" : "display: none;";
    }

    /** Retourne la date correspondant au choix courant */
    public Date getChoiceNameAsDate() {
        return new Date(Long.valueOf(choice.getName()));
    }

    /** Retourne le poids en entier de la liste courante */
    public int getWeightAsInt() {
        return (int) votingList.getWeight();
    }

    /** Retourne le numéro de la liste courante (index+1) */
    public int getVotingListNumber() {
        return poll.getVotingListDTOs().indexOf(votingList) + 1;
    }

    /** Retourne la chaîne correspondant à l'étape courante */
    public String getStepLegend() {
        Integer index = step.getIndex();

        // corrections selon le type de sondage
        if (poll.getPollType() == PollType.FREE) {
            if (step == PollStep.CHOICES) {
                index--;
            }
        }

        // mise en forme du message
        switch (step) {
        case POLL:
            return messages.format("pollLegend", index);
        case OPTIONS:
            return messages.format("optionsLegend", index);
        case LISTS:
            return messages.format("listsLegend", index);
        case CHOICES:
            return messages.format("choicesLegend", index);
        default:
            return "";
        }
    }

    public boolean isInPoll() {
        return step == PollStep.POLL;
    }

    public boolean isInLists() {
        return step == PollStep.LISTS;
    }

    public boolean isInChoices() {
        return step == PollStep.CHOICES;
    }

    public boolean isInOptions() {
        return step == PollStep.OPTIONS;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextChoices() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateChoices() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImgChoices() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    /** Envoi du mail de notification */
    private void sendMailNotification(List<PollAccountDTO> modifiedAccounts) {
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mails aux votants
        for (PollAccountDTO account : modifiedAccounts) {
            if (account.getEmail() != null) {
                String accountVoteURL = voteURL + ":" + account.getAccountId();

                data.put("to", account.getEmail());
                data.put("title", messages.format("votingEmail_subject", poll
                        .getTitle()));
                data.put("msg", messages.format("votingEmail_msg", poll
                        .getTitle(), account.getVotingId(), accountVoteURL));

                PreventRuleManager.emailAction(data);
            }
        }
    }

    /** Récupération des votants modifiés */
    private List<PollAccountDTO> getModifiedAccounts() {
        List<PollAccountDTO> modifiedAccounts = new ArrayList<PollAccountDTO>();

        // Parcours des listes de votants du sondage et comparaison
        // avec les listes de votants de la base de données
        for (VotingListDTO list : poll.getVotingListDTOs()) {
            for (PollAccountDTO newAccount : list.getPollAccountDTOs()) {
                PollAccountDTO oldAccount = servicePollAccount
                        .findPollAccountById(newAccount.getId());

                if (oldAccount.getEmail() == null
                        && newAccount.getEmail() != null) {
                    modifiedAccounts.add(newAccount);
                } else if (oldAccount.getEmail() != null
                        && newAccount.getEmail() != null) {
                    if (!newAccount.getVotingId().equals(
                            oldAccount.getVotingId())
                            || !newAccount.getEmail().equals(
                                    oldAccount.getEmail())) {
                        modifiedAccounts.add(newAccount);
                    }
                }
            }
        }

        return modifiedAccounts;
    }

    void onActivate(String id) {
        param = id;

        if (id != null && !"".equals(id)) {
            String pollId = id.split(":", 2)[0];

            // Réinitialisation des variables de session
            if (poll == null || !pollId.equals(poll.getPollId())) {
                poll = servicePoll.findPollByPollId(pollId);
                creator = null;
                newRule = null;
            }

            if (poll != null) {

                // Identification de l'utilisateur
                if (creator == null) {
                    creator = servicePollAccount.findPollAccountById(poll
                            .getCreatorId());
                }
                if (id.split(":", 2).length == 2) {
                    String creatorId = id.split(":", 2)[1];
                    if (creatorId.equals(creator.getAccountId())) {
                        userAllowed = true;
                    }
                }

                // Règles de notification
                if (newRule == null) {
                    initNotificationRule();
                }
            }
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else if (poll.isClosed()) {
            feedback.addError(messages.get("pollClosed"));
        } else if (!userAllowed) {
            feedback.addError(messages.get("userNotAllowed"));
        }
    }

    /**
     * Méthode appelée au moment de la désactivation de la page
     *
     * @return l'identifiant du sondage et du créateur
     */
    String onPassivate() {
        return param;
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
        dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                DateFormat.SHORT, currentLocale);

        // Réinitialisation des variables de session
        if (poll != null) {
            poll = servicePoll.findPollByPollId(poll.getPollId());
            creator = servicePollAccount.findPollAccountById(poll
                    .getCreatorId());
            initNotificationRule();
        }
        step = PollStep.POLL;
    }

    /** Initialisation de la règle de notification */
    private void initNotificationRule() {
        notificationCheckBox = false;
        newRule = new PreventRuleDTO("vote", 0, true,
                PreventRuleManager.EMAIL_ACTION);
        newRule.setPollId(poll.getId());
        for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
            if ("vote".equals(rule.getScope())) {
                notificationCheckBox = true;
                newRule = rule;
            }
        }
    }
}
