/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import javax.activation.MimetypesFileTypeMap;

import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.dom.Element;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.Response;
import org.slf4j.Logger;

/**
 * Composant qui affiche une image.
 *
 * Ce composant permet d'afficher une image dynamique (exemple : image uploadée
 * sur le serveur par un formulaire).
 *
 * <pre>
 * Utilisation :
 * &lt;t:image src=&quot;/img/image.png&quot; /&gt;
 * Pour afficher une image statique il est préférable d'utiliser :
 * &lt;img src=&quot;${asset:context:img}/image.png&quot; /&gt;
 * </pre>
 *
 * @author rannou
 * @version $Id: Image.java 2736 2009-08-19 15:48:01Z nrannou $
 */
public class Image {

    /** Chemin de l'image */
    @Parameter(required = true)
    private String _src;

    /** Texte alternatif */
    @Parameter
    private String _alt;

    /** Largeur de l'image */
    @Parameter
    private int _width;

    /** Hauteur de l'image */
    @Parameter
    private int _height;

    @Inject
    private Logger logger;

    @Inject
    private ComponentResources _resources;

    void beginRender(MarkupWriter writer) {
        Link link = _resources.createEventLink("imageLink", _src, _width,
                _height);
        Element img = writer.element("img", "src", link, "alt", _alt);

        _resources.renderInformalParameters(writer);
    }

    void afterRender(MarkupWriter writer) {
        writer.end();
    }

    /** Affichage de l'image */
    public StreamResponse onImageLink(final String src, final int width,
            final int height) {
        if (src == null || "".equals(src)) {
            return null;
        }

        File file = new File(src);
        FileInputStream stream = null;
        try {
            stream = new FileInputStream(file);
        } catch (FileNotFoundException e) {
            logger.error("Fichier inexistant : " + file.getAbsolutePath(), e);
        }

        MimetypesFileTypeMap mimes = new MimetypesFileTypeMap();
        String type = mimes.getContentType(file);

        return new ImageStreamResponse(stream, type);
    }

    /** Création du flux de l'image */
    class ImageStreamResponse implements StreamResponse {
        protected String contentType;
        protected InputStream stream;

        public ImageStreamResponse(InputStream stream, String contentType) {
            this.stream = stream;
            this.contentType = contentType;
        }

        public String getContentType() {
            return contentType;
        }

        public InputStream getStream() throws IOException {
            return stream;
        }

        public void prepareResponse(Response arg0) {

        }
    }
}