/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: AbstractVoteAction.java 3652 2012-08-28 16:30:53Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/vote/AbstractVoteAction.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll.vote;

import com.google.common.base.Charsets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollCommentVisibility;
import org.chorem.pollen.business.persistence.PollVoteVisibility;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.ui.PollenUIUtils;
import org.chorem.pollen.ui.actions.PageSkin;
import org.chorem.pollen.ui.actions.PollUriAware;
import org.chorem.pollen.ui.actions.PollenActionSupport;
import org.chorem.pollen.ui.actions.PollenUserSecurityAware;
import org.chorem.pollen.votecounting.strategy.VoteCountingStrategy;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;

/**
 * Abstract action for actions on the vote poll page.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @since 1.3
 */
public abstract class AbstractVoteAction extends PollenActionSupport implements ParameterAware, PollUriAware, PollenUserSecurityAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractVoteAction.class);

    public static final String PREPARE_VOTE_PAGE = "prepareVotePage";

    /**
     * Loaded poll.
     *
     * @since 1.3
     */
    private Poll poll;

    /**
     * Is feed exists for this poll ?
     *
     * @since 1.3
     */
    private boolean feedFileExisting;

    /**
     * Loaded poll account (from the pollUri).
     *
     * @since 1.3
     */
    private PollAccount pollAccount;

    /**
     * All votes for the poll.
     *
     * @since 1.3
     */
    private List<Vote> votes;

    /**
     * Current vote to treat (for create/updatevote action).
     *
     * @since 1.3
     */
    private Vote vote;

    /**
     * Comment author to display in UI.
     *
     * @since 1.3
     */
    private String commentAuthor;

    /**
     * Results of the poll.
     *
     * @since 1.3
     */
    private List<PollResult> results;

    /**
     * List of comments to display.
     *
     * @since 1.3
     */
    private List<Comment> comments;

    protected AbstractVoteAction() {
        super(PageSkin.VOTE);
    }

    /**
     * @return {@code true} if moderation is possible, {@code false} otherwise
     * @since 1.4
     */
    public abstract boolean isModerate();

    public Poll getPoll() {
        return poll;
    }

    public PollAccount getPollAccount() {
        return pollAccount;
    }

    public List<Vote> getVotes() {
        return votes;
    }

    public Vote getVote() {
        return vote;
    }

    public List<PollResult> getResults() {
        return results;
    }

    public List<Comment> getComments() {
        return comments;
    }

    public String getVoteSizeMessage() {
        String result = PollenUIUtils.getVoteSizeMessage(getPoll(),
                                                         getLocale());
        return result;
    }

    public String getVoteCountingTypeName() {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        String result = strategy.getStrategyName(getLocale());
        return result;
    }

    public String getVoteCountingTypeHelp() {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        String result = strategy.getStrategyHelp(getLocale());
        return result;
    }

    public String getPollVoteVisibilityName() {
        PollVoteVisibility strategy = getPoll().getPollVoteVisibility();
        String result = _(strategy.getI18nKey());
        return result;
    }

    public String getPollVoteVisibilityHelp() {
        PollVoteVisibility strategy = getPoll().getPollVoteVisibility();
        String result = _(strategy.getI18nHelpKey());
        return result;
    }

    /**
     * Is comment can be displayed (and then added) on this poll.
     *
     * @return {@code true} if comments can be displayed and then added or
     *         removed for this poll.
     * @since 1.4.5
     */
    public boolean isCommentAllowed() {
        PollCommentVisibility pollCommentVisibility =
                getPoll().getPollCommentVisibility();
        return pollCommentVisibility == PollCommentVisibility.EVERYBODY;
    }

    /**
     * Retourne si le choix fait partie du vote (s'il a été renseigné par le
     * votant).
     *
     * @param choice le choix concerné
     * @return true si le choix est dans le vote
     */
    public boolean isChoiceInVote(VoteToChoice choice) {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        boolean result = strategy.isChoiceInVote(choice.getVoteValue());
        return result;
//        Integer voteValue = choice.getVoteValue();
//        boolean result = false;
//        if (choice != null) {
//            switch (poll.getVoteCountingType()) {
//                case NORMAL:
//                    result = voteValue > 0;
//                    break;
//                case PERCENTAGE:
//                    result = true;
//                    break;
//                case CONDORCET:
//                    result = voteValue < 100;
//                    break;
//                case NUMBER:
//                    result = voteValue >= 0;
//            }
//        }
//        return result;
    }

    public String getChoiceValue(VoteToChoice choice) {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        String result = strategy.getDisplayVoteValue(choice.getVoteValue());
        return result;
//        switch (poll.getVoteCountingType()) {
//            case NORMAL:
//                result = "OK";
//                break;
//            default:
//                result = choice.getVoteValue() + "";
//        }
//        return result;
    }

    public String getCommentAuthor() {
        if (commentAuthor == null) {

            UserAccount user = getUserSecurityContext().getUserAccount();
            if (user == null) {

                // Use current pollAccount name for comment
                commentAuthor = getPollAccount().getVotingId();

            } else {

                // Connecter user name
                commentAuthor = user.getDisplayName();
            }
        }
        return commentAuthor;
    }

    public String getSummaryUrl() {
        PollUrl url = getPollUrlService().getPollSummaryUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        return url.getUrl();
    }

    public String getResultUrl() {
        PollUrl url = getPollUrlService().getPollResultUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        getUserSecurityContext().removeAccountIdWhenConnected(url);
        return url.getUrl();
    }

    public String getVoteMessages() {
        return _("pollen.common.voteNbVotes", poll.sizeVote());
    }

    public boolean isFeedFileExisting() {
        return feedFileExisting;
    }

    public boolean isCreatorOrAdmin() {
        return getUserSecurityContext().isCreator() ||
               getUserSecurityContext().isAdmin();
    }

    //FIXME-tchemit-2012-08-27, why ? in none free poll can not see votingId ?
    //FIXME-tchemit-2012-08-27 ...hum weird, just in anonymous vote (method will be removed after check...)
    public boolean isAccountFieldDisplayed() {
        return !poll.isAnonymous() || isRestrictedPoll() || isGroupPoll();
    }

    public boolean isPollChoiceOrVoteStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceStarted(now) || poll.isStarted(now);
    }

    public boolean isAnonymousVote() {
        PollVoteVisibility pollVoteVisibility = poll.getPollVoteVisibility();
        return pollVoteVisibility == PollVoteVisibility.NOBODY;
    }

    public boolean isPollChoiceRunning() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceRunning(now);
    }

    public boolean isPollStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.getBeginDate() == null || poll.getBeginDate().before(now);
    }

    public boolean isPollFinished() {
        Date now = serviceContext.getCurrentTime();
        return poll.getEndDate() != null && poll.getEndDate().before(now);
    }

    public boolean isDescNull(Choice choice) {
        return StringUtils.isEmpty(choice.getDescription());
    }

    public boolean isFreePoll() {
        return poll.isPollFree();
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isVoteAllowed() {
        return !isModerate() && getSecurityService().isCanVote(getUserSecurityContext());
    }

    public boolean isResultAllowed() {
        return getSecurityService().isCanAccessResult(getUserSecurityContext());
    }

    public String getPollCreatorName() {
        PollAccount creator = poll.getCreator();
        String result = creator.getVotingId();
        if (StringUtils.isBlank(result)) {
            result = creator.getEmail();
        }
        if (StringUtils.isBlank(result)) {
            result = _("pollen.common.undefined");
        }
        return result;
    }

    public String getPollBeginDate() {
        Date date = poll.getBeginDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public String getPollEndDate() {
        Date date = poll.getEndDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public boolean isModifyVoteAllowed(Vote vote) {
        return getSecurityService().isCanModifyVote(getUserSecurityContext(),
                                                    vote.getTopiaId());
    }

    public boolean isDeleteCommentAllowed(Comment comment) {
        return getSecurityService().isCanDeleteComment(getUserSecurityContext(),
                                                       comment);
    }

    public boolean isDeleteVoteAllowed(Vote vote) {
        return getSecurityService().isCanDeleteVote(getUserSecurityContext(),
                                                    vote.getTopiaId());
    }

    public String getResultValue(Choice choice) {

        String val = getPollResultsService().getResultValue(choice, results);
        return val;
    }

    public String getChoiceFragment() {
        VoteCountingStrategy strategy = getVoteCountingStrategy(poll);
        String result =
                "displayVote_" + strategy.getVoteValueEditorType().name() + ".jsp";
        return result;
    }

    public Date getChoiceAsDate(Choice choice) {
        return new Date(Long.valueOf(choice.getName()));
    }

    public void setCommentAuthor(String commentAuthor) {
        this.commentAuthor = commentAuthor;
    }

    public String prepareVotePage() throws Exception {

        // load poll
        loadPoll();

        // Current poll account
        loadPollAccount();

        // Get all votes
        // TODO no pagination for the moment, need to retrieve the correct page depends on current pollAccount
        List<Vote> allVotes = getVoteService().getAllVotes(poll);

        // only keep possible votes for current user security context
        votes = getSecurityService().filterVotes(poll,
                                                 allVotes,
                                                 getUserSecurityContext());

        if (isVoteAllowed()) {

            // load modifiable vote
            vote = getVoteService().getVoteEditable(poll, pollAccount);
        }

        if (isResultAllowed()) {

            // load poll results
            PollResultList pollResultList =
                    getPollResultsService().getResults(poll);

            results = pollResultList.getPollResults();

            if (log.isDebugEnabled()) {
                for (PollResult res : results) {
                    log.debug(res.getName() + ": " + res.getValue()
                              + ", (voteCounting=" + res.getVoteCountingType()
                              + ", byGroup=" + res.isByGroup() + ")");
                }
            }
        }

        if (isCommentAllowed()) {

            // load comments
            comments = getPollCommentService().getAllComments(poll.getPollId());
        }

        feedFileExisting = getPollFeedService().isFeedExists(poll);

        if (log.isDebugEnabled()) {
            Date now = serviceContext.getCurrentTime();
            log.debug("pollChoiceOrVoteStarted = " + isPollChoiceOrVoteStarted());
            log.debug("pollChoiceRunning       = " + isPollChoiceRunning());
            log.debug("pollRunning             = " + poll.isRunning(now));
//            log.info("accountFieldDisplayed   = " + isAccountFieldDisplayed());
            log.debug("creatorOrAdminUser      = " + isCreatorOrAdmin());
            log.debug("isVoteAllowed           = " + isVoteAllowed());
            log.debug("isCommentAllowed        = " + isCommentAllowed());
            log.debug("isResultAllowed         = " + isResultAllowed());
            log.debug("pollVoteVisibility      = " + poll.getPollVoteVisibility());
        }
        return INPUT;
    }

    public String escapeLineBreak(String text) {
        return text;
    }

    public String getImageChoiceName(Choice choice) {
        String name = choice.getName();
        try {
            String result = URLEncoder.encode(name, Charsets.UTF_8.name());
            return result;
        } catch (UnsupportedEncodingException e) {
            throw new PollenTechnicalException(
                    "Could not encode name " + name, e);
        }
    }

    protected void loadPollAccount() throws PollAccountNotFound {

        // Current poll account
        pollAccount = getPollService().getPollAccountEditable(
                getUserSecurityContext().getAccountId(),
                getUserSecurityContext().getUserAccount(), poll);
    }

    protected void loadPoll() throws PollNotFoundException {

        poll = getUserSecurityContext().getPoll();
    }

}
