package org.chorem.pollen.business.persistence;

/*
 * #%L
 * Pollen :: Persistence
 * $Id: Polls.java 3746 2012-11-20 23:58:03Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-persistence/src/main/java/org/chorem/pollen/business/persistence/Polls.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import org.chorem.pollen.PollenFunctions;
import org.chorem.pollen.votecounting.VoteCounting;
import org.chorem.pollen.votecounting.VoteCountingFactory;
import org.chorem.pollen.votecounting.model.GroupOfVoter;
import org.chorem.pollen.votecounting.model.GroupOfVoterBuilder;
import org.chorem.pollen.votecounting.model.SimpleVoterBuilder;
import org.chorem.pollen.votecounting.model.Voter;
import org.chorem.pollen.votecounting.model.VoterBuilder;

import java.util.Map;
import java.util.Set;

/**
 * Useful methods around {@link Poll}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.6
 */
public class Polls {

    /**
     * Collect all vote for all voters of the givne poll.
     * <p/>
     * The result (a set of simple voters) are stored in a fictiv group of voter.
     * <p/>
     * <strong>Note:</strong> Even if the poll is a group type, then no of his
     * group are used (just collect simple voters with no group linkage).
     *
     * @param poll the poll to scan
     * @return the aggregate group of voter containing simple voters with
     *         their votes.
     */
    public static GroupOfVoter toSimpleVotersGroup(Poll poll) {
        Preconditions.checkNotNull(poll);

        SimpleVoterBuilder builder = new SimpleVoterBuilder();

        for (Vote vote : poll.getVote()) {
            fillSimpleVoter(builder, vote);
        }
        Set<Voter> voter = builder.getVoters();
        GroupOfVoter result = GroupOfVoter.newVoter(null, 1.0, voter);
        return result;
    }

    /**
     * Build the group of voter of the poll, using his group to build sub
     * group of the poll.
     * <p/>
     * <strong>Note:</strong> This method can only be used for a group poll.
     *
     * @param poll the poll to scan
     * @return the aggregate group of group of voters with their votes
     */
    public static GroupOfVoter toGroupOfVoters(Poll poll) {
        Preconditions.checkNotNull(poll);
        Preconditions.checkArgument(
                poll.isPollGroup(),
                "Can only use this method for a group poll");

        GroupOfVoterBuilder builder = new GroupOfVoterBuilder();

        // create group of voter
        Map<PollAccount, Vote> voteByPollAccount = Maps.uniqueIndex(
                poll.getVote(), PollenFunctions.VOTE_TO_POLL_ACCOUNT);

        for (VotingList votingList : poll.getVotingList()) {

            builder.newGroupVoter(votingList.getTopiaId(),
                                  votingList.getWeight());

            for (PersonToList personToList : votingList.getPollAccountPersonToList()) {
                PollAccount voterAccount = personToList.getPollAccount();
                Vote vote = voteByPollAccount.get(voterAccount);
                if (vote != null) {
                    fillSimpleVoter(builder, vote);
                }
            }
        }

        Set<Voter> voter = builder.getVoters();
        GroupOfVoter result = GroupOfVoter.newVoter(null, 1.0, voter);
        return result;
    }

    public static VoteCounting getVoteCounting(VoteCountingFactory factory, Poll poll) {
        Preconditions.checkNotNull(factory);
        Preconditions.checkNotNull(poll);
        int id = poll.getVoteCountingType();
        VoteCounting result = factory.getVoteCounting(id);
        Preconditions.checkNotNull(
                result, "Could not find vote counting for id " + id);
        return result;
    }

    protected static void fillSimpleVoter(VoterBuilder builder, Vote vote) {
        builder.newVoter(vote.getPollAccount().getAccountId(), vote.getWeight());
        if (!vote.isChoiceVoteToChoiceEmpty()) {
            for (VoteToChoice voteToChoice : vote.getChoiceVoteToChoice()) {
                builder.addVoteForChoice(
                        voteToChoice.getChoice().getTopiaId(),
                        Double.valueOf(voteToChoice.getVoteValue()));
            }
        }

    }
}
