/*
 * #%L
 * Lima :: Swing
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package org.chorem.lima.ui.vatchart;

import org.chorem.lima.business.LimaServiceFactory;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.business.api.VatStatementService;
import org.chorem.lima.entity.VatStatement;
import org.chorem.lima.entity.VatStatementImpl;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.enums.VatStatementsChartEnum;
import org.chorem.lima.ui.importexport.ImportExport;
import org.jdesktop.swingx.JXTreeTable;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.tree.TreePath;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import static org.nuiton.i18n.I18n.t;

public class VatChartViewHandler implements ServiceListener {

    protected VatStatementService vatStatementService;

    protected VatChartView view;

    protected VatChartViewHandler(VatChartView view) {
        this.view = view;

        vatStatementService =
                LimaServiceFactory.getService(
                        VatStatementService.class);
        LimaServiceFactory.addServiceListener(ImportService.class, this);

        init();
    }

    public void init() {

        InputMap inputMap = view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();

        // add action on Ctrl + N
        String binding = "new-vatStatement";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_N, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addVatStatementMovement();
            }
        });

        // add action on Delete
        binding = "remove-vatStatement";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                removeVatStatement();
            }
        });

        // add action on Ctrl + M
        binding = "modify-vatStatement";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_M, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                updateVatStatement();
            }
        });

    }

    public void addVatStatementMovement() {

        VatChartTreeTableModel treeTableModel =
                (VatChartTreeTableModel) view.getTreeTable().getTreeTableModel();

        int selectedRow = view.getTreeTable().getSelectedRow();
        // get current selection path
        TreePath treePath;
        if (selectedRow == -1) {
            treePath = null;
        } else {
            treePath = view.getTreeTable().getPathForRow(selectedRow);
        }

        VatStatement newVatChartMovement = new VatStatementImpl();
        final VatChartMovementForm vatChartMovementForm =
                new VatChartMovementForm(view);
        VatStatement masterVatStatement = null;
        if (selectedRow != -1) {
            masterVatStatement = (VatStatement) treePath.getLastPathComponent();
        }
        newVatChartMovement.setMasterVatStatement(masterVatStatement);
        vatChartMovementForm.setVatStatement(newVatChartMovement);

        InputMap inputMap = vatChartMovementForm.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = vatChartMovementForm.getRootPane().getActionMap();
        String binding = "dispose";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                vatChartMovementForm.performCancel();
            }
        });

        // jaxx constructor don't call super() ?
        vatChartMovementForm.setLocationRelativeTo(view);
        vatChartMovementForm.setVisible(true);
        newVatChartMovement = vatChartMovementForm.getVatStatement();

        // null == cancel action
        if (newVatChartMovement != null) {

            // add it
            treeTableModel.addVatStatement(treePath, newVatChartMovement);
            refresh();
        }
    }

    public void updateVatStatement() {

        JXTreeTable treeTable = view.getTreeTable();
        VatChartTreeTableModel treeTableModel =
                (VatChartTreeTableModel) treeTable.getTreeTableModel();

        // get selected account
        int selectedRow = view.treeTable.getSelectedRow();
        TreePath treePath = view.treeTable.getPathForRow(selectedRow); // not null
        VatStatement vatStatement =
                (VatStatement) treePath.getLastPathComponent();
        //update Account or update SubLedger
        if (vatStatement != null) {
            // get current selection path
            if (selectedRow != -1) {
                treePath = view.treeTable.getPathForRow(selectedRow);
            } else {
                treePath = new TreePath(treeTableModel.getRoot());
            }
            final VatChartMovementForm vatChartMovementForm =
                    new VatChartMovementForm(view);
            if (vatStatement.getMasterVatStatement() == null) {
                VatStatement masterVatStatement = new VatStatementImpl();
                vatStatement.setMasterVatStatement(masterVatStatement);
            }
            vatChartMovementForm.setVatStatement(vatStatement);

            InputMap inputMap = vatChartMovementForm.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
            ActionMap actionMap = vatChartMovementForm.getRootPane().getActionMap();
            String binding = "dispose";
            inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
            actionMap.put(binding, new AbstractAction() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    vatChartMovementForm.performCancel();
                }
            });

            // jaxx constructor don't call super() ?
            vatChartMovementForm.setLocationRelativeTo(view);
            vatChartMovementForm.setVisible(true);
            // null == cancel action
            vatStatement = vatChartMovementForm.getVatStatement();
            //if action confirmed
            if (vatStatement != null) {
                // update it
                treeTableModel.updateVatStatement(
                        treePath, vatStatement);
            }
        }
    }

    public void importVatStatementChart() {
        final VatChartImportForm form =
                new VatChartImportForm();

        InputMap inputMap = form.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = form.getRootPane().getActionMap();
        String binding = "dispose";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                form.performCancel();
            }
        });

        form.setLocationRelativeTo(view);
        form.setVisible(true);

        Object value = form.getChartVatStatementCombo().getSelectedItem();
        // if action confirmed
        if (value != null) {
            if (form.getDeleteVatStatementChart().isSelected()) {
                vatStatementService.removeAllVatStatement();
            }
            VatStatementsChartEnum vatStatementsEnum = (VatStatementsChartEnum) value;
            ImportExport importExport = new ImportExport(view);
            importExport.importExport(ImportExportEnum.CSV_VAT_IMPORT,
                                      null, vatStatementsEnum.getDefaultFileUrl(), true);
            refresh();
        }
    }

    /** Ask for user to remove for selected account, and remove it if confirmed. */
    public void removeVatStatement() {
        JXTreeTable treeTable = view.getTreeTable();
        VatChartTreeTableModel treeTableModel =
                (VatChartTreeTableModel) treeTable.getTreeTableModel();

        // Any row selected
        int selectedRow = view.treeTable.getSelectedRow();
        if (selectedRow != -1) {
            int n = JOptionPane.showConfirmDialog(view,
                                                  t("lima.vatStatement.remove.confirm"),
                                                  t("lima.vatStatement.remove.title"),
                                                  JOptionPane.YES_NO_OPTION,
                                                  JOptionPane.QUESTION_MESSAGE);
            if (n == JOptionPane.YES_OPTION) {
                // update view of treetable
                TreePath treePath = treeTable.getPathForRow(selectedRow);
                VatStatement vatStatement =
                        (VatStatement) treePath.getLastPathComponent();
                if (!treeTableModel.isLeaf(vatStatement)) {
                    JOptionPane.showMessageDialog(
                            view,
                            t("lima.vatStatement.delete.parent"),
                            t("lima.info"),
                            JOptionPane.INFORMATION_MESSAGE);
                } else {
                    treeTableModel.removeVatStatementObject(
                            treePath, vatStatement);
                }
            }
            refresh();
        }
    }

    public void refresh() {
        JXTreeTable treeTable = view.getTreeTable();
        VatChartTreeTableModel treeTableModel =
                (VatChartTreeTableModel) treeTable.getTreeTableModel();
        treeTableModel.refreshTree();
        //refresh view
        view.repaint();
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        if (methodeName.contains("VatStatements")
            || methodeName.contains("importAll")
            || methodeName.contains("importAs")) {
            refresh();
        }
    }


}
