/*
 * #%L
 * Lima Swing
 * 
 * $Id: LimaMain.java 3051 2010-11-29 14:57:16Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.4.2/lima-swing/src/main/java/org/chorem/lima/LimaMain.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima;

import static org.nuiton.i18n.I18n._;
import java.util.List;
import java.util.Locale;
import javax.swing.SwingUtilities;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.AccountServiceMonitorable;
import org.chorem.lima.business.HttpServerServiceMonitorable;
import org.chorem.lima.business.ejbinterface.AccountService;
import org.chorem.lima.entity.Account;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.MainView;
import org.chorem.lima.ui.MainViewHandler;
import org.chorem.lima.ui.opening.OpeningView;
import org.chorem.lima.util.ErrorHelper;
import org.nuiton.i18n.I18n;
import org.nuiton.util.StringUtil;

/**
 * Lima main.
 * 
 * @author ore
 * @version $Revision: 3051 $
 * 
 * Last update : $Date: 2010-11-29 15:57:16 +0100 (lun., 29 nov. 2010) $
 * By : $Author: echatellier $
 */
public class LimaMain {

    /** Log. */
    private static final Log log = LogFactory.getLog(LimaMain.class);

    /** Lima configuration. */
    public static LimaConfig config;

    /** splash */
    private static LimaSplash splash;
    
    /**
     * Lima main method.
     * 
     * @param args program args
     */
    public static void main(String[] args) {
        
        if (log.isInfoEnabled()) {
            log.info("Lima start at " + new java.util.Date());
            log.info("Args: " + java.util.Arrays.toString(args));
        }
        
        try {
            
            // init root context
            final LimaContext context = init(args);

            // do actions
            config = context.getContextValue(LimaConfig.class);
            config.doAction(LimaConfig.Action.AFTER_INIT_STEP);

            // display main ui
            if (config.isLaunchui()) {
                launch(context);
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error(_("lima.common.globalexception"), ex);
            }
            ErrorHelper.showErrorDialog(_("lima.common.globalexception"), ex);
            System.exit(1);
        }
    }

    /**
     * initialisation de l'application :
     * <p/>
     * chargement du context
     *
     * @param args les arguments passes a l'application
     * @return le context de l'application
     * @throws Exception pour toute erreur pendant l'init
     */
    public static synchronized LimaContext init(String... args) throws Exception {

        // update splash
        splash = new LimaSplash();
        splash.initSplash();
        
        // to enable javassist on webstart, must remove any securityManager,
        // see if this can be dangerous (should not be since jnlp is signed ?)
        // moreover it speeds up the loading :)
        System.setSecurityManager(null);

        // init i18n
        long t0 = System.nanoTime();
        
        // FIXME cause NPE
        // I18n.setInitializer(new DefaultI18nInitializer("lima-swing-i18n"));

        I18n.init(Locale.FRANCE);
        if (log.isDebugEnabled()) {
            log.debug("i18n loading time : " + (StringUtil.convertTime(t0, System.nanoTime())));
        }

        
        Runtime.getRuntime().addShutdownHook(new ShutdownHook());

        // init root context
        LimaContext context = LimaContext.init();
        LimaConfig config = context.getContextValue(LimaConfig.class);
        config.parse(args);

        context.initI18n();

        // prepare ui look&feel and load ui properties
        try {
            SwingUtil.initNimbusLoookAndFeel();
        } catch (Exception e) {
            // could not find nimbus look-and-feel
            if (log.isWarnEnabled()) {
                log.warn(_("lima.warning.nimbus.landf"));
            }
        }

        /* init date converter
        Converter converter = ConverterUtil.getConverter(java.util.Date.class);
        if (converter != null) {
            ConvertUtils.deregister(java.util.Date.class);
            DateConverter dateConverter = new DateConverter();
            dateConverter.setUseLocaleFormat(true);
            ConvertUtils.register(dateConverter, java.util.Date.class);
        }

        if (log.isDebugEnabled()) {
            log.debug("Context init done in " + (StringUtil.convertTime(t0, System.nanoTime())));
        }*/
        return context;
    }
    

    protected static void launch(LimaContext context) throws Exception {
        
        splash.drawVersion(config.getVersion());
        splash.updateProgression(0.1, _("lima.splash.1"));

        // do init ui
        MainViewHandler uiHandler = context.getContextValue(MainViewHandler.class);
        final MainView ui = uiHandler.initUI(context, context.getConfig().isFullScreen());
        ui.setLocationRelativeTo(null);

        // defaut display home view (not closeable, but might be !)
        uiHandler.showHomeView(context);

        // load accounts and test if there is an account plan defined
        // if not, call #loadDefaultAccount()
        splash.updateProgression(0.7, _("lima.splash.2"));
        
        //start http server
        LimaServiceFactory.getInstance().getService(HttpServerServiceMonitorable.class).start();
       
        AccountService accountService =
            LimaServiceFactory.getInstance().getService(
                    AccountServiceMonitorable.class);
        
        List<Account> accounts = accountService.getChildrenAccounts(null);
        if (accounts.isEmpty()) {
            if (log.isInfoEnabled()) {
                log.info("Propose for defaut account loading");
            }
            OpeningView openingView = new OpeningView();
            openingView.setSize(800, 400);
            openingView.setLocationRelativeTo(null);
            openingView.setVisible(true);
        }
        splash.updateProgression(1, _("lima.splash.3"));
        // show ui
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                ui.setVisible(true);
            }
        });
        
    }

    public static class ShutdownHook extends Thread {

        public ShutdownHook() {
            super("Shutdown Lima");
        }

        @Override
        public void run() {
            try {
                super.run();

                LimaContext.get().close();
                LimaServiceFactory.getInstance().destroy();
                // force to kill main thread

                if (log.isInfoEnabled()) {
                    log.info(_("lima.init.closed", new java.util.Date()));
                }
                Runtime.getRuntime().halt(0);
            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error(_("lima.init.errorclosing"), ex);
                }
                Runtime.getRuntime().halt(1);
            }
        }
    }
}
