/*
 * #%L
 * Lima Swing
 * 
 * $Id: LimaConfig.java 3051 2010-11-29 14:57:16Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.4.2/lima-swing/src/main/java/org/chorem/lima/LimaConfig.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima;

import static org.nuiton.i18n.I18n._;
import java.beans.PropertyChangeListener;
import java.util.Locale;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.actions.MiscAction;
import org.chorem.lima.entity.LimaCallaoDAOHelper;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;
import org.nuiton.util.converter.ConverterUtil;

/**
 * La configuration de l'application.
 *
 * @author chemit
 * @version $Revision: 3051 $
 * 
 * Last update : $Date: 2010-11-29 15:57:16 +0100 (lun., 29 nov. 2010) $
 * By : $Author: echatellier $
 */
public class LimaConfig extends ApplicationConfig {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(LimaConfig.class);
    
    protected static LimaConfig instance;
    
    private static final String configFile = "lima-config.properties";
    
    /**
     * La version du logiciel.
     */
    protected Version version;

    /**
     * Get copyrighy text (include version).
     * 
     * @return copyright text
     */
    public String getCopyrightText() {
        return "Version " + getVersion() + " Codelutin @ 2008-2010";
    }

    /**
     * Version as string.
     * 
     * @return le nombre global ex: 3.2.0.0
     */
    public String getVersion() {
        return version.toString();
    }

    /**
     * Lima config constructor.
     * 
     * Define all default options and action alias.
     */
    public LimaConfig() {

        super();

        // set defaut option (included configuration file name : important)
        for (Option o : Option.values()) {
            setDefaultOption(o.key, o.defaultValue);
        }

        // set action alias
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }

        // ajout des alias (can be set in option enum ?)
        addAlias("--disableui", "--launchui false");

        // initialisation des répertoires
        // TODO what is it for ?
        //FileUtil.setCurrentDirectory(getLimaUserDirectory());
        //getLimaUserDirectory().mkdirs();
    }

    public static LimaConfig getInstance() {
        if (instance == null) {
            instance = new LimaConfig();
            instance.loadConfiguration(configFile);
        }
        return instance;
    }
    
    protected void loadConfiguration(String configFileName){

        instance.setConfigFileName(configFileName);
        try {
            instance.parse(new String[0]);
        } catch (ArgumentsParserException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't read configuration", ex);
            }
        }
        instance.setOption("topia.persistence.classes", LimaCallaoDAOHelper.getImplementationClassesAsString());
}
    
    @Override
    public void parse(String[] args) throws ArgumentsParserException {
        
        // super parse, read config file etc...
        super.parse(args);
        
        // on ne peut pas connaitre la version avant la lecture du fichier
        // on supprime tout ce qui est apres - (-SNAPSHOT, -rc-1 ,...)
        String sVersion = VersionUtil.removeSnapshot(getOption("application.version"));
        version = VersionUtil.valueOf(sVersion);
        
    }

    /**
     * Get application locale.
     * 
     * @return configuration application locale
     */
    public Locale getLocale() {
        String local = getOption(Option.LOCALE.key);
        Locale result = ConverterUtil.convert(Locale.class, local);
        return result;
    }

    /**
     * Locale setter for command line parameters.
     * 
     * @param locale new locale
     */
    public void setLocale(String locale) {
        setOption(Option.LOCALE.key, locale);
    }

    /**
     * Change locale (not command line version).
     * Save user file.
     * 
     * @param newLocale new locale
     */
    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        saveForUser();
        firePropertyChange("locale", null, newLocale);
    }

    /**
     * Launch ui configuration value.
     * 
     * @return {@code true} if ui must be displayed
     */
    public boolean isLaunchui() {
        boolean launchUI = getOptionAsBoolean(Option.LAUNCH_UI.key);
        return launchUI;
    }

    /**
     * Launch ui setter for command line parameters.
     * 
     * @param launchui new lauch ui value
     */
    public void setLaunchui(String launchui) {
        setOption(Option.LAUNCH_UI.key, launchui);
    }

    /**
     * Change full screen mode.
     * Save user file.
     * 
     * @param fullscreen new fullscreen mode
     */
    public void setFullscreen(boolean fullscreen) {
        boolean oldValue = isFullScreen();
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        saveForUser();
        firePropertyChange("fullscreen", oldValue, fullscreen);
    }

    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    /**
     * Get support email address.
     * 
     * @return support email
     */
    public String getSupportEmail() {
        return getOption(Option.SUPPORT_EMAIL.key);
    }

    /**
     * Return true if ejb mode is configured as remote.
     * 
     * @return {@code true} if remote mode should be used
     */
    public boolean isEJBRemoteMode() {
        boolean result = getOptionAsBoolean(Option.OPENEJB_REMOTEMODE.key);
        return result;
    }
    
    public String getHostAdress() {
        return getOption(Option.LIMA_HOST_ADDRESS.key);
    } 

    /**
     * Used in ????
     */
    public static final String[] DEFAULT_JAXX_PCS = { "fullScreen", "locale" };

    /**
     * Used in ????
     */
    public void removeJaxxPropertyChangeListener() {
        PropertyChangeListener[] toRemove = jaxx.runtime.JAXXUtil.findJaxxPropertyChangeListener(DEFAULT_JAXX_PCS, getPropertyChangeListeners());
        if (toRemove == null || toRemove.length == 0) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("before remove : " + getPropertyChangeListeners().length);
            log.debug("toRemove : " + toRemove.length);

        }
        for (PropertyChangeListener listener : toRemove) {
            removePropertyChangeListener(listener);
        }
        if (log.isDebugEnabled()) {
            log.debug("after remove : " + getPropertyChangeListeners().length);
        }
    }

    /**
     * Lima option definition.
     * 
     * Contains all lima configuration key, with defaut value and
     * information for jaxx configuration frame ({@link #type},
     * {@link #_transient}, {@link #_final}...)
     */
    public static enum Option implements OptionDef {

        CONFIG_FILE(CONFIG_FILE_NAME, _("lima.config.configFileName.description"), "lima-config.properties", String.class, true, true),
        LOCALE("locale", _("lima.config.locale.description"), "fr_FR", Locale.class, false, false),
        FULL_SCREEN("lima.ui.fullscreen", _("lima.config.ui.fullscreen.description"), "false", Boolean.class, false, false),
        LAUNCH_UI("lima.ui.launchui", _("lima.config.ui.flaunchui.description"), "true", Boolean.class, true, true),
        SUPPORT_EMAIL("lima.misc.supportemail", _("lima.misc.supportemail.description"), "", String.class, false, false),
        OPENEJB_REMOTEMODE("lima.openejb.remotemode", _("lima.openejb.remotemode.description"), "false", String.class, false, false),
        LIMA_HOST_ADDRESS("lima.host.address", _("lima.config.host.adress"), "localhost", String.class, false, false);
        
        public final String key;
        public final String description;
        public String defaultValue;
        public final Class<?> type;
        public boolean _transient;
        public boolean _final;

        private Option(String key, String description, String defaultValue,
                Class<?> type, boolean _transient, boolean _final) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }

        @Override
        public boolean isFinal() {
            return _final;
        }
        
        @Override
        public void setFinal(boolean _final){
            this._final=_final;
        }
        
        @Override
        public boolean isTransient() {
            return _transient;
        }
        
        @Override
        public void setTransient(boolean _transient) {
           this._transient=_transient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue=defaultValue;
        }
        
        @Override
        public String getDescription() {
            return description;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }
    }

    /**
     * Lima action definition.
     */
    public static enum Action {

        HELP(_("lima.action.commandline.help"), MiscAction.class.getName() + "#help", "-h", "--help");

        /** Before init action step. */
        public static final int BEFORE_EXIT_STEP = 0;
        /** After init action step. */
        public static final int AFTER_INIT_STEP = 1;

        public String description;
        public String action;
        public String[] aliases;

        private Action(String description, String action, String... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }
    }
}
