/*
 * #%L
 * Refcomp :: services
 * 
 * $Id: StorageService.java 139 2011-03-22 17:11:05Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-1.0/refcomp-services/src/main/java/fr/inra/refcomp/services/StorageService.java $
 * %%
 * Copyright (C) 2010 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.services;

import fr.inra.refcomp.entities.Agent;
import fr.inra.refcomp.entities.AgentSkill;
import fr.inra.refcomp.entities.EntitiesList;
import fr.inra.refcomp.entities.Department;
import fr.inra.refcomp.entities.Frequency;
import fr.inra.refcomp.entities.Skill;
import fr.inra.refcomp.entities.Unit;
import fr.inra.refcomp.entities.User;
import java.io.Reader;
import java.io.Writer;
import java.util.List;
import fr.inra.refcomp.entities.Cati;
import java.util.Locale;
import java.util.Map;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.WikittyUser;

/**
 * User: couteau
 * Date: 30 nov. 2010
 */
public interface StorageService {

    WikittyProxy getProxy();

    void setProxy(WikittyProxy proxy);

    /**
     * Get all the catis.
     *
     * @return a list containing all the catis.
     */
    List<Cati> getCatis();

    /**
     * Save a cati in database.
     *
     * @param cati the cati to save
     * @return the save cati
     * @throws InvalidEntityException if the cati is not valid and cannot be saved
     */
    Cati saveCati(Locale locale, Cati cati) throws InvalidEntityException;

    /**
     * Return a new cati created in database.
     *
     * @return the new cati
     */
    Cati createCati();

    /**
     * Delete a cati
     *
     * @param cati the cati to delete
     */
    void deleteCati(Cati cati);

    /**
     * Get a cati with its id
     *
     * @param id the cati to get's id
     * @return the cati
     */
    Cati getCati(String id);

    /**
     * Get all the frequencies.
     *
     * @return a list containing all the frequencies.
     */
    List<Frequency> getFrequencies();

    /**
     * Save a frequency in database.
     *
     * @param frequency the frequency to save
     * @return the save frequency
     * @throws AlreadyExistingException if a frequency with the same name already
     *                                  exist in database
     */
    Frequency saveFrequency(Locale locale, Frequency frequency)
            throws InvalidEntityException;

    /**
     * Return a new frequency created in database.
     *
     * @return the new frequency
     */
    Frequency createFrequency();

    /**
     * Delete a frequency
     *
     * @param frequency the frequency to delete
     * @param force true to force the deletion even if used
     * @throws UsedException if not forcing deletion and frequency still used in skills
     */
    void deleteFrequency(Frequency frequency, boolean force)
            throws UsedException;

    /**
     * Get a frequency with its id
     *
     * @param id the frequency to get's id
     * @return the frequency
     */
    Frequency getFrequency(String id);

    /**
     * Get all the users.
     *
     * @return a list containing all the users.
     */
    List<User> getUsers();

    /**
     * Save a user in database.
     *
     * @param user the user to save
     * @return the saved user
     */
    User saveUser(Locale locale, String context, User user)
            throws InvalidEntityException;

    /**
     * Return a new user created in database.
     *
     * @return the new user
     */
    User createUser();

    /**
     * Delete a user
     *
     * @param user the user to delete
     */
    void deleteUser(User user);

    /**
     * Get a user with its id
     *
     * @param id the user to get's id
     * @return the user
     */
    User getUser(String id);

    Agent getAgent(String id);

    EntitiesList getAgentAndDependencies(String id);

    Agent setAgent(User user);

    void unsetAgent(String id);

    /**
     * Save an agent in database.
     *
     * @param agent the agent to save
     * @return the saved agent
     */
    Agent saveAgent(Locale locale, String context, Agent agent)
            throws InvalidEntityException;

    /**
     * Save a skill in database.
     *
     * @param locale the user locale for error messages
     * @param skill the skill to save
     * @return the saved user
     * @throws InvalidEntityException if the skill is not valid
     */
    Skill saveSkill(Locale locale, Skill skill)
            throws InvalidEntityException;

    /**
     * Return a new user created in database.
     *
     * @param parentId the parent skill wikitty Id
     * @return the new user
     */
    Skill createSkill(String parentId);

    /**
     * Delete a skill
     *
     * @param skill the skill to delete
     * @throws UsedException If the skill is used and we cannot delete the skill
     */
    void deleteSkill(Skill skill) throws UsedException;

    /**
     * Get a skill with its id
     *
     * @param id the skill to get's id
     * @return the skill
     */
    Skill getSkill(String id);

    List<Skill> getChildrenSkill(String id);

    /**
     * Log the user in
     * @param login the user login
     * @param password the user password
     * @return true if logged in, false otherwise
     */
    Boolean login(String login, String password);

    void logout();

    Boolean isLoggedIn();

    /**
     * Save a department in database.
     *
     * @param department the department to save
     * @return the saved department
     */
    Department saveDepartment(Locale locale, Department department)
            throws InvalidEntityException;

    /**
     * Save a department in database.
     *
     * @param department the department to save
     * @param units the units linked to the department
     * @return the saved department
     */
    Department saveDepartment(Locale locale,
                              Department department, List<String> units)
            throws InvalidEntityException;

    /**
     * Return a new department created in database.
     *
     * @return the new department
     */
    Department createDepartment();

    /**
     * Delete a department
     *
     * @param department the department to delete
     */
    void deleteDepartment(Department department);

    /**
     * Get a department with its id
     *
     * @param id the department to get's id
     * @return the department
     */
    Department getDepartment(String id);

    /**
     * Get all the departments.
     *
     * @return a list containing all the departments.
     */
    List<Department> getDepartments();

    /**
     * Get all departments for a unit
     * @param id the unit id
     * @return the departments
     */
    List<Department> getDepartments(String id);


    /**
     * Save a unit in database.
     *
     * @param unit the unit to save
     * @return the saved unit
     */
    Unit saveUnit(Locale locale, Unit unit)
            throws InvalidEntityException;

    /**
     * Save a unit in database.
     *
     * @param unit the unit to save
     * @param departments the departments that got the unit
     * @return the saved unit
     */
    Unit saveUnit(Locale locale, Unit unit, List<String> departments)
            throws InvalidEntityException;

    /**
     * Return a new unit created in database.
     *
     * @return the new unit
     */
    Unit createUnit();

    /**
     * Delete a unit
     *
     * @param unit the unit to delete
     */
    void deleteUnit(Unit unit);

    /**
     * Get a unit with its id
     *
     * @param id the unit to get's id
     * @return the unit
     */
    Unit getUnit(String id);

    /**
     * Get all the units for a department
     * @param id the department id
     * @return the list of units
     */
    List<Unit> getUnits(String id);

    /**
     * Get all units
     * @return the department's units
     */
    List<Unit> getUnits();

    void saveEntity(BusinessEntity entity);

    void deleteEntity(BusinessEntity entity);

    /**
     * Get the administrator email address from config
     * @return the admin email address
     */
    String getAdminEmail();

    AgentSkill createAgentSkill(String agentId);

    void saveAgentSkill(Locale locale, AgentSkill skill)
            throws InvalidEntityException;

    boolean isAdmin();

    boolean isAdmin(String id);

    boolean isAgent();

    void adminCreationIfDoNotExist();

    /**
     * Get the skills for an agent
     * @param agentId the wikittyId of the agent to get the skills of
     * @return the agent skills and its dependencies
     */
    EntitiesList getSkills(String agentId);

    AgentSkill addSkill(String agentId);

    EntitiesList addSkill(String agentId, Skill skill, String frequencyId);

    void removeSkill(String skillId);

    public WikittyUser getLoggedInUser();

    EntitiesList saveMySkills(Locale locale, List<AgentSkill> skills)
            throws InvalidEntityException;

    AgentSkill getAgentSkill(String skillId);

    EntitiesList searchUser(String search, String cati, String department,
                               String unit, String frequency);

    EntitiesList searchSkill(String search, String cati, String department,
                               String unit, String frequency);

    void setAdmin(String agentId, boolean admin);

    void importCSV(Reader reader);

    void exportSkillsReferentiel(Writer writer);

    void exportEntitiesReferentiel(Writer writer);

    void exportUserBase(Writer writer);

    String getPath(String id);

    void createPrivateGroupIfDoNotExist();

    void setPrivateSkills(List<String> ids);

    void setPublicSkills(List<String> ids);

    void clear();

    boolean moveSkill(String destination, String node);

    boolean sendPassword(Locale locale, String userId);

    Map<String, Skill> getRootSkill(List<String> skills);

    List<Skill> getAllSkills();

}
