/*
 * #%L
 * Wao :: Web Interface
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.components;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.io.ImportResults;
import fr.ifremer.wao.ui.data.ErrorReport;
import fr.ifremer.wao.ui.data.ImportEngine;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.zip.GZIPInputStream;

/**
 * This component is used to import a {@code csvFile} using a form. The page
 * using this component must provide a {@link ImportEngine} to execute the
 * import when csv file is uploaded. An event "imported" will be triggered
 * after success. The page can use an "onImported(ErrorReport report)" method
 * to display errors and informations from import.
 *
 * Created: 18 juin 2010
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: ImportFieldSet.java 650 2010-10-03 15:14:48Z fdesbois $
 */
public class ImportFieldSet {

    /** Event triggered after form submission success */
    protected static final String EVENT_IMPORTED = "imported";

    protected static final String MIMETYPE_GZIP = "application/x-gzip";

    protected static final String FORMAT_DEFAULT_VALUE = "CSV";

//    protected static final String MIMETYPE_CSV = "text/csv";

    /** Label to display on fieldset and in submit button title */
    @Parameter(required = true, defaultPrefix = BindingConstants.LITERAL)
    @Property
    private String label;

    /** engine to execute the import */
    @Parameter(required = true)
    private ImportEngine engine;

    @Parameter(value = FORMAT_DEFAULT_VALUE, defaultPrefix = BindingConstants.LITERAL)
    @Property
    private String format;

    @Inject
    private WaoManager manager;

    @Inject
    private ComponentResources resources;

    @Inject
    private Messages messages;

    @Inject
    private Logger logger;

    @Property
    private UploadedFile file;

    /**
     * ON_SUCCESS :: Callback method called when success event was triggered from
     * importForm form submission. The import will be executed using the engine
     * in parameter. Errors will be reported in a {@link ErrorReport} object
     * to be managed after import success. An event "imported" will be triggered"
     * to manage those errors.
     */
    @Log
    void onSuccessFromImportForm() {
        final ErrorReport errorReport = new ErrorReport();

        try {
            // Execute import using the engine
            ImportResults result = engine.execute(getStream());

            if (result != null) {
                // Add info on nbRows imported and refused

                StringBuilder info = new StringBuilder().
                        append(result.getNbRowsImported()).
                        append(" lignes importées");

                if (result.getNbRowsImportedNew() > 0) {
                    info.append(" dont ").
                         append(result.getNbRowsImportedNew()).
                         append(" nouvelles");
                }

                info.append(", ").
                     append(result.getNbRowsRefused()).
                     append(" refusées.");

                errorReport.addInfo(info.toString());

                // Add import errors
                List<String> errors = result.getErrors();
                errorReport.addError(errors.toArray(new String[errors.size()]));
            }

        } catch (WaoBusinessException eee) {

            // Add fatal errors
            errorReport.addError(eee.getMessage());
        } catch (WaoException eee) {

            // Add unpredictable errors
            String[] errors =
                    manager.getErrorMessages(eee, messages, logger);
            errorReport.addError(errors);

        } catch (IOException eee) {
            errorReport.addError("Erreur de format du fichier ! Seul les formats [csv, gz, kml] sont autorisés.");
            logger.error("Error on import file", eee);
        }

        // Trigger event "imported" with ErroReport in argument
        resources.triggerEvent(EVENT_IMPORTED, new Object[] {errorReport}, null);
    }

    protected InputStream getStream() throws IOException {
        InputStream input = file.getStream();
        String contentType = file.getContentType();

        if (logger.isDebugEnabled()) {
            logger.debug("Content type : " + file.getContentType());
        }

        if (contentType.equals(MIMETYPE_GZIP)) {
            if (logger.isDebugEnabled()) {
                logger.debug("Gzip file");
            }
            input = new GZIPInputStream(input);

        }
//        else if (!contentType.equals(MIMETYPE_CSV)) {
//            throw new IOException("unsupported format : " + contentType);
//        }
        return input;
    }
}
