/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Contacts.java 680 2010-10-15 16:11:20Z fdesbois $
 * $HeadURL: svn+ssh://bleny@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.5.4/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Contacts.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactFilter;
import fr.ifremer.wao.bean.ContactFilterImpl;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.io.ImportResults;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.base.AbstractFilteredPage;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.ContactDataSource;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.ContactModelFactory;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Field;
import org.apache.tapestry5.PersistenceConstants;
import org.apache.tapestry5.RenderSupport;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.IncludeJavaScriptLibrary;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.json.JSONObject;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.nuiton.util.DateUtil;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * Contacts
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: Contacts.java 680 2010-10-15 16:11:20Z fdesbois $
 */
@SuppressWarnings({"UnusedDeclaration"})
@RequiresAuthentication({UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER})
@IncludeStylesheet("context:css/contacts.css")
@IncludeJavaScriptLibrary("context:js/contacts.js")
public class Contacts extends AbstractFilteredPage {

    @Inject
    private Logger logger;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser user;
    
    @Inject
    private ServiceContact serviceContact;

    @Environmental
    private RenderSupport renderSupport;

    @Log
    void setupRender() throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("RESET DATA");
            logger.debug("User : " + user.getFullName());
        }
        contacts = null;
        contactsForm.clearErrors();
        // Initialize filters if needed
        if (isFiltersVisible()) {
            initSelectFilters(true, true, true);
        }
        // Initialize fullView to true for admin user
        if (fullView == null) {
            fullView = user.isAdmin();
        }
        // The company of connected user will be contributed to abstractFilteredPage
        initCompanyFilter();
    }

    /**
     * Add script to renderSupport
     */
    @Log
    void afterRender() {
        addCommentScript();
        addSendEmailScript();
    }

    /**************************** CONTACT FILTERS *****************************/

    @Persist
    private ContactFilter contactFilter;

    @InjectComponent
    private Zone filtersZone;

    @InjectComponent
    private Zone importExportZone;

    private boolean reset;

    @Property
    private ContactState stateFilter;

    public ContactFilter getContactFilter() throws WaoException {
        if (contactFilter == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Init contactFilter");
            }
            contactFilter = new ContactFilterImpl();
            // Initialized to 12 months before the current day
            Date fromDate = DateUtil.createDateAfterToday(0, -3, 0);
            contactFilter.setFromDate(fromDate);
        }
        return contactFilter;
    }

    public boolean isFiltersVisible() {
        boolean companyFiltered = getContactFilter().getCompany() != null &&
                                        user.isAdmin();
        return (getContactFilter().isFiltered() || companyFiltered) &&
                        StringUtils.isEmpty(getContactFilter().getBoatName());
    }

    @Override
    protected SamplingFilter getFilter() throws WaoException {
        return getContactFilter();
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return true;
    }

    public boolean canFilterExtraComments() {
        return user.isAdmin() || user.isCoordinator();
    }

    Object onActionFromShowFilters() {
        // Initialize filters
        initSelectFilters(true, true, true);
        return filtersZone.getBody();
    }

    Object onActionFromShowImportExport() {
        return importExportZone.getBody();
    }

    void onSelectedFromReset() {
        reset = true;
    }

    @Log
    Object onSuccessFromFiltersForm() throws WaoException {
        if (isEdited()) {
            return filtersZone.getBody();
        }
        if (reset) {
            contactFilter = null;
        }
//        else {
//            filtersVisible = true;
//        }
        return this;
    }

    /**************************** CONTACT IMPORT/EXPORT ***********************/

    @Property
    private UploadedFile contactsCsvFile;

    /**
     * Only administrator and coordinator with no readOnly rights
     * can import/export contacts.
     *
     * @return true if import/export of contacts can be done
     */
    public boolean canImportExport() {
        return (user.isAdmin() ||
                    user.getRole().equals(UserRole.COORDINATOR)) &&
                !user.isReadOnly();
    }

    @Log
    void onSuccessFromImportContacts() throws WaoException {
        if (canImportExport()) {
            try {
                ImportResults result = serviceContact.importContactCsv(user,
                        contactsCsvFile.getStream());
                // Suppress persitant list of contacts
                contacts = null;
                layout.addInfo(result.getNbRowsImported() + " contacts " +
                        "importés,  " + result.getNbRowsRefused() +
                        " refusés.");
                for (String error : result.getErrors()) {
                    layout.addInfo(error);
                }
            } catch (WaoBusinessException eee) {
                layout.addError(eee.getMessage());
            }
        }
    }

    StreamResponse onActionFromExportShowContacts() {
        if (canImportExport()) {
            return new ExportStreamResponse("wao-contacts") {

                @Override
                public InputStream getStream() throws IOException {
                    InputStream result;
                    try {
                        result = serviceContact.exportContactCsv(
                                getContactFilter());
                    } catch (WaoException eee) {
                        throw new IOException(eee);
                    }
                    return result;
                }
            };
        }
        return null;
    }

    /**************************** CONTACT DISPLAY MODE ************************/

    @Persist
    @Property
    private Boolean fullView;

    /**
     * ACTION:: Used to change the display mode of the contacts table.
     * This change affect the loading of the css style over the main table.
     *
     * @see #getGridClass()
     */
    void onActionFromToggleDisplayMode() {
        fullView = !fullView;
    }

    /**************************** CONTACT LIST ********************************/

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private BeanModelSource beanModelSource;

    @Inject
    private ComponentResources resources;

    @Inject
    private ContactModelFactory contactModelFactory;

    @Persist
    private ContactDataSource contacts;
//    private Map<String, Contact> contacts;

    @Property
    private Contact contact;

    private BeanModel<Contact> contactModel;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<WaoUser> userSelectModel;

    @Property
    @Persist(PersistenceConstants.FLASH)
    private String contactUserId;

    private boolean even = true;

    public ContactDataSource getContacts() throws WaoException {
        if (contacts == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Create DataSource");
            }
            contacts = new ContactDataSource(getContactFilter(), serviceContact);
        }
        return contacts;
    }

    public BeanModel<Contact> getContactModel() {
        if (contactModel == null) {

            if (user.isAdmin() || user.isCoordinator()) {
                contactModel = fullView ?
                    contactModelFactory.buildAdminContactModel(beanModelSource, resources) :
                    contactModelFactory.buildCoordinatorContactModel(beanModelSource, resources);
            } else {
                contactModel =
                    contactModelFactory.buildContactModel(beanModelSource, resources);
            }
        }
        return contactModel;
    }

    public GenericSelectModel<WaoUser> getUserSelectModel() {
        if (userSelectModel == null) {
            List<WaoUser> users = serviceUser.getObservers(
                    user.getCompany(), true);
            if (logger.isDebugEnabled()) {
                logger.debug("Nb users : " + users.size());
            }
            userSelectModel = new GenericSelectModel<WaoUser>(users,
                    WaoUser.class, "fullName", "id", propertyAccess);
        }
        return userSelectModel;
    }

    public String getCommentDisplayText(String comment) {
        if (comment != null && comment.length() > 20) {
            return comment.substring(0, 20) + "...";
        }
        return comment;
    }

    public String getCommentTooltip(String comment) {
        return manager.getTooltipText(comment);
    }

    public String getSampleRowDescription() {
        return manager.getTooltipSampleRow(contact.getSampleRow());
    }

    public String getBoatDescription() {
        return manager.getTooltipBoat(contact.getBoat());
    }

    public String getTooltipExportFrom() throws WaoException {
        if (getContactFilter().getFromDate() != null) {
            return "depuis le " + getDateFormat().format(getContactFilter().getFromDate());
        }
        return "";
    }

    public String getGridClass() {
        return fullView ? "admin" : "user";
    }

    public String getRowClass() {
        String result = manager.getContactStyle(contact, user.isAdmin());
        if (contact.getTopiaId().equals(contactSelectedId)) {
            result = "selected";
        }
        return result;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public boolean isEditionMode() {
        //return contact.getTopiaId().equals(contactEditedId);
        return contactEdited != null && contact.equals(contactEdited);
    }

    public boolean isEmpty(Boolean validation) {
        return validation == null;
    }

    /**************************** CONTACT ROW ACTION **************************/

    /**
     * Flag to know if it's only edition (=true) or save action (=false)
     */
    private boolean edited;

    private boolean deleted;

    @Persist
    @Property
    private Contact contactEdited;

    @Persist
    private String oldComment;

    @Persist
    private ContactState oldState;

    public boolean hasActions() {
        return !user.isAdmin() && contact.getValidationCompany() == null
                && !user.isReadOnly();
    }

    /**
     * Display validation actions. Evo #2063 : only coordinator can validate
     * for a company.
     *
     * @return true if the validation actions can be displayed
     */
    public boolean hasValidationActions() {
        return (user.isAdmin() ||
                user.getRole().equals(UserRole.COORDINATOR)) &&
                !user.isReadOnly();
    }

    public boolean canValidate() {
        // Can't validate during edition of the contact row
        if (isEditionMode()) {
            return false;
        }
        switch (user.getRole()) {
            case ADMIN:
                return contact.getValidationProgram() == null &&
                        BooleanUtils.isTrue(contact.getValidationCompany());
            // Evo #2063 : only coordinator can validate
            case COORDINATOR:
                ContactState state = contact.getContactState();
                boolean boardingDone =
                        state.equals(ContactState.BOARDING_DONE) &&
                        contact.getDataInputDate() != null;
                return contact.getValidationCompany() == null &&
                        (state.isUnfinishedState() || boardingDone);
            default:
                return false;
        }
    }

    public boolean canUnvalidate() {
        switch (user.getRole()) {
            case ADMIN:
                return contact.getValidationProgram() != null;
            // Evo #2063 : only coordinator can unvalidate
            case COORDINATOR:
                return contact.getValidationCompany() != null &&
                    contact.getValidationProgram() == null;
            default:
                return false;
        }
    }

    void onSelectedFromAcceptContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Accept contact : " + contactId);
        }
        contactEdited = getContacts().get(contactId);
        if (user.isAdmin()) {
            contactEdited.setValidationProgram(Boolean.TRUE);
        } else {
            // For company accepted, addRealTideTime
//            contactEdited.getSampleRow().addRealTideTime(contactEdited);
            contactEdited.setValidationCompany(Boolean.TRUE);
        }
    }

    void onSelectedFromRefuseContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Refuse contact : " + contactId);
        }
        contactEdited = getContacts().get(contactId);
        if (user.isAdmin()) {
            // For program refused, removeRealTideTime
//            contactEdited.getSampleRow().removeRealTideTime(contactEdited);
            contactEdited.setValidationProgram(Boolean.FALSE);
        } else {
            contactEdited.setValidationCompany(Boolean.FALSE);
        }
    }

    void onSelectedFromUnvalidateContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Unvalidate contact : " + contactId);
        }
        contactEdited = getContacts().get(contactId);
        if (user.isAdmin()) {
            // For program unvalidate from previous refused validation, addRealTideTime
//            if (BooleanUtils.isFalse(contactEdited.getValidationProgram())) {
//                contactEdited.getSampleRow().addRealTideTime(contactEdited);
//            }
            contactEdited.setValidationProgram(null);
        } else {
            // For company unvalidate from previous accepted validation,
            // removeRealTideTime
//            if (BooleanUtils.isTrue(contactEdited.getValidationCompany())) {
//                contactEdited.getSampleRow().removeRealTideTime(contactEdited);
//            }
            contactEdited.setValidationCompany(null);
        }
    }

    @Log
    void onSelectedFromEditContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Edit contact : " + contactId);
            logger.debug("Contact : " + getContacts().get(contactId));
            logger.debug("Set observerId : " + getContacts().get(contactId).getObserver());
        }
        contactEdited = getContacts().get(contactId); //prepareContactEdited(contactId);
        contactUserId = contactEdited.getObserver().getId();
        //contactEditedId = contactId;
        contactSelectedId = contactId;
        oldComment = contactEdited.getComment();
        oldState = contactEdited.getContactState();
        edited = true;
    }

    @Log
    void onSelectedFromDeleteContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Delete contact : " + contactId);
        }
        contactEdited = getContacts().get(contactId);
        deleted = true;
    }

    @Log
    void onSelectedFromSaveContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Save contact : " + contactId);
            logger.debug("Observer Id : " + contactUserId);
        }
        // ContactEdited is in session, previously set by Edit action
//        contactEdited.setState(contactState.toString());
        WaoUser contactUser = getUserSelectModel().findObject(contactUserId);
        contactEdited.setObserver(contactUser);

        if (logger.isDebugEnabled()) {
            logger.debug("Comment : " + contactEdited.getComment());
        }
    }

    void onSelectedFromCancelEditContact() throws WaoException {
        contactEdited = null;
        edited = true;
    }

    /**************************** CONTACT SAVE ********************************/

    @Persist(PersistenceConstants.FLASH)
    private String contactSelectedId;

    @InjectComponent
    private Form contactsForm;

    @InjectComponent
    private Zone gridZone;

    @InjectComponent
    private Field beginDate;

    @InjectComponent
    private Field endDate;

    @InjectComponent
    private Field nbObservants;

    @InjectComponent
    private Field comment;

    @InjectComponent
    private Field inputDate;

    @Inject
    private WaoManager manager;

    @Inject
    private Messages messages;

    @Property
    private boolean sendEmail;

    protected void addSendEmailScript() {
        // Ask user to send an email if not already sent
        if (contactEdited != null && !contactEdited.getEmailSent()) {

            String confirmMessage =
                    "Souhaitez vous envoyer un email de demande d\\'ajout " +
                    "du navire à votre portefeuille Allegro afin de pouvoir " +
                    "saisir les données ?";

            // TODO-fdesbois-2010-07-27 : manage array of states in JavaScript
            renderSupport.addScript("new ContactSendEmail('%s', '%s');",
                    confirmMessage,
                    // Check state BOARDING_DONE
                    ContactState.BOARDING_DONE.name());

            renderSupport.addScript("new ContactSendEmail('%s', '%s');",
                    confirmMessage,
                    // Check state BOARDING_EXPECTED
                    ContactState.BOARDING_EXPECTED.name());
        }
    }

    @Log
    void onValidateFormFromContactsForm() {
        contactsForm.clearErrors();
        // Validation for saving contact depends on contactState (only edition
        // form)
        if (!edited && contactEdited != null) {
            ContactState contactState = contactEdited.getContactState();
            if (logger.isDebugEnabled()) {
                logger.debug("For state : " + contactState);
            }

            Date begin = contactEdited.getTideBeginDate();
            Date end = contactEdited.getTideEndDate();
            Date input = contactEdited.getDataInputDate();
            SampleRow row = contactEdited.getSampleRow();
            //DateFormat dateFormat = new SimpleDateFormat("MM/yyyy");

            if (begin != null && !row.isValid(begin)) {
                contactsForm.recordError(beginDate,
                        "La date de début de la marée doit correspondre à un " +
                        "mois valide (non vide) de la ligne " + row.getCode());
            }

            if (begin != null && end != null && end.before(begin)) {
                contactsForm.recordError(endDate, "La date de fin de la marée" +
                        " ne peut pas être antérieure à celle de début");
            }

            Date current = manager.getCurrentDate();

            if (end != null && end.after(current)) {
                contactsForm.recordError(endDate, "La date de fin de la marée" +
                        " ne peut pas être postérieure à la date du jour");
            }

            if (end != null && input != null && end.after(input)) {
                contactsForm.recordError(inputDate, "La date de saisie des" +
                        " données ne peut pas être antérieure à la date de" +
                        " fin de la marée");
            }

            if (input != null && input.after(current)) {
                contactsForm.recordError(inputDate, "La date de saisie des" +
                        " données ne peut pas être postérieure à la date" +
                        " du jour");
            }

            // Non abouti, Refus ou Refus Définitif
            if (contactState.isUnfinishedState()) {
                String newComment = contactEdited.getComment();

                boolean commentDefined = StringUtils.isNotEmpty(newComment);
                boolean commentChanged = commentDefined &&
                        !newComment.equals(oldComment);

                // Ano #2540 : NPE on oldState, extract boolean for
                // previous unfinishedState case
                boolean previousUnfinishedState = commentDefined &&
                        oldState != null && oldState.isUnfinishedState();

                // Ano #2440 : no restriction if previous state is unfinished
                if (previousUnfinishedState || commentChanged) {

                    // RAZ des champs
                    contactEdited.setTideBeginDate(null);
                    contactEdited.setTideEndDate(null);
                    contactEdited.setNbObservants(0);
                    contactEdited.setMammalsCapture(false);
                    contactEdited.setMammalsObservation(false);
                    contactEdited.setDataInputDate(null);

                } else {
                    contactsForm.recordError(comment, "Vous devez ajouter" +
                            " un commentaire pour l'état" +
                            " '" + contactState.libelle() + "'");
                }
            // Embarquement Réalisé
            } else if (contactState.equals(ContactState.BOARDING_DONE)) {
                if (begin == null) {
                    contactsForm.recordError(beginDate, "La date de début de" +
                            " marée est obligatoire pour l'état" +
                            " '" + contactState.libelle() + "'");
                }
                if (end == null) {
                    contactsForm.recordError(endDate, "La date de fin de" +
                            " marée est obligatoire pour l'état" +
                            " '" + contactState.libelle() + "'");
                }
                if (contactEdited.getNbObservants() == 0) {
                    contactsForm.recordError(nbObservants, "Il ne peut y" +
                            " avoir aucun observateur pour l'état" +
                            " '" + contactState.libelle() + "'");
                }
            }
        }
    }

    @Log
    Object onSuccessFromContactsForm() {
        if (!edited && contactEdited != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Contact save : " + contactEdited);
                logger.debug("Contact sendEmail : " + sendEmail);
            }
            serviceContact.saveContact(contactEdited, deleted);

            try {
                if (sendEmail && serviceContact.sendContactDoneEmail(contactEdited)) {
                    layout.addInfo("Un email a été envoyé pour l'ajout du navire au portefeuille ALLEGRO.");
                }
            } catch (WaoBusinessException eee) {
                String message = manager.getErrorMessage(eee, messages, logger);
                layout.addError(message);
            }

            contactSelectedId = contactEdited.getTopiaId();
            oldComment = null;
            contactEdited = null;
        }
        return this;
    }

    @Log
    Object onFailureFromContactsForm() {
        if (logger.isDebugEnabled()) {
            logger.debug("Contact can't be saved with errors");
        }
        // The contact is not saved, the contact must be editable to show form
        // and correct errors
        edited = true;
        return contactsForm;
        //return gridZone;
    }

    public void createNewContact(Boat boat, SampleRow sampleRow) throws WaoException {
        contact = serviceContact.getNewContact(user.getUser(), sampleRow, boat);
        // Check boat not null and validation for create the new contact
        if (boat != null && boat.canCreateContact(user.getCompany())) {
            serviceContact.saveContact(contact, Boolean.FALSE);
            contactSelectedId = contact.getTopiaId();
        } else {
            layout.addError("Un contact en cours existe déjà pour ce navire");
        }
    }

    /****************************** COMMENT POPUP *****************************/

    @Property
    private String contactId;

    @Property
    private String extraComment;

    protected void addCommentScript() {
       renderSupport.addScript("commentController = new ContactComment('" +
               contactsForm.getClientId() + "');");
    }

    /**
     * Retrieve the current contact comment depends on edition. Also a comment
     * can be added separatly by admin or coordinator.
     *
     * @return the current contact comment used for edition in commentWindow
     * @see #getCommentData()
     */
    public String getContactComment() {
        String comment = null;
        if (isEditionMode()) {
            comment = contact.getComment();
        } else {
            switch (user.getRole()) {
                case ADMIN:
                    comment = contact.getCommentAdmin(); break;
                case COORDINATOR:
                    comment = contact.getCommentCoordinator();
            }
        }
        return comment == null ? "" : comment.replace("'", "\'");
    }

    /**
     * Prepare comment data for commentController javascript. This data is
     * used to open the comment window in javascript.
     *
     * @return a JSONObject that contains data needed for javascript
     */
    public JSONObject getCommentData() {
        JSONObject json = new JSONObject();
        if (isEditionMode()) {
            json.put("formId", contactsForm.getClientId());
        }
        json.put("id", contact.getId());
        json.put("edited", isEditionMode());
        json.put("unfinished", contact.getContactState().isUnfinishedState());
        json.put("comment", getContactComment());
        return json;
    }

    @Log
    void onSuccessFromCommentForm() {
        serviceContact.saveComment(contactId, user.getRole(), extraComment);
    }

}
