/*
 * *##% 
 * Wao :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.wao.ui.components;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoBusinessException.Type;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.WaoProperty;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.base.WaoPage;
import fr.ifremer.wao.ui.pages.Index;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.PersistenceConstants;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectContainer;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.Request;
import org.slf4j.Logger;

/**
 * Layout
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 409 $
 *
 * Mise a jour: $Date: 2010-04-01 20:21:22 +0200 (jeu., 01 avril 2010) $
 * par : $Author: fdesbois $
 */
public class Layout {

    @Parameter(required = true, defaultPrefix = BindingConstants.LITERAL)
    private String pageTitle;

    @Parameter(required = true, defaultPrefix = BindingConstants.LITERAL)
    private String contentId;

    /**
     * Objet de session représentant l'utilisateur identifié
     */
    @SessionState
    private WaoUser currentUser;

    /**
     * vrai si l'utilisateur s'est identifié, faux sinon
     */
    //@SuppressWarnings("unused")
    //@Property
    //private boolean currentUserExists;

    @InjectComponent
    private FeedBack connexionFeedback;

    @InjectComponent
    private FeedBack contentFeedback;

    @InjectContainer
    private WaoPage page;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private Request request;

    @Inject
    private Logger logger;

    private boolean fatal = false;

    void setupRender() {
        if(isCurrentUserExists() && !canShowPage()) {
            contentFeedback.addError("Vous n'êtes pas autorisé à accéder" +
                    " à cette page !");
        }
    }

    boolean afterRender() {
        // Check if fatal errors has been added to the layout
        // Reload the page if feedBack hasn't been reloaded yet to display
        // errors
        if (fatal && contentFeedback.hasErrors() &&
                !contentFeedback.hasBeeanReloaded()) {
            if (logger.isInfoEnabled()) {
                logger.info("[ FATAL ] Reload the page to display" +
                        " errors");
            }
            contentFeedback.reload();
            return false;
        }
        // Clear errors after page render
        contentFeedback.clearErrors();
        fatal = false;
        return true;
    }

    /**
     * Test if the current user in session is a valid user.
     * Test topia id existence because WaoUser can be empty when session is lost
     *
     * @return true if the current user is a valide user (not empty).
     * @see fr.ifremer.wao.ui.services.AppModule#
     * contributeApplicationStateManager(MappedConfiguration)
     */
    public boolean isCurrentUserExists() {
        return StringUtils.isNotEmpty(currentUser.getTopiaId());
    }

    public String getContextPath() {
        return request.getContextPath();
    }

    public String getVersion() {
        return WaoProperty.APP_VERSION.getValue();
    }

    /**
     * User for connexion form data
     */
    @Property
    private String login;

    @Property
    private String password;

    @Persist(PersistenceConstants.FLASH)
    @Property
    private String email;

    public String getPageTitle() {
        return this.pageTitle;
    }

    public String getContentId() {
        return this.contentId;
    }
    
    public void setCurrentUser(WaoUser user) {
        this.currentUser = user;
    }

    public WaoUser getCurrentUser() {
        return this.currentUser;
    }

    public FeedBack getFeedBack() {
        return this.contentFeedback;
    }

    public String getAdminClass() {
        return this.currentUser.isAdmin() ? " admin" : "";
    }

    public String getSamplingSelected() {
        return this.contentId.equals("so-sampling") ? "selected" : "";
    }

    public String getBoatsSelected() {
        return this.contentId.equals("so-boats") ? "selected" : "";
    }

    public String getContactsSelected() {
        return this.contentId.equals("so-contacts") ? "selected" : "";
    }

    public String getSynthesisSelected() {
        return this.contentId.equals("so-synthesis") ? "selected" : "";
    }

    public String getAdminSelected() {
        return this.contentId.equals("so-admin") ? "selected" : "";
    }

    public boolean canShowPage() {
        return page.canDisplay();
    }

    public String getAccessText() {
        String text = currentUser.getUserRole().getLibelle();
        if (currentUser.getReadOnly()) {
            text += " lecture seule";
        }
        return text;
    }
    
//    @InjectPage
//    private Administration administration;
//
//    public boolean canAccessAdministration() {
//        return administration.canDisplay();
//    }

    public boolean displayBody() {
        return canShowPage() && !fatal;
    }

    /**
     * Add an info message to the layout.
     *
     * @param message of the info
     */
    public void addInfo(String message) {
        contentFeedback.addInfo(message);
    }

    /**
     * Add an error to the layout : display the body any way.
     *
     * @param message of the error
     */
    public void addError(String... message) {
        contentFeedback.addError(message);
    }

    /**
     * Add a fatal error to the layout : don't display the page body.
     *
     * @param message of the error
     */
    public void addFatal(String message) {
        contentFeedback.addError(message);
        fatal = true;
    }

    @Log
    Object onSuccessFromConnexionForm() throws WaoException {
        try {
            // FIXME-FD20100120 Problem with NPE in Topia --> see Ano #243 on nuiton.org
            if (login != null && password != null) {
                currentUser = serviceUser.connect(login, password);
            }
        } catch (WaoBusinessException eee) {
            if (eee.getType().equals(Type.BAD_CONNECTION)) {
                connexionFeedback.addError(eee.getMessage());
                email = login;
            } else if (eee.getType().equals(Type.ILLEGAL_CONNECTION)) {
                connexionFeedback.addInfo(eee.getMessage());
            }
        }
        return null;
    }

    public boolean hasConnexionErrors() {
        return connexionFeedback.hasErrors();
    }
    void onSuccessFromForgetPassword() throws WaoException {
        try {
            serviceUser.forgetPassword(email);
            connexionFeedback.addInfo("Un email avec votre nouveau mot de passe vous a été envoyé.");
        } catch (WaoBusinessException eee) {
            connexionFeedback.addError(eee.getMessage());
        }
    }

    @Log
    Object onActionFromExit() {
         setCurrentUser(null);
         request.getSession(false).invalidate();
         return Index.class;
    }
        
}
