package fr.ifremer.tutti.ui.swing.util.computable;

/*
 * #%L
 * Tutti :: UI
 * $Id: ComputableDataTableCell.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/computable/ComputableDataTableCell.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer, CodeLutin, Tony CHEMIT
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import jaxx.runtime.JAXXUtil;

import javax.swing.AbstractCellEditor;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.text.DecimalFormat;

/**
 * Editor for TuttiComputedOrNotData
 *
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputableDataTableCell extends DefaultTableCellRenderer {
    private static final long serialVersionUID = 1L;

    public static TableCellRenderer newRender(TableCellRenderer renderer,
                                              WeightUnit weightUnit,
                                              Color computedDataColor) {

        return new TuttiComputedOrNotDataTableCellRenderer(renderer,
                                                           true,
                                                           weightUnit.getNumberDigits(),
                                                           computedDataColor);
    }

    public static TableCellEditor newEditor(Class type,
                                            WeightUnit weightUnit,
                                            Color computedDataColor) {

        return new TuttiComputedOrNotDataTableCellEditor(type,
                                                         false,
                                                         true,
                                                         weightUnit.getNumberDigits(),
                                                         weightUnit.getNumberEditorPattern(),
                                                         computedDataColor);
    }

    public static class TuttiComputedOrNotDataTableCellEditor
            extends AbstractCellEditor
            implements TableCellEditor, FocusListener, AncestorListener {

        private static final long serialVersionUID = 1L;

        protected final ComputableDataEditor numberEditor;

        protected ComputableData data;

        /** constructor */
        public TuttiComputedOrNotDataTableCellEditor(Class type,
                                                     boolean useSign,
                                                     boolean useFloat,
                                                     Integer decimalNumber,
                                                     String numberPattern,
                                                     Color computedDataColor) {

            numberEditor = new ComputableDataEditor();
            numberEditor.setComputedDataColor(computedDataColor);
            numberEditor.getTextField().setHorizontalAlignment(SwingConstants.RIGHT);
            numberEditor.getTextField().addFocusListener(this);
            numberEditor.getTextField().addAncestorListener(this);
            numberEditor.getTextField().setBorder(new LineBorder(Color.GRAY, 2));
            numberEditor.setSelectAllTextOnError(true);

            numberEditor.setNumberPattern(numberPattern);
            numberEditor.setModelType(type);
            numberEditor.setUseSign(useSign);
            numberEditor.setUseFloat(useFloat);
            numberEditor.setDecimalNumber(decimalNumber);
            numberEditor.init();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value,
                                                     boolean isSelected, int row, int column) {

            data = (ComputableData) value;
            numberEditor.setModel(data.getData());

            // Check nullity and set the text that will be selected with the current value
            if (data.getData() != null) {
                numberEditor.setModelText(String.valueOf(data.getData()));
            }
            return numberEditor;
        }

        public ComputableDataEditor getNumberEditor() {
            return numberEditor;
        }

        @Override
        public ComputableData getCellEditorValue() {
            return data;
        }

        @Override
        public void focusGained(FocusEvent e) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    numberEditor.getTextField().requestFocus();
                    numberEditor.getTextField().selectAll();
                }
            });
        }

        @Override
        public void focusLost(FocusEvent e) {
        }

        @Override
        public void ancestorAdded(AncestorEvent event) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    numberEditor.getTextField().requestFocus();
                    numberEditor.getTextField().selectAll();
                }
            });
        }

        @Override
        public void ancestorRemoved(AncestorEvent event) {
        }

        @Override
        public void ancestorMoved(AncestorEvent event) {
        }

        @Override
        public boolean stopCellEditing() {
            boolean result = super.stopCellEditing();
            // Reset previous data to avoid keeping it on other cell edition
            if (result) {
                data.setData(numberEditor.getModel());

                numberEditor.setBean(null);

                data = null;
            }
            return result;
        }
    }

    public static class TuttiComputedOrNotDataTableCellRenderer
            implements TableCellRenderer {

        protected final TableCellRenderer delegate;

        protected Integer decimalNumber;

        protected Color computedDataColor;

        protected boolean useFloat;

        public TuttiComputedOrNotDataTableCellRenderer(
                TableCellRenderer delegate,
                boolean useFloat,
                Integer decimalNumber,
                Color computedDataColor) {

            this.delegate = delegate;
            this.useFloat = useFloat;
            this.decimalNumber = decimalNumber;
            this.computedDataColor = computedDataColor;
        }

        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {

            ComputableData data = (ComputableData) value;
            Number dataValue = data.getData();
            Font font;
            Color foreground;
            String text;
            if (dataValue == null) {
                dataValue = data.getComputedData();
                font = TuttiUI.TEXTFIELD_COMPUTED_FONT;
                foreground = computedDataColor;

                if (useFloat && decimalNumber != null && dataValue != null) {
                    DecimalFormat decimalFormat = TuttiEntities.getDecimalFormat(1, decimalNumber);
                    text = JAXXUtil.getStringValue(decimalFormat.format(dataValue));
                } else {
                    text = JAXXUtil.getStringValue(dataValue);
                }

            } else {
                font = TuttiUI.TEXTFIELD_NORMAL_FONT;
                foreground = Color.BLACK;
                text = JAXXUtil.getStringValue(dataValue);
            }

            Component component = delegate.getTableCellRendererComponent(table,
                                                                         text,
                                                                         isSelected,
                                                                         hasFocus,
                                                                         row,
                                                                         column);

            if (isSelected) {
                font = font.deriveFont(Font.BOLD);
            }
            component.setFont(font);
            component.setForeground(foreground);
            if (component instanceof JLabel) {
                JLabel jLabel = (JLabel) component;
                jLabel.setHorizontalAlignment(RIGHT);
            }

            return component;
        }

    }
}
