package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosFrequencyUIHandler.java 1469 2013-12-19 13:28:53Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-3.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/frequency/BenthosFrequencyUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.shared.application.swing.util.Cancelable;
import fr.ifremer.shared.application.type.WeightUnit;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.FrequencyConfigurationMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyUIHandler extends AbstractTuttiTableUIHandler<BenthosFrequencyRowModel, BenthosFrequencyUIModel, BenthosFrequencyUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosFrequencyUIHandler.class);

    private BenthosFrequencyCellComponent.FrequencyCellEditor frequencyEditor;

    private Map<Integer, SpeciesProtocol> speciesProtocol;

    private Map<String, Caracteristic> lengthStepCaracteristics;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected WeightUnit weightUnit;

    public BenthosFrequencyUIHandler() {
        super(BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
              BenthosFrequencyRowModel.PROPERTY_NUMBER,
              BenthosFrequencyRowModel.PROPERTY_WEIGHT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosFrequencyTableModel getTableModel() {
        return (BenthosFrequencyTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(BenthosFrequencyRowModel row) {
        return row.getLengthStepCaracteristic() != null &&
               row.getLengthStep() != null && row.getLengthStep() > 0 &&
               ((row.getNumber() == null && row.getWeight() == null) ||
                (row.getNumber() != null && row.getNumber() > 0 &&
                 (getModel().getNbRowsWithWeight() == 0 || row.getWeight() != null && row.getWeight() > 0)));
    }

    @Override
    protected void onModelRowsChanged(List<BenthosFrequencyRowModel> rows) {
        super.onModelRowsChanged(rows);

        BenthosFrequencyUIModel model = getModel();
        model.setEmptyRows(Sets.<BenthosFrequencyRowModel>newHashSet());
        if (CollectionUtils.isNotEmpty(rows)) {
            for (BenthosFrequencyRowModel row : rows) {
                model.updateRowWithWeight(row);
            }
            for (BenthosFrequencyRowModel row : rows) {
                recomputeRowValidState(row);
            }
        }
        model.recomputeTotalNumberAndWeight();
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 BenthosFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        BenthosFrequencyUIModel model = getModel();

        // keep number of rows with weight
        int nbRowsWithWeight = model.getNbRowsWithWeight();

        // update rowWithWeight cache
        model.updateRowWithWeight(row);

        // check if no row had a weight, then if one of them now has a weight,
        // the other ones must have one too to be valid
        boolean recomputeAllRows = nbRowsWithWeight != model.getNbRowsWithWeight();

        if (recomputeAllRows) {
            if (log.isInfoEnabled()) {
                log.info("Revalidate all rows");
            }
            for (BenthosFrequencyRowModel r : model.getRows()) {
                recomputeRowValidState(r);
            }
        }

        model.recomputeTotalNumberAndWeight();

        if (!recomputeAllRows) {
            if (log.isInfoEnabled()) {
                log.info("Revalidate the single selected row");
            }
            recomputeRowValidState(row);
        }
        model.updateEmptyRow(row);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<BenthosFrequencyRowModel> rowMonitor,
                                             BenthosFrequencyRowModel row) {
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          BenthosFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);
        ui.getValidator().doValidate();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<BenthosFrequencyUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(BenthosFrequencyUI ui) {

        super.beforeInit(ui);

        this.weightUnit = getConfig().getBenthosWeightUnit();
        SampleCategoryModel sampleCategoryModel =
                getDataContext().getSampleCategoryModel();

        BenthosFrequencyUIModel model =
                new BenthosFrequencyUIModel(weightUnit, sampleCategoryModel);

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(BenthosFrequencyUI ui) {

        initUI(ui);

        List<Caracteristic> lengthStepCaracterics =
                Lists.newArrayList(getDataContext().getLengthStepCaracteristics());

        lengthStepCaracteristics = TuttiEntities.splitById(lengthStepCaracterics);

        BenthosFrequencyUIModel model = getModel();

        if (getContext().isProtocolFilled()) {

            // FIXME 20130128 kmorin: the species have no technical id
            speciesProtocol =
                    getContext().getPersistenceService().toBenthosProtocolMap();

        } else {
            speciesProtocol = Maps.newHashMap();
        }

        Caracteristic modelCaracteristic = model.getLengthStepCaracteristic();
        initBeanFilterableComboBox(ui.getLengthStepCaracteristicComboBox(),
                                   lengthStepCaracterics,
                                   modelCaracteristic);

        // get precision from the pmfm
        Float precision = null;
        if (modelCaracteristic != null) {
            precision = model.getLengthStepCaracteristic().getPrecision();
        }
        if (precision == null) {
            precision = 1f;
        }
        model.setStep(precision);

        model.setMinStep(null);
        model.setMaxStep(null);

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) BenthosFrequencyUIHandler.this.ui.getRafaleStepField().getModel();

                    applyRafaleStep(step);

                    //select text
                    JTextField field = (JTextField) e.getSource();
                    field.selectAll();
                }
            }
        });

        // when lengthStepCaracteristic changed, let's updates all row with the new value
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_LENGHT_STEP_CARACTERISTIC, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Caracteristic newValue = (Caracteristic) evt.getNewValue();
                if (getModel().getRows()!=null) {
                    for (BenthosFrequencyRowModel rowModel : getModel().getRows()) {
                        rowModel.setLengthStepCaracteristic(newValue);
                        // get precision from the pmfm
                        Float precision = null;
                        if (newValue != null) {
                            precision = newValue.getPrecision();
                        }
                        if (precision == null) {
                            precision = .5f;
                        }
                        getModel().setStep(precision);
                        recomputeRowValidState(rowModel);
                    }
                }
                BenthosFrequencyUIHandler.this.ui.getValidator().doValidate();
            }
        });

        // when configuration mode change, let's focus the best component (see http://forge.codelutin.com/issues/4035)
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_CONFIGURATION_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                final FrequencyConfigurationMode newValue = (FrequencyConfigurationMode) evt.getNewValue();
                SwingUtilities.invokeLater(
                        new Runnable() {
                            @Override
                            public void run() {
                                JComponent componentToFocus = getComponentToFocus(newValue);
                                if (componentToFocus != null) {
                                    componentToFocus.grabFocus();
                                }
                            }
                        }
                );
            }
        });

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN,
                                  table);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    BenthosFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN,
                                    table);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.WEIGHT,
                                  weightUnit,
                                  table);
        }

        // create table model
        BenthosFrequencyTableModel tableModel =
                new BenthosFrequencyTableModel(
                        weightUnit,
                        columnModel,
                        model);


        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        installTableKeyListener(columnModel, table);

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    protected JComponent getComponentToFocus() {
        FrequencyConfigurationMode configurationMode = getModel().getConfigurationMode();
        JComponent componentToFocus = getComponentToFocus(configurationMode);
        if (componentToFocus == null) {
            componentToFocus = getUI().getLengthStepCaracteristicComboBox();
        }
        return componentToFocus;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        frequencyEditor = null;

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        getModel().setSimpleCount(null);

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isDebugEnabled()) {
            log.debug("Cancel UI " + ui);
        }

        // close dialog
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addLengthStepCaracteristic() {

        // compute list of possible caracteristics (all but the one in the select box)
        List<Caracteristic> allNumericCaracteristic = getPersistenceService().getAllNumericCaracteristic();
        List<Caracteristic> toSelect = Lists.newArrayList(allNumericCaracteristic);
        List<Caracteristic> knownCaracteristics = getUI().getLengthStepCaracteristicComboBox().getData();
        toSelect.removeAll(knownCaracteristics);

        // open a dialog to select it

        BeanFilterableComboBox<Caracteristic> editor =
                new BeanFilterableComboBox<Caracteristic>();
        editor.setBeanType(Caracteristic.class);
        editor.setShowReset(true);

        initBeanFilterableComboBox(editor, toSelect, null);

        int response = JOptionPane.showConfirmDialog(
                getTopestUI(),
                editor,
                _("tutti.editBenthosFrequencies.title.addLengthStepCaracteristic"),
                JOptionPane.OK_CANCEL_OPTION);

        Caracteristic selectedItem;
        if (response == JOptionPane.OK_OPTION) {
            selectedItem = (Caracteristic) editor.getSelectedItem();

//            // FIXME ? Should we add it to the combo box universe?
//            ui.getLengthStepCaracteristicComboBox().getData().add(selectedItem);
        } else {
            // user cancel selection
            selectedItem = null;
        }
        // set to model
        getModel().setLengthStepCaracteristic(selectedItem);
    }

    public void generateLengthSteps() {

        BenthosFrequencyUIModel model = getModel();
        BenthosFrequencyTableModel tableModel = getTableModel();

        Map<Float, BenthosFrequencyRowModel> rowsByStep =
                getTableModel().getRowCache();

        Float minStep = model.getLengthStep(model.getMinStep());
        Float maxStep = model.getLengthStep(model.getMaxStep());
        Caracteristic lengthStepCaracteristic = model.getLengthStepCaracteristic();

        Set<Float> existingKeys = Sets.newHashSet(rowsByStep.keySet());
        List<BenthosFrequencyRowModel> rows = Lists.newArrayList(rowsByStep.values());

        for (float i = minStep, step = model.getStep(); i <= maxStep;
             i = TuttiEntities.getRoundedLengthStep(i + step, true)) {

            if (!existingKeys.contains(i)) {

                // add it
                BenthosFrequencyRowModel newRow = tableModel.createNewRow();
                newRow.setLengthStep(i);
                newRow.setLengthStepCaracteristic(lengthStepCaracteristic);
                rows.add(newRow);
            }
        }
        Collections.sort(rows);
        model.setRows(rows);

        // select first cell in table (see http://forge.codelutin.com/issues/2496)
        TuttiUIUtil.doSelectCell(getUI().getTable(), 0, 1);
    }

    public void applyRafaleStep(Float step) {

        if (log.isDebugEnabled()) {
            log.debug("Will apply rafale step: " + step);
        }
        BenthosFrequencyUIModel model = getModel();
        BenthosFrequencyTableModel tableModel = getTableModel();

        Map<Float, BenthosFrequencyRowModel> rowsByStep = tableModel.getRowCache();

        float aroundLengthStep = model.getLengthStep(step);

        BenthosFrequencyRowModel row = rowsByStep.get(aroundLengthStep);

        int rowIndex;

        if (row != null) {

            // increments current row
            Integer number = row.getNumber();
            row.setNumber((number == null ? 0 : number) + 1);
            rowIndex = tableModel.updateRow(row);

        } else {

            // create a new row

            row = tableModel.createNewRow();
            row.setLengthStep(aroundLengthStep);
            row.setNumber(1);
            row.setValid(isRowValid(row));

            // get new index
            List<Float> steps = Lists.newArrayList(rowsByStep.keySet());
            steps.add(aroundLengthStep);

            Collections.sort(steps);

            rowIndex = steps.indexOf(aroundLengthStep);

            tableModel.addNewRow(rowIndex, row);
        }
        Integer totalNumber = model.getTotalNumber();
        if (totalNumber == null) {
            totalNumber = 0;
        }
        model.setTotalNumber(totalNumber + 1);

        getTable().scrollRowToVisible(rowIndex);
    }

    public void editBatch(BenthosFrequencyCellComponent.FrequencyCellEditor editor) {

        BenthosBatchRowModel speciesBatch = editor.getEditRow();

        BenthosFrequencyUIModel model = getModel();
        model.clearWithWeightRows();
        model.setNextEditableRowIndex(editor.getNextEditableRowIndex());
        model.setTotalNumber(null);
        model.setTotalWeight(null);
        model.setSimpleCount(null);
        model.setMinStep(null);
        model.setMaxStep(null);

        frequencyEditor = editor;

        Caracteristic lengthStepCaracteristic = null;
        Float lengthStep;

        List<BenthosFrequencyRowModel> editFrequency = Lists.newArrayList();

        if (speciesBatch != null) {

            List<BenthosFrequencyRowModel> frequency =
                    speciesBatch.getFrequency();

            // try to load existing frequency

            if (CollectionUtils.isNotEmpty(frequency)) {

                BenthosFrequencyTableModel tableModel = getTableModel();

                for (BenthosFrequencyRowModel rowModel : frequency) {

                    BenthosFrequencyRowModel newRow = tableModel.createNewRow();
                    newRow.setLengthStepCaracteristic(rowModel.getLengthStepCaracteristic());
                    newRow.setLengthStep(rowModel.getLengthStep());
                    newRow.setNumber(rowModel.getNumber());
                    newRow.setWeight(rowModel.getWeight());
                    editFrequency.add(newRow);
                }

                // use first frequency row length step caracteristics

                BenthosFrequencyRowModel rowModel = frequency.get(0);
                lengthStepCaracteristic = rowModel.getLengthStepCaracteristic();
                lengthStep = rowModel.getLengthStep();

                if (log.isInfoEnabled()) {
                    log.info("Use existing lengthStep " +
                             "caracteristic / step " +
                             decorate(lengthStepCaracteristic) + " / " +
                             lengthStep);
                }
            }

            BenthosBatchRowModel previousSiblingRow =
                    frequencyEditor.getPreviousSiblingRow();

            if (lengthStepCaracteristic == null && previousSiblingRow != null) {

                // try to get it from his previous brother row
                List<BenthosFrequencyRowModel> previousFrequency =
                        previousSiblingRow.getFrequency();

                if (CollectionUtils.isNotEmpty(previousFrequency)) {

                    // use the first frequency length step caracteristic / step
                    BenthosFrequencyRowModel rowModel = previousFrequency.get(0);
                    lengthStepCaracteristic =
                            rowModel.getLengthStepCaracteristic();
                    lengthStep = rowModel.getLengthStep();
                    if (log.isInfoEnabled()) {
                        log.info("Use previous sibling existing lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }

            if (lengthStepCaracteristic == null) {

                Species species = speciesBatch.getSpecies();

                SpeciesProtocol sProtocol =
                        speciesProtocol.get(species.getReferenceTaxonId());

                if (sProtocol != null) {


                    String lengthStepPmfmId = sProtocol.getLengthStepPmfmId();

                    lengthStepCaracteristic =
                            this.lengthStepCaracteristics.get(lengthStepPmfmId);
                    lengthStep = sProtocol.getLengthStep();

                    if (log.isInfoEnabled()) {
                        log.info("Use existing from protocol lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will edit batch row: " + speciesBatch + " with " +
                      editFrequency.size() + " frequency");
        }

        FrequencyConfigurationMode mode = FrequencyConfigurationMode.AUTO_GEN;
        if (lengthStepCaracteristic == null) {
            Integer taxonId = speciesBatch.getSpecies().getReferenceTaxonId();
            SpeciesProtocol speciesProtocol = this.speciesProtocol.get(taxonId);
            if (speciesProtocol == null || speciesProtocol.getLengthStepPmfmId() == null) {
                mode = FrequencyConfigurationMode.SIMPLE_COUNTING;
            }
        }
        Integer number = speciesBatch.getNumber();
        if (number != null && editFrequency.isEmpty()) {
            mode = FrequencyConfigurationMode.SIMPLE_COUNTING;
            model.setSimpleCount(number);
        }
        // make sure configuration mode will be rebind
        model.setConfigurationMode(null);
        model.setConfigurationMode(mode);

        // connect model to validator
        ui.getValidator().setBean(model);

        // always sort row by their length
        // see http://forge.codelutin.com/issues/2482
        Collections.sort(editFrequency);

        model.setLengthStepCaracteristic(lengthStepCaracteristic);
        model.setRows(editFrequency);

//        // compute total number
//        int totalNumber = model.computeTotalNumber(false);
//        model.setTotalNumber(totalNumber);
//
//        // compute total weight
//        Float totalWeight = null;
//        if (model.isAllRowsWithWeight()) {
//            totalWeight = model.computeTotalWeight(false);
//        }
//        model.setTotalWeight(totalWeight);

        // keep batch (will be used to push back editing entry)
        model.setBatch(speciesBatch);
    }

    public void reset() {

        // remove all frequencies
        getModel().setRows(Lists.<BenthosFrequencyRowModel>newArrayList());
    }

    public void saveAndClose() {

        if (log.isDebugEnabled()) {
            log.debug("Save And Close UI " + ui);
        }

        boolean doSave = canSaveFrequencies();

        if (doSave) {

            frequencyEditor.save(getModel(), true);
            closeUI(ui);
        }
    }

    public void saveAndContinue() {

        if (log.isDebugEnabled()) {
            log.debug("Save And Continue UI " + ui);
        }

        boolean doSave = canSaveFrequencies();

        if (doSave) {
            frequencyEditor.save(getModel(), false);
        }
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected JComponent getComponentToFocus(FrequencyConfigurationMode mode) {
        JComponent componentToFocus = null;
        if (mode != null) {
            boolean withLengthStepCaracteristic =
                    getModel().getLengthStepCaracteristic() != null;
            switch (mode) {
                case AUTO_GEN:
                    if (withLengthStepCaracteristic) {

                        componentToFocus = ui.getMinStepField();
                    } else {
                        componentToFocus = ui.getLengthStepCaracteristicComboBox();
                    }
                    break;
                case RAFALE:

                    if (withLengthStepCaracteristic) {

                        componentToFocus = ui.getRafaleStepField();
                    } else {
                        componentToFocus = ui.getLengthStepCaracteristicComboBox();
                    }

                    break;
                case SIMPLE_COUNTING:
                    componentToFocus = ui.getSimpleCountingField();
                    break;
                default:
                    componentToFocus = null;
            }
        }
        return componentToFocus;
    }

    protected boolean canSaveFrequencies() {
        boolean doSave = true;

        // check for doublon
        // check that we do not have doublon in length
        // see http://forge.codelutin.com/issues/2499
        Set<Float> lengths = Sets.newHashSet();

        Float doublon = null;
        int index = 0;
        List<BenthosFrequencyRowModel> rows = getModel().getRows();
        for (BenthosFrequencyRowModel row : rows) {
            Float lengthStep = row.getLengthStep();
            if (!lengths.add(lengthStep)) {

                // already exist
                doublon = lengthStep;
                break;
            }
            index++;
        }
        if (doublon != null) {

            // can't save mensurations (found doublon)
            String message =
                    _("tutti.editBenthosFrequencies.error.length.doublon",
                      doublon, index + 1);
            getContext().getErrorHelper().showErrorDialog(
                    message);

            // focus to first error row
            TuttiUIUtil.selectFirstCellOnRow(getTable(), index, false);
            doSave = false;
        }

        // ask user if there is some rows we can't save
        // see http://forge.codelutin.com/issues/4046
        if (doSave && getModel().isSomeRowsWithWeightAndOtherWithout()) {

            // there is some rows with weight and other without
            // ask user what to do

            String htmlMessage = String.format(
                    CONFIRMATION_FORMAT,
                    _("tutti.editBenthosFrequencies.askBeforeSave.message"),
                    _("tutti.editBenthosFrequencies.askBeforeSave.help"));
            int answer = JOptionPane.showConfirmDialog(
                    getTopestUI(),
                    htmlMessage,
                    _("tutti.editBenthosFrequencies.askBeforeSave.title"),
                    JOptionPane.YES_NO_OPTION,
                    JOptionPane.QUESTION_MESSAGE);

            switch (answer) {
                case JOptionPane.YES_OPTION:

                    // ok can save
                    break;
                default:

                    // do not save
                    doSave = false;
            }
        }

        return doSave;
    }
}
