package fr.ifremer.tutti.ui.swing.content;

/*
 * #%L
 * Tutti :: UI
 * $Id: UpdateApplicationAction.java 1223 2013-09-25 13:31:19Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.7/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/UpdateApplicationAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.ui.swing.RunTutti;
import fr.ifremer.tutti.ui.swing.TuttiApplicationUpdaterCallBack;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationUpdater;

import java.io.File;

import static org.nuiton.i18n.I18n._;

/**
 * To update jre / i18n or tutti using the {@link ApplicationUpdater} mecanism.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class UpdateApplicationAction extends AbstractTuttiAction<TuttiUIContext, MainUI, MainUIHandler> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(UpdateApplicationAction.class);

    public UpdateApplicationAction(MainUIHandler handler) {
        super(handler, true);
        setActionDescription(_("tutti.main.action.updateApplication.tip"));
        types = TuttiApplicationUpdaterCallBack.UpdateType.values();
    }

    protected TuttiApplicationUpdaterCallBack.UpdateType[] types;

    protected boolean reload;

    @Override
    public boolean prepareAction() throws Exception {
        boolean doAction = super.prepareAction();

        if (doAction) {
            // check application url is reachable
            TuttiUIContext context = getContext();
            doAction = context.checkUpdateApplicationReachable();
        }
        return doAction;
    }

    @Override
    public void releaseAction() {
        super.releaseAction();
        types = TuttiApplicationUpdaterCallBack.UpdateType.values();
    }

    @Override
    public void doAction() throws Exception {

        reload = false;

        TuttiUIContext context = getContext();
        TuttiApplicationConfig config = context.getConfig();

        File current = config.getTuttiBasedir();
        if (current == null || !current.exists()) {

            // can not update application
            if (log.isWarnEnabled()) {
                log.warn("No application base directory defined, skip updates.");
            }
        } else {

            String url = config.getUpdateApplicationUrl();
            File dest = new File(config.getTuttiBasedir(), "NEW");

            if (log.isInfoEnabled()) {
                log.info(String.format("Try to update jre, i18N, help or tutti (current application location: %s), using update url: %s", current, url));
            }

            ProgressionModel progressionModel = new ProgressionModel();
            context.getActionUI().getModel().setProgressionModel(progressionModel);
            progressionModel.setMessage(_("tutti.updateApplication.checkUpdates"));

            TuttiApplicationUpdaterCallBack callback =
                    new TuttiApplicationUpdaterCallBack(this, progressionModel);

            callback.setTypes(types);

            ApplicationUpdater up = new ApplicationUpdater();
            up.update(url,
                      current,
                      dest,
                      false,
                      callback,
                      progressionModel);

            if (callback.isApplicationUpdated()) {

                reload = true;

            } else {

                sendMessage(_("tutti.updateApplication.noUpdate"));
            }
        }
    }

    public void setTypes(TuttiApplicationUpdaterCallBack.UpdateType... types) {
        this.types = types;
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        if (reload) {
            // wait 1 second to be sure action ui is up
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                if (log.isWarnEnabled()) {
                    log.warn("Could not wait 1 second...", e);
                }
            }

            // tell user restart will be done

            getHandler().showSuccessMessage(_("tutti.updateApplication.title.success"),
                                            _("tutti.updateApplication.message.success"));

            CloseApplicationAction action = TuttiActionHelper.createLogicAction(
                    getHandler(), CloseApplicationAction.class);
            action.setExitCode(RunTutti.UPATE_EXIT_CODE);
            TuttiActionHelper.runAction(action);
        }
    }

    public boolean isReload() {
        return reload;
    }
}