package fr.ifremer.tutti.ui.swing.content.operation.catches;

/*
 * #%L
 * Tutti :: UI
 * $Id: CleanBatchWeightsAction.java 1250 2013-09-30 12:34:52Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.6/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/CleanBatchWeightsAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.InvalidBatchModelException;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.service.catches.WeightCleaningService;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JOptionPane;

import static org.nuiton.i18n.I18n._;

/**
 * To remove weight in double for species and benthos rows.
 * <p/>
 * Created on 9/27/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 2.6
 */
public class CleanBatchWeightsAction extends AbstractTuttiAction<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(CleanBatchWeightsAction.class);

    protected final WeightCleaningService cleaningService;

    protected int nbErrors;

    public CleanBatchWeightsAction(EditCatchesUIHandler handler) {
        super(handler, false);
        cleaningService = getContext().getWeightCleaningService();
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean doAction = super.prepareAction();
        if (doAction) {


            // do a check
            FishingOperation fishingOperation = getModel().getFishingOperation();
            Multimap<String, String> errors = cleaningService.checkFishingOperation(fishingOperation.getId());

            nbErrors = errors.size();

            if (nbErrors == 0) {

                // no errors

                sendMessage(_("tutti.editCatchBatch.action.cleanWeights.no.double.weight.detected"));
                doAction = false;
            } else {

                // show errors to user and ask him to apply change

                String errorsStr = cleaningService.errorsToString(errors);
                String htmlMessage = String.format(
                        AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                        _("tutti.editCatchBatch.action.cleanWeights.double.weight.detected", errorsStr),
                        _("tutti.editCatchBatch.action.cleanWeights.help"));

                int answer = JOptionPane.showConfirmDialog(
                        getContext().getActionUI(),
                        htmlMessage,
                        _("tutti.editCatchBatch.action.cleanWieghts.resume.title"),
                        JOptionPane.OK_CANCEL_OPTION,
                        JOptionPane.QUESTION_MESSAGE);

                doAction = (answer == JOptionPane.OK_OPTION);

            }
        }
        return doAction;
    }

    @Override
    public void doAction() throws Exception {

        EditCatchesUIModel model = getModel();
        FishingOperation fishingOperation = getModel().getFishingOperation();
        model.setLoadingData(true);

        cleaningService.cleanFishingOperation(fishingOperation.getId());

        try {

            getUI().getSpeciesTabContent().getHandler().selectFishingOperation(fishingOperation);
        } catch (InvalidBatchModelException e) {

            // invalid sample category model for species batches
            if (log.isDebugEnabled()) {
                log.debug("Invalid sample category model for species batches", e);
            }
        }

        try {
            getUI().getBenthosTabContent().getHandler().selectFishingOperation(fishingOperation);
        } catch (InvalidBatchModelException e) {

            // invalid sample category model for benthos batches
            if (log.isDebugEnabled()) {
                log.debug("Invalid sample category model for benthos batches", e);
            }
        }
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        sendMessage(_("tutti.editCatchBatch.action.cleanWeights.done", nbErrors));
    }

}