package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditCruiseUIHandler.java 1061 2013-05-31 09:45:22Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/EditCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.CloseableUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.SwingUtilities;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Handler of UI {@link EditCruiseUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditCruiseUIHandler extends AbstractTuttiUIHandler<EditCruiseUIModel, EditCruiseUI> implements CloseableUI {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EditCruiseUIHandler.class);

    public static final String CRUISE_CARD = "cruise";

    public static final String GEAR_CARACTERISTICS_CARD = "gearCaracteristics";

    public static String getTitle(boolean exist) {

        String result;
        if (exist) {
            result = _("tutti.editCruise.title.edit.cruise");
        } else {
            result = _("tutti.editCruise.title.create.cruise");
        }
        return result;
    }

    /**
     * Persistence service.
     *
     * @since 0.1
     */
    private final PersistenceService persistenceService;

    public EditCruiseUIHandler(TuttiUI parentUi, EditCruiseUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        persistenceService = context.getPersistenceService();
    }

    @Override
    public void beforeInitUI() {

        getDataContext().resetValidationDataContext();

        EditCruiseUIModel model = new EditCruiseUIModel();

        if (context.isCruiseFilled()) {

            // load existing cruise
            Cruise cruise = getDataContext().getCruise();

            model.fromBean(cruise);

            // load attachments
            List<Attachment> attachments = persistenceService.getAllAttachments(
                    model.getObjectType(), model.getObjectId());
            model.addAllAttachment(attachments);

            if (log.isDebugEnabled()) {
                log.debug("Will edit cruise: " + cruise.getId());
            }
        } else {

            // create new cruise
            Program program = getDataContext().getProgram();
            model.setProgram(program);
            model.setMultirigNumber(1);


            if (log.isDebugEnabled()) {
                log.debug("Will create a new cruise from program: " + program);
            }
        }
        listModelIsModify(model);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Person> users = persistenceService.getAllPerson();

        EditCruiseUIModel model = getModel();

        List<Vessel> scientificVesselList = getDataContext().getScientificVessels();
        List<Vessel> fishingVesselList = getDataContext().getFishingVessels();
        List<Vessel> allVesselList = Lists.newArrayList();
        allVesselList.addAll(scientificVesselList);
        allVesselList.addAll(fishingVesselList);

        List<Gear> scientificGearList = getDataContext().getScientificGears();
        List<Gear> fishingGearList = getDataContext().getFishingGears();
        List<Gear> allGearList = Lists.newArrayList();
        allGearList.addAll(scientificGearList);
        allGearList.addAll(fishingGearList);

        // Change vessel list and gear list when vessel type changes
        model.addPropertyChangeListener(EditCruiseUIModel.PROPERTY_VESSEL_TYPE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                VesselTypeEnum vesselType = (VesselTypeEnum) evt.getNewValue();

                Predicate<Gear> gearPredicate = null;

                List<Vessel> allVesselList = Lists.newArrayList();


                switch (vesselType) {
                    case FISHING:
                        gearPredicate = TuttiEntities.IS_FISHING_GEAR;
                        allVesselList.addAll(getDataContext().getFishingVessels());
                        break;

                    case SCIENTIFIC:
                        gearPredicate = TuttiEntities.IS_SCIENTIFIC_GEAR;

                        allVesselList.addAll(getDataContext().getScientificVessels());
                        break;
                    default: {
                        allVesselList.addAll(getDataContext().getScientificVessels());
                        allVesselList.addAll(getDataContext().getFishingVessels());
                    }
                }

                ui.getGearList().getHandler().clearFilters();
                ui.getVesselComboBox().setData(null);
                ui.getVesselComboBox().setData(allVesselList);

                if (gearPredicate != null) {
                    ui.getGearList().getHandler().addFilter(gearPredicate);
                }

            }
        });

        initBeanFilterableComboBox(ui.getProgramComboBox(),
                                   Lists.newArrayList(persistenceService.getAllProgram()),
                                   model.getProgram(),
                                   DecoratorService.ONLY_NAME);

        initBeanFilterableComboBox(ui.getDepartureLocationComboBox(),
                                   Lists.newArrayList(persistenceService.getAllHarbour()),
                                   model.getDepartureLocation());

        initBeanFilterableComboBox(ui.getReturnLocationComboBox(),
                                   Lists.newArrayList(persistenceService.getAllHarbour()),
                                   model.getReturnLocation());

        initBeanFilterableComboBox(ui.getVesselComboBox(),
                                   allVesselList,
                                   model.getVessel());

        initBeanList(ui.getGearList(),
                     allGearList,
                     model.getGear());

        initBeanList(ui.getHeadOfMissionList(),
                     users,
                     model.getHeadOfMission());

        initBeanList(ui.getHeadOfSortRoomList(),
                     users,
                     model.getHeadOfSortRoom());

        ui.getGearList().getSelectedList().addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {

                if (SwingUtilities.isRightMouseButton(e)) {
                    JList jList = (JList) e.getSource();
                    Point point = e.getPoint();
                    int selectedIndex = jList.locationToIndex(point);
                    Rectangle rect = jList.getCellBounds(selectedIndex, selectedIndex);
                    if (rect.contains(point)) {
                        jList.setSelectedIndex(selectedIndex); //select the item
                    } else {
                        jList.clearSelection();
                    }

                    Gear gear = (Gear) jList.getSelectedValue();
                    boolean editMenuEnabled = false;
                    boolean viewMenuEnabled = false;
                    if (gear != null) {
                        editMenuEnabled = !getModel().isCreate() && !getModel().isModify()
                                          && (!gear.isScientificGear() || persistenceService.isTemporary(gear));
                        viewMenuEnabled = true;
                        ui.setContextValue(gear);
                    }
                    ui.getEditGearCaracteristicsItem().setEnabled(editMenuEnabled);
                    ui.getViewGearCaracteristicsItem().setEnabled(viewMenuEnabled);
                    ui.getGearMenuPopup().show(jList, e.getX(), e.getY()); //and show the menu
                }
            }
        });

        ui.getGearList().getSelectedList().addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_CONTEXT_MENU) {

                    JList source = (JList) e.getSource();

                    // get the lowest selected row
                    int[] selectedRows = source.getSelectedIndices();
                    int lowestRow = -1;
                    for (int row : selectedRows) {
                        lowestRow = Math.max(lowestRow, row);
                    }
                    Rectangle r = source.getCellBounds(lowestRow, lowestRow);

                    // get the point in the middle lower of the cell
                    Point p = new Point(r.x + r.width / 2, r.y + r.height);

                    Object[] gears = source.getSelectedValues();

                    boolean editMenuEnabled = false;
                    boolean viewMenuEnabled = false;
                    if (gears != null && gears.length == 1) {
                        Gear gear = (Gear) gears[0];
                        editMenuEnabled = !getModel().isCreate() && !getModel().isModify()
                                          && (!gear.isScientificGear() || persistenceService.isTemporary(gear));
                        viewMenuEnabled = true;
                        ui.setContextValue(gear);
                    }
                    ui.getEditGearCaracteristicsItem().setEnabled(editMenuEnabled);
                    ui.getViewGearCaracteristicsItem().setEnabled(viewMenuEnabled);
                    ui.getGearMenuPopup().show(source, p.x, p.y); //and show the menu

                }
            }
        });

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        model.setVesselType(VesselTypeEnum.SCIENTIFIC);

        // if new fishingOperation can already cancel his creation
        model.setModify(model.isCreate());

        registerValidators(validator);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getProgramComboBox();
    }

    @Override
    protected Set<String> getPropertiesToIgnore() {
        Set<String> result = super.getPropertiesToIgnore();
        result.add(EditCruiseUIModel.PROPERTY_ATTACHMENT);
        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();

        ui.getAttachmentsButton().onCloseUI();
    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen(
                getModel().isValid(),
                getModel().isModify(),
                _("tutti.editCruise.askCancelEditBeforeLeaving.cancelSaveCruise"),
                _("tutti.editCruise.askSaveBeforeLeaving.saveCruise"),
                ui.getSaveButton().getAction()
        );
        return result;
    }

    @Override
    public SwingValidator<EditCruiseUIModel> getValidator() {
        return ui.getValidator();
    }

    public void generateCampaignName() {

        EditCruiseUIModel model = getModel();
        String name = model.getGeneratedCampaignName();
        model.setName(name);
    }
}
