package fr.ifremer.tutti.ui.swing.content.catches;

/*
 * #%L
 * Tutti :: UI
 * $Id: CatchesUIHandler.java 20 2012-11-26 10:27:11Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/catches/CatchesUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.CampaignBean;
import fr.ifremer.tutti.persistence.entities.TraitBean;
import fr.ifremer.tutti.persistence.entities.TuttiBeans;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Date;
import java.util.List;

/**
 * Handler of UI {@link CatchesUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class CatchesUIHandler extends AbstractTuttiUIHandler<CatchesUIModel> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(CatchesUIHandler.class);

    private final CatchesUI ui;

    private final PersistenceService persistenceService;

    public CatchesUIHandler(TuttiUIContext context, CatchesUI ui) {
        super(context);
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {

        CatchesUIModel model = new CatchesUIModel();

        String campaignId = context.getCampaignId();

        // load existing campaign
        CampaignBean campaign = persistenceService.getCampaign(campaignId);
        model.setCampaign(campaign);

        List<TraitBean> traits = persistenceService.getAllTraits(campaignId);
        model.setTrait(traits);

        if (log.isInfoEnabled()) {
            log.info("Loaded " + traits.size() + " trait(s).");
        }

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        CatchesUIModel model = ui.getModel();

        initUI(ui);

        List<TraitBean> traits = model.getTrait();

        initBeanComboBox(ui.getTraitComboBox(),
                         traits,
                         model.getSelectedTrait());

        model.addPropertyChangeListener(CatchesUIModel.PROPERTY_SELECTED_TRAIT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                selectTrait((TraitBean) evt.getNewValue());
            }
        });

        model.addPropertyChangeListener(CatchesUIModel.PROPERTY_TRAIT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                ui.getTraitComboBox().setData(null);
                ui.getTraitComboBox().setData((List<TraitBean>) evt.getNewValue());
            }
        });

        if (!traits.isEmpty()) {

            model.setSelectedTrait(traits.get(0));
        }
    }

    @Override
    public void onCloseUI() {

        // ui will be saved so we do not want to keep selected tab indexes
        ui.getTabPane().setSelectedIndex(0);
        ui.getTraitTabContent().getTraitTabPane().setSelectedIndex(0);
    }

    @Override
    protected CatchesUIModel getModel() {
        return ui.getModel();
    }

    public void createNewTrait() {

        // deselect selected trait
        ui.getModel().setSelectedTrait(null);

        // use a new empty trait
        TraitBean newTrait = new TraitBean();
        newTrait.setCampaign(ui.getModel().getCampaign());
        newTrait.setDate(new Date());
        selectTrait(newTrait);
    }

    public void selectTrait(TraitBean selectedTrait) {

        if (log.isInfoEnabled()) {
            log.info("New selected trait: " + selectedTrait);
        }
        // back to trait tab
        ui.getTabPane().setSelectedIndex(0);

        // back to general tab of trait tabs
        ui.getTraitTabContent().getTraitTabPane().setSelectedIndex(0);

        // propagate trait to his tabs
        ui.getTraitTabContent().selectTrait(selectedTrait);
        ui.getFishesTabContent().selectTrait(selectedTrait);
        ui.getBenthosTabContent().selectTrait(selectedTrait);

        // repaint tabs
        ui.getTabPane().repaint();

//        ui.getPlanctonTabContent().selectTrait(selectedTrait);
//        ui.getMacroDechetsTabContent().selectTrait(selectedTrait);
//        ui.getCaptureAccidentellesTabContent().selectTrait(selectedTrait);
//        ui.getObservationIndividuellesTabContent().selectTrait(selectedTrait);
    }

    public void saveTrait(TraitBean toSave) {

        // persist the trait

        boolean create = toSave.getId() == null;

        TraitBean savedTrait;
        if (create) {
            savedTrait = persistenceService.createTrait(toSave);
        } else {
            savedTrait = persistenceService.saveTrait(toSave);
        }

        CatchesUIModel model = ui.getModel();

        // add the saved trait to trait list

        List<TraitBean> data = model.getTrait();

        TraitBean existingTrait = TuttiBeans.findById(data, savedTrait.getId());
        if (existingTrait != null) {
            data.remove(existingTrait);
        }
        data.add(savedTrait);

        // update model trait list
        model.setTrait(data);

        // update model selected trait
        model.setSelectedTrait(savedTrait);
    }
}
