/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;
import org.nuiton.util.FileUtil;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyService;
import org.sharengo.wikitty.WikittyUtil;

import com.jurismarches.vradi.VradiConstants.FormTypeTemplateEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;

/**
 * Class containing the metods to manage the form types :
 * - creation, update, retrieving
 * - cache filling
 *
 * @author schorlet
 * @date 2010-01-28 08:08:53
 * @version $Revision: 684 $ $Date: 2010-04-07 22:59:41 +0200 (mer., 07 avril 2010) $
 */
public class FormTypeManager {
    private static final Log log = LogFactory.getLog(FormTypeManager.class);

    private final WikittyProxy proxy;

    public FormTypeManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    public FormTypeManager() {
        proxy = ServiceFactory.getWikittyProxy();
    }

    /**
     * Compute the specified extension and return a new one with ordered fields.
     * Fields order is specified by their tag values.
     */
    private WikittyExtension computeExtension(WikittyExtension extension) {
        if (log.isDebugEnabled()) {
            log.debug("computeExtension(extension)");
        }
        
        Collection<String> fieldNames = extension.getFieldNames();
        List<String> fieldOrder = new ArrayList<String>();
        List<String> orderlessFields = new ArrayList<String>();
        
        for (String fieldName : fieldNames) {
            FieldType fieldType = extension.getFieldType(fieldName);

            try {
                Integer.valueOf(fieldType.getTagValue("rank"));
                fieldOrder.add(fieldName);
                
            } catch(Exception eee) {
                log.warn(fieldName + " does not have a valid rank");
                orderlessFields.add(fieldName);
            }
        }
        
        LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        
        int size = fieldOrder.size();
        for(int i = 0; i < size; i++) {
            String key = fieldOrder.get(i);
            fields.put(key, extension.getFieldType(key));
        }
        
        size = orderlessFields.size();
        for(int i = 0; i < size; i++) {
            String key = orderlessFields.get(i);
            fields.put(key, extension.getFieldType(key));
        }
        
        WikittyExtension fieldOrderedExtension =
                new WikittyExtension(extension.getName(),
                        extension.getVersion(),
                        extension.getRequires(),
                        fields);
        fieldOrderedExtension.setTagValues(extension.getTagValues());

        return fieldOrderedExtension;
    }

    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     */
    public List<WikittyExtension> getAllFormTypes() {
        if (log.isDebugEnabled()) {
            log.debug("getAllFormTypes()");
        }
        
        List<WikittyExtension> extensions = new ArrayList<WikittyExtension>();

        WikittyService wikittyService = proxy.getWikittyService();
        List<String> allExtensionIds = wikittyService
                .getAllExtensionsRequires(Form.EXT_FORM);
        Map<String, WikittyExtension> lastVersions =
                new HashMap<String, WikittyExtension>();

        for (String extensionId : allExtensionIds) {
            String extensionName  = WikittyExtension.computeName(extensionId);
            if(lastVersions.get(extensionName) == null) {
                WikittyExtension extension = wikittyService
                        .restoreExtensionLastVersion(extensionName);
                extension = computeExtension(extension);
                lastVersions.put(extensionName, extension);
            }
        }
        extensions.addAll(lastVersions.values());
        
        return extensions;
    }

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the paramater
     */
    public WikittyExtension getFormType(String name) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getFormType(" + name + ")");
        }
        
        try {
            WikittyService wikittyService = proxy.getWikittyService();
            WikittyExtension wikittyExtension = wikittyService.restoreExtensionLastVersion(name);
            return wikittyExtension;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    /**
     * Retrieves the fields of the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve the fields
     * @return a map containing the field names and their type
     */
    public Map<String, FieldType> getFormTypeFields(String name) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getFormTypeFields(" + name + ")");
        }
        
        try {
            WikittyExtension extension = getFormType(name);
            Map<String, FieldType> fields = new HashMap<String, FieldType>();
            
            for (String fieldName : extension.getFieldNames()) {
                fields.put(fieldName, extension.getFieldType(fieldName));
            }
    
            return fields;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    /**
     * Updates the form type whose name is 'name'
     *
     * @param name the name of the form type to update
     * @param fields the new fields of the form type
     * @param requires the new requires of the form type
     * @param tagValues the new tag values of the form type
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) throws VradiException {
        
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + name + ", fields, requires, tagValues)");
        }
        
        if (name != null) {
            try {
                WikittyExtension lastVersion = getFormType(name);
                String newVersion = null;
                
                if (lastVersion != null) {
                    newVersion = WikittyUtil.incrementMajorRevision(lastVersion.getVersion());
                } else {
                    newVersion = WikittyUtil.DEFAULT_VERSION;
                }
                
                WikittyExtension extension = new WikittyExtension(name, newVersion,
                        requires, new LinkedHashMap<String, FieldType>(fields));
                
                if (tagValues != null) {
                    for (Map.Entry<String, String> entry : tagValues.entrySet()) {
                        extension.addTagValue(entry.getKey(), entry.getValue());
                    }
                }
                
                List<WikittyExtension> extensions = Arrays.asList(extension);
                proxy.getWikittyService().storeExtension(extensions);
                
                log.info(String.format("FormType named %s saved with id: %s and version: %s",
                        name, extension.getId(), newVersion));
                
                return extension;
            
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new VradiException(e);
            }
        }
        
        return null;
    }

    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(WikittyExtension extension) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + extension.toDefinition() + ")");
        }
        
        try {
            Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
            
            for (String fieldName : extension.getFieldNames()) {
                FieldType fieldType = extension.getFieldType(fieldName);
                fields.put(fieldName, fieldType);
            }
    
            WikittyExtension updated = updateFormType(extension.getName(), fields,
                    extension.getRequires(), extension.getTagValues());
            
            return updated;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    /**
     * Associates a new template file to the specified extension
     *
     * @param extension
     * @param template
     * @return the file copied in the vradi template folder
     * @throws IOException
     */
    public File addTemplate(WikittyExtension extension, File template)
            throws VradiException {
        if(extension == null || template == null) {
            return null;
        }
        
        try {
            File templatesDir = Configuration.getInstance().getTemplatesDir();
            File templateDir = new File(templatesDir, extension.getName());
            templateDir.mkdirs();
            
            File copyOfTemplate = new File(templateDir, template.getName());
            FileUtil.copy(template, copyOfTemplate);
            return copyOfTemplate;
            
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    /**
     * 
     * @param extensionName
     * @param templateName
     * @return
     */
    public File getTemplate(String extensionName, String templateName) {
        if(extensionName == null || templateName == null) {
            return null;
        }
        
        File templatesDir = Configuration.getInstance().getTemplatesDir();
        File template = new File(new File(templatesDir, extensionName), templateName);
        return template.exists() ? template : null;
    }

    /**
     * Lists the template files associated with the specified extension
     * @param extension
     * @return
     */
    public File[] getTemplates(WikittyExtension extension) {
        if(extension == null) {
            return null;
        }
        
        File templatesDir = Configuration.getInstance().getTemplatesDir();
        File templateDir = new File(templatesDir, extension.getName());
        File[] result = templateDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return !file.isDirectory() && !file.getName().endsWith(".xml");
            }
        });
        return result;
    }

    public Map<String, String> getAssociatedFields(String extensionName,
            String templateName) throws VradiException {
        if(extensionName == null || templateName == null) {
            return null;
        }

        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            File templatesDir = Configuration.getInstance().getTemplatesDir();
            File targetFile = new File(new File(templatesDir, extensionName), templateName + ".xml");
            document = sxb.build(targetFile);
            
        } catch (IOException e) {
            log.warn(e.getMessage());
            return null;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
        
        Element root = document.getRootElement();
        Map<String, String> fieldMap = new HashMap<String, String>();
        for(Object child : root.getChildren(FormTypeTemplateEnum.FIELD_ELEMENT.toString())) {
            Element field = (Element) child;
            String templateField =
                    field.getAttributeValue(FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.toString());
            String extensionField =
                    field.getAttributeValue(FormTypeTemplateEnum.FIELD_EXTENSION_ATTRIBUTE.toString());
            fieldMap.put(templateField, extensionField);
        }
        return fieldMap;
    }

    public void setAssociatedFields(String extensionName, String templateName,
            Map<String, String> fieldMap) throws VradiException {

        if(extensionName == null || fieldMap == null) {
            return;
        }
        
        File templatesDir = Configuration.getInstance().getTemplatesDir();
        File templateDir = new File(templatesDir, extensionName);
        templateDir.mkdirs();
        File targetFile = new File(templateDir, templateName + ".xml");

        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            document = sxb.build(targetFile);
        } catch (IOException e) {
            document = new Document(new Element(FormTypeTemplateEnum.FIELDS_ELEMENT.toString()));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
        
        try {
            Element root = document.getRootElement();
            root.removeChildren(FormTypeTemplateEnum.FIELD_ELEMENT.toString());
            
            for(Map.Entry<String, String> field : fieldMap.entrySet()) {
                Element child = new Element(FormTypeTemplateEnum.FIELD_ELEMENT.toString());
                child.setAttribute(FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.toString(),
                        field.getKey());
                if(field.getValue() != null) {
                    child.setAttribute(FormTypeTemplateEnum.FIELD_EXTENSION_ATTRIBUTE.toString(),
                            field.getValue());
                }
                root.addContent(child);
            }
            
            XMLOutputter xmlOuputter = new XMLOutputter();
            xmlOuputter.output(document, new FileWriter(targetFile));
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }
    
}
