/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.input.SAXBuilder;
import org.nuiton.util.StringUtil;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.XmlStreamConfig;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;

/**
 * Class containing the methods to manage the binding of the xml streams fields
 * with the form fields :
 * - xml field bindings creation, update and retrieving
 * - xml streams retrieving
 * - form creation with the data from an xml stream
 *
 * @author schorlet
 * @date 2010-01-22 20:18:29
 * @version $Revision: 842 $ $Date: 2010-05-05 10:04:39 +0200 (mer., 05 mai 2010) $
 */
public class BindingManager {
    private static final Log log = LogFactory.getLog(BindingManager.class);

    private final WikittyProxy proxy;
    private final FormTypeManager formTypeManager;
    private final FormManager formManager;
    private Timer timer = null;
    private TimerTask xmlStreamTask = null;

    public BindingManager(WikittyProxy proxy, FormTypeManager formTypeManager,
            FormManager formManager) {
        this.proxy = proxy;
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    public BindingManager(FormTypeManager formTypeManager, FormManager formManager) {
        this.proxy = ServiceFactory.getWikittyProxy();
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    /**
     * Retrieves the xml field bindings whose xml stream is xmlStream
     *
     * @param xmlStream the xml stream associated with the xml field bindings
     * we want to retrieve
     * @return a list containing the xml field bindings associated with
     * the xml stream xmlStream
     */
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();

        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            if (log.isDebugEnabled()) {
                log.debug("getXmlFieldBindings(" + xmlStream.getName() + ")");
                log.debug(xmlStream.getXmlFieldBinding());
            }

            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            
            List<XmlFieldBinding> restore = proxy.restore(XmlFieldBinding.class, bindings);
            list.addAll(restore);
        }

        return list;
    }

    /**
     * Retrieves the xml field binding whose id is xmlFieldBindingId
     *
     * @param xmlFieldBindingId the id of the xml field binding we want to retrieve
     * @return the xml field binding whose id is xmlFieldBindingId
     */
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding(" + xmlFieldBindingId + ")");
        }
        
        XmlFieldBinding xmlFieldBinding = proxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        return xmlFieldBinding;
    }

    /**
     * Updates the xml field bindings given in parameters
     *
     * @param bindings the list of the xml field bindings to update
     * @return the list of the xml field bindings up to date
     */
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings(bindings)");
        }
        
        try {
            List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();
        
            if (bindings != null) {
                bindings = proxy.store(bindings);
                list.addAll(bindings);
            }
            
            return list;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }
    
    public XmlStream updateXmlStream(XmlStream xmlStream, List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlStream(xmlStream, bindings)");
        }
        
        try {
            bindings = updateXmlFieldBindings(bindings);
            
            xmlStream.clearXmlFieldBinding();
            for (XmlFieldBinding binding : bindings) {
                xmlStream.addXmlFieldBinding(binding.getWikittyId());
            }
            
            XmlStream stream = proxy.store(xmlStream);
            return stream;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }
    
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams()");
        }
        
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM)
                .criteria();
        
        PagedResult<XmlStream> xmlStreams = proxy.findAllByCriteria(XmlStream.class, criteria);
        List<XmlStream> all = xmlStreams.getAll();
        
        List<XmlStream> list = new ArrayList<XmlStream>();
        list.addAll(all);
        
        return list;
    }
    
    public XmlStream getXmlStream(String xmlStreamId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream(" + xmlStreamId + ")");
        }

        XmlStream xmlStream = proxy.restore(XmlStream.class, xmlStreamId);
        return xmlStream;
    }
    
    private static class BindingContext {
        int dateParsingError = 0;
        int numberParsingError = 0;
        int nbCreated = 0;
    }
    
    private FormImpl createForm(WikittyExtension formType, List<XmlFieldBinding> bindings,
             org.jdom.Element feed, BindingContext bindingContext) {
        FormImpl form = new FormImpl();
        Wikitty wikitty = form.getWikitty();
        wikitty.addExtension(formType);
        
        for (XmlFieldBinding binding : bindings) {
            String fqFormField = binding.getFormField();
            FieldType fieldType;
            try {
                fieldType = wikitty.getFieldType(fqFormField);
            } catch (Exception e) {
                continue;
            }
            
            fillFormField(wikitty, fieldType, binding, feed, bindingContext);
        }
        
        return form;
    }
    
    private void fillFormField(Wikitty wikitty, FieldType fieldType, XmlFieldBinding binding,
            org.jdom.Element feed, BindingContext bindingContext) {
        
        String fqFormField = binding.getFormField();
        Set<String> xmlFields = binding.getXmlField();
        
        if (xmlFields == null || xmlFields.isEmpty()) {
            // no mapping
            String defaultValue = binding.getDefaultValue();
            fillFormField2(wikitty, fieldType, fqFormField, defaultValue, bindingContext);
            return;
        }
            
        for (String xmlField : xmlFields) {
            org.jdom.Element child = feed.getChild(xmlField);
            String feedValue = null;
            
            // get feed field text
            if (child != null) {
                feedValue = child.getTextTrim();
            }
            
            // get default value
            if (feedValue == null || feedValue.isEmpty()) {
                feedValue = binding.getDefaultValue();
            }
            
            fillFormField2(wikitty, fieldType, fqFormField, feedValue, bindingContext);
        }
    }
    
    private void fillFormField2(Wikitty wikitty, FieldType fieldType, String fqFormField,
            String feedValue, BindingContext bindingContext) {
        // if no value then return
        if (feedValue == null || feedValue.isEmpty()) {
            return;
        }

        switch (fieldType.getType()) {
        case DATE:
            Date date = DateParser2.parse(feedValue);
            if (date != null) {
                wikitty.setFqField(fqFormField, WikittyUtil.solrDateFormat.format(date));
            } else {
                bindingContext.dateParsingError++;
            }
            break;

        case NUMERIC:
            if (NumberUtils.isNumber(feedValue)) {
                Double value = Double.valueOf(feedValue);
                 wikitty.setFqField(fqFormField, value);
            } else {
                bindingContext.numberParsingError++;
            }
            break;

        default:
            Object fieldValue = wikitty.getFqField(fqFormField);
            String newValue = null;

            if (fieldValue != null) {
                newValue = fieldValue + "\n" + feedValue;
            } else {
                newValue = feedValue;
            }

             wikitty.setFqField(fqFormField, newValue);
        }
     }
    
    /**
     * Creates forms with the data of an xml stream
     *
     * @param xmlStream
     * @param vradiUser
     * @return An array containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     * 
     * @throws VradiException for various possible errors
     */
    public int[] getFormsFromXmlStream(XmlStream xmlStream,
            VradiUser vradiUser) throws VradiException {
        
        // TODO EC-20100428 return a serializable structure (easier to use)
        int[] results = new int[4];
        
        String lastItemRecorded = PropertiesManager.getLastItemOfXmlStream(
                xmlStream.getName());
        if (log.isDebugEnabled()) {
            log.debug("getFormsFromXmlStream(" + xmlStream.getName() + ", "
                    + lastItemRecorded);
        }

        if (xmlStream.getFormTypeName() == null) {
            throw new VradiException("xmlStream.formTypeName is null");
        }
        
        WikittyExtension formType = formTypeManager.getFormType(
                xmlStream.getFormTypeName());
        if (formType == null) {
            throw new VradiException("Extension of name xmlStream.formTypeName does not exists");
        }

        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            URL rssUrl = new URL(xmlStream.getUrl());
            log.info("Reading xmlStream url from: " + rssUrl);
            document = sxb.build(rssUrl);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't read xml stream", e);
            }
            throw new VradiException(e);
        }
        
        org.jdom.Element racine = document.getRootElement();
        List<org.jdom.Element> elements = null;
        
        if (racine.getChild(VradiConstants.CHANNEL) != null) {
            org.jdom.Element channel = racine.getChild(VradiConstants.CHANNEL);
            elements = channel.getChildren(VradiConstants.ITEM);
        } else if (racine.getChild(VradiConstants.ITEM) != null) {
            elements = racine.getChildren(VradiConstants.ITEM);
        } else if (racine.getChild(VradiConstants.ENTRY) != null) {
            elements = racine.getChildren(VradiConstants.ENTRY);
        }

        if (elements == null) {
            if (log.isWarnEnabled()) {
                log.warn("Enable to find items or entries in xmlStream");
            }
            return results;
        }

        int index = 0;
        boolean itemAlreadyRecorded = false;
        List<String> encryptedElements = new ArrayList<String>();
        
        while(index < elements.size() && !itemAlreadyRecorded) {
            org.jdom.Element element = elements.get(index);
            StringBuffer sb = new StringBuffer();

            List<org.jdom.Element> fields = element.getChildren();
            for (org.jdom.Element field : fields) {
                sb.append(field.getText());
            }

            String encryptedItemValue = null;
            try {
                encryptedItemValue = StringUtil.encodeMD5(sb.toString());
            } catch (NoSuchAlgorithmException eee) {
                if (log.isFatalEnabled()) {
                    log.fatal("No MD5 algorithm found");
                }
            }

            //check if the element has not yet been read
            if (lastItemRecorded != null
                    && lastItemRecorded.equals(encryptedItemValue)) {
                itemAlreadyRecorded = true;
            } else {
                encryptedElements.add(encryptedItemValue);
                index++;
            }
        }
        index--;
        
        if(index >= 0) {
            String lastItemOfXmlStream = null;
            List<Form> forms = new ArrayList<Form>();

            List<XmlFieldBinding> bindings = proxy.restore(XmlFieldBinding.class,
                new ArrayList<String>(xmlStream.getXmlFieldBinding()));

            String formId = VradiConstants.FORM_ID_DATE_FORMAT.format(new Date());
            String toTreatId = formManager.getNonTraiteStatus().getWikittyId();
            BindingContext bindingContext = new BindingContext();
            
            // for each element of the xml stream
            for (int i = index ; i >= 0 ; i--) {
                org.jdom.Element feed = elements.get(i);

                //create the form with the info from the xml stream
                FormImpl form = createForm(formType, bindings, feed, bindingContext);
                bindingContext.nbCreated++;
                
                form.setId(formId + form.getWikittyId());
                form.setXmlStreamURL(xmlStream.getUrl());
                form.setStatus(toTreatId);
                forms.add(form);
                lastItemOfXmlStream = encryptedElements.get(i);
                if(forms.size() > 1000) {
                    formManager.updateForms(forms);
                    PropertiesManager.setLastItemOfXmlStream(xmlStream.getName(), lastItemOfXmlStream);
                    forms.clear();
                }
            }
            
            if (!forms.isEmpty()) {
                formManager.updateForms(forms);
                PropertiesManager.setLastItemOfXmlStream(xmlStream.getName(), lastItemOfXmlStream);
            }

            results[0] = bindingContext.nbCreated;
            results[2] = bindingContext.dateParsingError;
            results[3] = bindingContext.numberParsingError;

        }

        // equals to : elements.size() - bindingContext.nbCreated
        results[1] = elements.size() - results[0];

        // TODO EC-20100428 : redirect log output into file
        if (log.isInfoEnabled()) {
            log.info("Form import from stream, created = " + results[0]);
            log.info("Form import from stream, already existing = " + results[1]);
            log.info("Form import from stream, dateParsingError = " + results[2]);
            log.info("Form import from stream, numberParsingError = " + results[3]);
        }

        return results;
    }

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param intervalUnit unit of te interval between 2 retrievings (minute, hour or day)
     * @param intervalValue interval value between two retrievings
     * @param hour hour of the retrieving if the unit is day
     * @param minute of the retrieving if the unit is day or hour
     * @throws VradiException
     */
    public void autoLoadFormsFromXmlStreams(String intervalUnit,
                            int intervalValue, Integer hour, Integer minute)
            throws VradiException {
        int delayInMinute = 0;
        int intervalUnitInMinutes = 0;
        Calendar cal = new GregorianCalendar();
        int calHour = cal.get(Calendar.HOUR_OF_DAY);
        int calMinute = cal.get(Calendar.MINUTE);
        int diffMinute = 0;
        int diffHour = 0;
        if(minute != null) {
            diffMinute = (60 //for the diff to be positive
                    + minute - calMinute)
                    % 60 //to have only the minutes
                    ;
            if(hour != null) {
                diffHour = (24 //for the diff to be positive
                        + hour - calHour
                        - 1 // remove one hour in case the actual minute
                            // is greater than the user-defined minute
                        + ((60 + minute - calMinute) / 60)) // readd one hour
                            //  if finally the actual minute
                            // is lower than the user-defined minute
                        % 24; //to have only the hours
            }
        }
        if(log.isDebugEnabled()) {
            log.debug("delay : " + diffHour +  " hours "
                    + diffMinute + " minutes");
        }
        if(XmlStreamConfig.HOURS.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 60;
            delayInMinute = diffMinute;
        } else if(XmlStreamConfig.DAYS.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 60 * 24;
            delayInMinute = diffMinute + diffHour * 60;
        } else if(XmlStreamConfig.MINUTES.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 1;
            delayInMinute = minute;
        }
        int interval = intervalValue * intervalUnitInMinutes * 60 * 1000;
        int delay = delayInMinute * 60 * 1000;
        autoLoadFormsFromXmlStreams(delay, interval);
        PropertiesManager.setXmlStreamConfig(intervalUnit,
                String.valueOf(intervalValue),
                hour == null ? null : String.valueOf(hour),
                minute == null ? null : String.valueOf(minute));
    }

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param delay the delay before the first retrieving
     * @param period interval between two retrievings
     * @throws VradiException
     */
    protected void autoLoadFormsFromXmlStreams(long delay, long period)
            throws VradiException {
        if(timer == null) {
            timer = new Timer();
        }
        if(xmlStreamTask != null) {
            xmlStreamTask.cancel();
            timer.purge();
        }

        xmlStreamTask = new TimerTask() {
            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("Update xml streams task");
                }

                List<XmlStream> xmlStreams = getAllXmlStreams();
                for (XmlStream xmlStream : xmlStreams) {
                    if(log.isDebugEnabled()) {
                        log.debug("Loading forms from " + xmlStream.getName());
                    }
                    try {
                        getFormsFromXmlStream(xmlStream, null);
                    } catch(VradiException eee) {
                        log.error("can't create forms from stream : " + xmlStream.getName());
                    }
                }
            }
        };
        
        timer.scheduleAtFixedRate(xmlStreamTask, delay, period);
    }
}
