/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants.XmlStreamConfig;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.dto.VradiCartographyDTO;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.dto.VradiUserDTO;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.ImportExportManager;
import com.jurismarches.vradi.services.managers.PropertiesManager;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * @author morin
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    /**
     * Proxy to store and retrieve data.
     */
    protected WikittyProxy proxy;
    
    protected ThesaurusManager thesaurusManager;
    protected BindingManager bindingManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected ClientManager clientManager;
    protected SearchManager searchManager;
    protected ImportExportManager importExportManager;
    
    public VradiStorageServiceImpl() {
        proxy = ServiceFactory.getWikittyProxy();
        
        thesaurusManager = new ThesaurusManager(proxy);
        clientManager = new ClientManager(proxy);
        formTypeManager = new FormTypeManager(proxy);
        
        searchManager = new SearchManager(proxy, thesaurusManager);
        formManager = new FormManager(proxy, thesaurusManager, searchManager);
        bindingManager = new BindingManager(proxy, formTypeManager, formManager);
        importExportManager = new ImportExportManager(proxy);

        try {
            String[] config = PropertiesManager.getXmlStreamConfig();
            String intervalUnit = config[0];
            if(intervalUnit != null) {
                Integer intervalValue = config[1] == null ? null :
                        Integer.valueOf(config[1]);
                Integer hour = config[2] == null ? null :
                        Integer.valueOf(config[2]);
                Integer minute = config[3] == null ? null :
                        Integer.valueOf(config[3]);
                autoLoadFormsFromXmlStreams(intervalUnit, intervalValue,
                        hour, minute);
            } else {
                autoLoadFormsFromXmlStreams(XmlStreamConfig.DAYS.toString(), 1, 0, 0);
            }
        } catch(VradiException e) {
            log.error(e.getMessage(), e);
        }
    }

    @Override
    public <E extends BusinessEntity> E getEntity(String id, Class<E> clazz)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getEntity(" + id + ", " + clazz.getSimpleName() + ")");
        }
        
        E entity = proxy.restore(clazz, id);
        return entity;
    }

    @Override
    public <E extends BusinessEntity> List<E> getEntities(List<String> ids, Class<E> clazz)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getEntities(" + clazz.getSimpleName() + ")");
        }

        List<E> entity = proxy.restore(clazz, ids);
        return entity;
    }

    @Override
    public <E extends BusinessEntity> E updateEntity(E entity)
            throws VradiException {
        
        if (entity == null) {
            return entity;
        }
    
        if (log.isDebugEnabled()) {
            log.debug("updateEntity(id:" + entity.getWikittyId() +
                    "; version:" + entity.getWikittyVersion() + ")");
        }
        
        try {
            if(QueryMaker.class.isAssignableFrom(entity.getClass())) {
                archiveQueries((QueryMaker)entity);
            }
            
            // updates the entity
            entity = proxy.store(entity);
            
            if (log.isDebugEnabled()) {
                log.debug("updateEntity(id:" + entity.getWikittyId() +
                        "; version:" + entity.getWikittyVersion() + ")");
            }
                
            return entity;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    @Override
    public <E extends BusinessEntity> E[] updateEntities(E ... entities)
            throws VradiException {

        List<E> list = Arrays.asList(entities);
        list = updateEntities(list);

        return  list.toArray(entities);
    }

    @Override
    public <E extends BusinessEntity> List<E> updateEntities(List<E> entities)
            throws VradiException {

        if (entities == null || entities.size() == 0) {
            return entities;
        }

        try {
            if (log.isDebugEnabled()) {
                log.debug("updateEntities(entities)");
            }

            for (BusinessEntity entity : entities) {
                if (log.isDebugEnabled()) {
                    log.debug("updateEntities(id:" + entity.getWikittyId() +
                            "; version:" + entity.getWikittyVersion() + ")");
                }

                if (QueryMaker.class.isAssignableFrom(entity.getClass())) {
                    archiveQueries((QueryMaker)entity);
                }
            }

            entities = proxy.store(entities);

            return entities;

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }


    @Override
    public void deleteEntity(String id) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteEntity(entity)");
        }

        if (id != null && !id.isEmpty()) {
            try {
                proxy.delete(id);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new VradiException(e);
            }
        }
    }
    
    @Override
    public void deleteEntity(BusinessEntity entity) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteEntity(entity)");
        }
        
        if (entity != null) {
            try {
                proxy.delete(entity.getWikittyId());
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new VradiException(e);
            }
        }
    }

    @Override
    public User getUser(String userId) throws VradiException {
        return clientManager.getUser(userId);
    }

    @Override
    public Client getClient(String clientId) throws VradiException {
        return clientManager.getClient(clientId);
    }

    @Override
    public Group getGroup(String groupId) throws VradiException {
        return clientManager.getGroup(groupId);
    }
    
    @Override
    public String getQueryHistoryFile(String id) {
        return clientManager.getQueryHistoryFile(id);
    }
    
    @Override
    public List<User> getGroupUsers(String groupId) throws VradiException {
        return clientManager.getGroupUsers(groupId);
    }

    /**
     * @deprecated since 20100507 unused
     */
    @Override
    @Deprecated
    public List<User> getClientUsers(String clientId)
            throws VradiException {
        return clientManager.getClientUsers(clientId);
    }

    @Override
    public List<Client> getGroupClients(String groupId)
            throws VradiException {
        return clientManager.getGroupClients(groupId);
    }

    @Override
    public Client getClientByUserId(String userId) throws VradiException {
        return clientManager.getClientByUserId(userId);
    }

    @Override
    public List<Client> getAllClients() throws VradiException {
        return clientManager.getAllClients();
    }

    @Override
    public List<User> getAllUsers() throws VradiException {
        return clientManager.getAllUsers();
    }

    @Override
    public List<VradiUserDTO> getAllUserDTOs() throws VradiException {
        return clientManager.getAllUserDTOs();
    }
    
    @Override
    public List<Group> getAllGroups() throws VradiException {
        return clientManager.getAllGroups();
    }

    @Override
    public Form updateForm(Form form) throws VradiException {
        return formManager.updateForm(form);
    }

    @Override
    public List<Form> updateForms(List<Form> forms) throws VradiException {
        return formManager.updateForms(forms);
    }

    @Override
    public VradiFormPageDTO findForms(String query, final VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {
        return findForms2(query, null, null, null, null, null, null, formPageDTO);
    }

    @Override
    public VradiFormPageDTO findForms2(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate,
            List<String>[] thesaurus, String[] statusIds,
            final VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {

        VradiFormPageDTO result = searchManager.findForms(query, extension, dateType, beginDate, endDate,
                thesaurus, statusIds, formPageDTO);
        return result;
    }

    @Override
    public VradiCartographyDTO getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, java.util.List<String>[] thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException {
        
        VradiCartographyDTO cartography = searchManager.getThesaurusCartography(
                query, extension, dateType, beginDate, endDate, thesaurus, statusIds);
        
        return cartography;
    }
    
    @Override
    public Map<QueryMaker, List<VradiQueryBean>> findQueriesReturningForm(Form form)
            throws VradiException {
        return searchManager.findQueriesReturningForm(form);
    }

    @Override
    public List<WikittyExtension> getAllFormTypes() {
        return formTypeManager.getAllFormTypes();
    }

    @Override
    public WikittyExtension getFormType(String name) throws VradiException {
        return formTypeManager.getFormType(name);
    }

    @Override
    public Map<String, FieldType> getFormTypeFields(String name) throws VradiException {
        return formTypeManager.getFormTypeFields(name);
    }

    @Override
    public WikittyExtension updateFormType2(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) throws VradiException {
        
        return formTypeManager.updateFormType(name, fields, requires, tagValues);
    }

    @Override
    public WikittyExtension updateFormType(WikittyExtension extension) throws VradiException {
        return formTypeManager.updateFormType(extension);
    }

    @Override
    public Form getForm(String formId) {
        return formManager.getForm(formId);
    }

    @Override
    public void deleteForm(String formId) {
        formManager.deleteForm(formId);
    }
    
    @Override
    public List<Form> getForms(List<String> formIds) {
        return formManager.getForms(formIds);
    }

    @Override
    public List<Form> getAllForms() {
        return formManager.getAllForms();
    }

    @Override
    public List<XmlStream> getAllXmlStreams() {
        return bindingManager.getAllXmlStreams();
    }

    @Override
    public List<XmlFieldBinding> updateXmlFieldBindings(
            List<XmlFieldBinding> bindings) throws VradiException {
        return bindingManager.updateXmlFieldBindings(bindings);
    }

    @Override
    public XmlStream updateXmlStream(XmlStream xmlStream, List<XmlFieldBinding> bindings)
             throws VradiException {
        return bindingManager.updateXmlStream(xmlStream, bindings);
    }

    @Override
    public XmlStream getXmlStream(String xmlStreamId)
            throws VradiException {
        return bindingManager.getXmlStream(xmlStreamId);
    }

    @Override
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        return bindingManager.getXmlFieldBinding(xmlFieldBindingId);
    }

    @Override
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        return bindingManager.getXmlFieldBindings(xmlStream);
    }

    @Override
    public void bindForms() throws VradiException {
        formManager.bindForms();
    }

    @Override
    public List<Sending> createSending(Session session, QueryMaker queryMaker, List<Form> formsToBind) throws VradiException {
        return formManager.createAllSending(session, queryMaker, formsToBind, false);
    }

    @Override
    public List<Sending> removeAllSending(Session session, Form form, QueryMaker queryMaker)
            throws VradiException{
        return formManager.removeAllSending(session, form, queryMaker);
    }

    @Override
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        return formManager.getSessions(sessionDate);
    }

    @Override
    public int[] getFormsFromXmlStream(XmlStream xmlStream,
                                          VradiUser vradiUser)
            throws VradiException {
        return bindingManager.getFormsFromXmlStream(xmlStream, vradiUser);
    }

    @Override
    public VradiUser updateVradiUser(VradiUser vradiUser)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateVradiUser(vradiUser)");
        }
        
        Search search = Search.query().eq(VradiUser.FQ_FIELD_NAME, vradiUser.getName());
        Criteria criteria = search.criteria();
        
        PagedResult<VradiUser> users = proxy.findAllByCriteria(VradiUser.class, criteria);
        List<VradiUser> result = users.getAll();
        
        if (result.size() > 0) {
            log.debug("user " + vradiUser + " already exists");
            return null;
            
        } else {
            return proxy.store(vradiUser);
        }
    }

    @Override
    public VradiUser logVradiUser(String vradiUserName,
                                  String vradiUserPassword) {
        if (log.isDebugEnabled()) {
            log.debug("logVradiUser(" + vradiUserName + ", " + vradiUserPassword + ")");
        }
        
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_NAME, vradiUserName)
                .eq(VradiUser.FQ_FIELD_PASSWORD, vradiUserPassword)
                .criteria();
        
        VradiUser vradiUser = proxy.findByCriteria(VradiUser.class, criteria);
        return vradiUser;
    }

    @Override
    public void updateSendings(List<Sending> sendings) {
        if (log.isDebugEnabled()) {
            log.debug("updateSendings(sendings)");
        }
        
        proxy.store(sendings);
    }

    @Override
    public void importData(File file) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("importData(file)");
        }
        // TODO: create an archive of vradi.home directory before
        /*
        log.warn("WARN: deleting all database");
        WikittyService service = proxy.getWikittyService();
        service.clear();
        */

        String toURI = file.toURI().toString();
        log.info("importing file: " + toURI);
        
        proxy.syncImportFromUri(toURI);
        log.info("import done");
    }

    @Override
    public String exportData() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("exportData()");
        }
        
        log.info("exporting all data");

        try {
            Search search = Search.query().keyword("*");
            Criteria criteria = search.criteria();

            String export = proxy.syncExportAllByCriteria(criteria);
            log.info("export done");
            return export;

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }

    /*
     * @see com.jurismarches.vradi.services.VradiStorageService#importAsCSV(java.net.URL)
     */
    @Override
    public void importAsCSV(String uri) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Importing data as CSV with url = " + uri);
        }

        try {
            importExportManager.syncImportFromCSVUri(uri);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", eee);
            }
        }
    }

    /*
     * @see com.jurismarches.vradi.services.VradiStorageService#exportAsCSV()
     */
    @Override
    public String exportAsCSV(Criteria criteria) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Exporting data as CSV with criteria " + criteria);
        }

        String result = null;
        try {
            result = importExportManager.syncExportCSVAllByCriteria(criteria);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", eee);
            }
        }
        return result;
    }

    @Override
    public void reindexData() {
        if (log.isDebugEnabled()) {
            log.debug("reindexData()");
        }
        
        UpdateResponse response = proxy.getWikittyService().syncEngin();
        if (log.isDebugEnabled()) {
            log.debug(response.toString());
        }
    }

    @Override
    public TreeNodeImpl getRootThesaurus() throws VradiException {
        return thesaurusManager.getRootThesaurus();
    }

    @Override
    public List<TreeNodeImpl> getAllThesaurus() throws VradiException {
        return thesaurusManager.getAllThesaurus();
    }

    @Override
    public TreeNodeImpl getThesaurus(String thesaurusId) throws VradiException {
        return thesaurusManager.getThesaurus(thesaurusId);
    }

    @Override
    public List<TreeNodeImpl> getChildrenThesaurus(String thesaurusId)
            throws VradiException {
        return thesaurusManager.getChildrenThesaurus(thesaurusId);
    }

    @Override
    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        return thesaurusManager.getNbFormsForThesaurus(thesaurusId);
    }

    @Override
    public Status getStatus(String statusId) throws VradiException {
        return formManager.getStatus(statusId);
    }

    @Override
    public List<Status> getStatuses(List<String> statusIds)
            throws VradiException {
        return formManager.getStatuses(statusIds);
    }

    @Override
    public List<Status> getAllStatuses() throws VradiException {
        return formManager.getAllStatuses();
    }

    @Override
    public Status updateStatus(Status status) throws VradiException {
        return formManager.updateStatus(status);
    }

    @Override
    public List<Status> updateStatuses(List<Status> statuses)
            throws VradiException {
        return formManager.updateStatuses(statuses);
    }

    @Override
    public void deleteStatus(String statusId) throws VradiException {
        List<String> statusIds = Arrays.asList(statusId);
        formManager.deleteStatuses(statusIds);
    }

    @Override
    public void deleteStatuses(List<String> statusIds) throws VradiException {
        formManager.deleteStatuses(statusIds);
    }

    public List<TreeNodeImpl> proposeThesaurus(Form form, List<TreeNodeImpl> thesaurus) throws VradiException {
        return thesaurusManager.proposeThesaurus(form, thesaurus);
    }

    @Override
    public void archiveQueries(QueryMaker queryMaker)
            throws VradiException {
        clientManager.archiveQueries(queryMaker);
    }

    @Override
    public void autoLoadFormsFromXmlStreams(String intervalUnit,
            int intervalValue, Integer hour, Integer minute)
            throws VradiException {
        bindingManager.autoLoadFormsFromXmlStreams(intervalUnit, intervalValue,
                hour, minute);
    }

    @Override
    public void uploadFiles(Form form, List<File> files) throws IOException {
        formManager.uploadFiles(form, files);
    }

    @Override
    public void uploadAttachments(Form form, List<File> attachments)
            throws IOException {
        formManager.uploadAttachments(form, attachments);
    }

    @Override
    public File addTemplate(WikittyExtension extension, File template)
            throws VradiException {
        return formTypeManager.addTemplate(extension, template);
    }

    @Override
    public File[] getTemplates(WikittyExtension extension) {
        return formTypeManager.getTemplates(extension);
    }

    @Override
    public File getTemplate(String extensionName, String templateName) {
        return formTypeManager.getTemplate(extensionName, templateName);
    }

    @Override
    public void setAssociatedFields(String extensionName,
                                           String templateName,
                                    Map<String, String> fieldMap)
            throws VradiException, IOException {

        formTypeManager.setAssociatedFields(extensionName, templateName, fieldMap);

    }

    @Override
    public Map<String, String> getAssociatedFields(String extensionName,
                                                   String templateName)
            throws VradiException {
        return formTypeManager.getAssociatedFields(extensionName, templateName);
    }


    @Override
    public Map<QueryMaker, List<VradiQueryBean>> getQueriesToModifyAfterThesaurusModification(
            String thesaurusName) {
        return thesaurusManager.getQueriesToModifyAfterThesaurusModification(thesaurusName);
    }
}
