/*
 * #%L
 * Vradi :: Services Web
 * 
 * $Id: FileServlet.java 1870 2010-12-27 12:35:52Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.2/vradi-services-web/src/main/java/com/jurismarches/vradi/services/FileServlet.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLDecoder;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import com.jurismarches.vradi.VradiConstants;

/**
 * Vradi file service (upload/download).
 * 
 * @author chatellier
 * @version $Revision: 1870 $
 * 
 * Last update : $Date: 2010-12-27 13:35:52 +0100 (lun., 27 déc. 2010) $
 * By : $Author: sletellier $
 */
public class FileServlet extends HttpServlet {

    /** serialVersionUID. */
    private static final long serialVersionUID = -5368725458920800210L;

    /** log. */
    private static final Log log = LogFactory.getLog(FileServlet.class);

    /** File service instance. */
    protected FileService fileService = new FileServiceImpl();

    /**
     * Get parameters from URI.
     * 
     * If servlet is called with url : /vradi/file/TEMPLATE/AO
     * and mapped on /file/* in web.xml
     * 
     * Parameters will be { "TEMPLATE", "AO" }
     * 
     * Another example is :
     * Called url : /vradi/file/PDF/xyz/toto.pdf
     * Parameters will be { "PDF", "xyz/toto.pdf" }
     * 
     * @param request
     * @return
     */
    protected String[] getParametersFormRequestURI(HttpServletRequest request) {

        String params = request.getRequestURI();
        params = StringUtils.removeStart(params, request.getContextPath());
        params = StringUtils.removeStart(params, request.getServletPath());
        params = StringUtils.removeStart(params, "/");
        String[] paramsArray = params.split("/");

        return paramsArray;
    }

    /*
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException {

        // get is used to download file
        // paramsArray = type and uri
        String [] paramsArray = getParametersFormRequestURI(request);

        // get file type
        String uploadType = paramsArray[0];
        if (log.isDebugEnabled()) {
            log.debug("Managing download type : " + uploadType);
        }
        VradiConstants.FileType fileType = VradiConstants.FileType.valueOf(uploadType);
        if (fileType == null) {
            throw new ServletException("Invalid request : unknown upload type : " + uploadType);
        }

        try {
            File file = null;
            switch (fileType) {
                case TEMPLATE : {
                    // two elements : extensionname/file.ext
                    String extensionName = paramsArray[1];
                    String templateName = paramsArray[2];
                    file = fileService.downloadTemplate(false, extensionName, templateName);
                    break;
                }
                case PDF : {
                    // one element : file.ext
                    String uri = paramsArray[1];
                    file = fileService.downloadPDF(false, uri);
                    break;
                }
                case WEB_HARVEST : {
                    // one element : file.ext
                    String uri = paramsArray[1];
                    file = fileService.downloadWebHarvestScript(false, uri);
                    break;
                }
                case EMBEDDED_FORM_FILES : {
                    // two element : formid/file.ext
                    String formId = paramsArray[1];
                    String fileName = paramsArray[2];
                    fileName = URLDecoder.decode(fileName, "UTF-8");
                    file = fileService.downloadFormEmbeddedFile(false, fileName, formId);
                    break;
                }
                case ATTACHMENTS_FORM_FILES : {
                    // two element : formid/file.ext
                    String formId = paramsArray[1];
                    String fileName = paramsArray[2];
                    fileName = URLDecoder.decode(fileName, "UTF-8");
                    file = fileService.downloadFormAttachment(false, fileName, formId);
                    break;
                }
                case ATTACHMENTS_SESSION_FILES : {
                    // two element : formid/file.ext
                    String formId = paramsArray[1];
                    String fileName = paramsArray[2];
                    fileName = URLDecoder.decode(fileName, "UTF-8");
                    file = fileService.downloadSessionAttachment(false, fileName, formId);
                    break;
                }
                case QUERY_HISTORY : {
                    // two element : groupId/file.ext
                    String uri = paramsArray[1];
                    file = fileService.downloadRequestHistory(false, uri);
                    break;
                }
            }

            if (file != null && file.isFile()) {
                goGetFile(request, response, file, fileType);
            }
            else {
                if (log.isDebugEnabled()) {
                    log.debug("Can't find file for params : " + Arrays.toString(paramsArray));
                }
                response.setStatus(HttpServletResponse.SC_NOT_FOUND);
            }

        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't download file", eee);
            }
            throw new ServletException("Can't download file", eee);
        }

    }

    /**
     * Stream file to response output to download.
     * 
     * @param request request
     * @param response response
     * @param file file to download
     */
    protected void goGetFile(HttpServletRequest request,
            HttpServletResponse response, File file, VradiConstants.FileType fileType) throws IOException {
        
        // get some infos
        InputStream in = null;
        OutputStream output = null;
        try {
            output = response.getOutputStream();
            ServletContext context = getServletConfig().getServletContext();
            String mimetype = context.getMimeType(file.getName());

            String mimeType = (mimetype != null) ? mimetype : fileType.getMineType();
            if (log.isDebugEnabled()) {               
                log.debug("Sent with mime type : " + mimeType);
            }
            
            // add header
            response.setContentType(mimeType);
            response.setContentLength((int)file.length());
            response.setHeader("Content-Disposition", "attachment; filename=\"" + file.getName() + "\"");
    
            // stream file to response output
            in = new DataInputStream(new FileInputStream(file));
            IOUtils.copy(in, output);
        }
        finally {
            IOUtils.closeQuietly(in);
            if (output != null) {
                output.flush();
                output.close();
            }
        }

    }

    /*
     * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException {

        // post is used to upload file

        // Check that we have a file upload request
        boolean isMultipart = ServletFileUpload.isMultipartContent(request);
        if (isMultipart) {
            
            String [] paramsArray = getParametersFormRequestURI(request);

            // Create a factory for disk-based file items
            FileItemFactory factory = new DiskFileItemFactory();

            // Create a new file upload handler
            ServletFileUpload upload = new ServletFileUpload(factory);

            // Parse the request
            try {

                // first, parse multipart parameters
                String uploadType = paramsArray[0];
                if (log.isDebugEnabled()) {
                    log.debug("Managing upload type : " + uploadType);
                }
                VradiConstants.FileType fileType = VradiConstants.FileType.valueOf(uploadType);

                // currently, can't manage multiples files due to
                // returned fileuri response
                List<FileItem> items = upload.parseRequest(request);
                if (items.size() != 1) {
                    throw new ServletException("Invalid request : " + items.size() + " files found !");
                }
                FileItem item = items.get(0);
                String fileName = item.getName();

                File tempDirectory = FileUtil.createTempDirectory("vradi-temp", "");
                File tempFile = new File(tempDirectory, fileName);
                item.write(tempFile);

                String fileUri = null;
                switch (fileType) {
                    case TEMPLATE : {
                        // single name here
                        String extensionName = paramsArray[1];
                        fileUri = fileService.uploadTemplate(tempFile, extensionName);
                        break;
                    }
                    case WEB_HARVEST : {
                        fileUri = fileService.uploadWebHarvestScript(tempFile);
                        break;
                    }
                    case ATTACHMENTS_FORM_FILES : {
                        // single name here
                        String formId = paramsArray[1];
                        fileUri = fileService.uploadFormAttachment(tempFile, formId);
                        break;
                    }
                    case ATTACHMENTS_SESSION_FILES: {
                        // single name here
                        String formId = paramsArray[1];
                        fileUri = fileService.uploadSessionAttachment(tempFile, formId);
                        break;
                    }
                    case EMBEDDED_FORM_FILES : {
                        // single name here
                        String formId = paramsArray[1];
                        fileUri = fileService.uploadFormEmbeddedFile(tempFile, formId);
                        break;
                    }
                    case PDF : {
                        // PDF are generated server side,  can't be uploaded
                        throw new ServletException("Forbidden operation");
                    }
                    case QUERY_HISTORY: {
                        // Request history are generated server side,  can't be uploaded
                        throw new ServletException("Forbidden operation");
                    }
                    case TEMP_FILE : {
                        // no name here (not important)
                        fileUri = fileService.uploadTempFile(tempFile);
                        break;
                    }
                }

                // return file uri as response
                // correspond to uri relative to fileType upload
                response.setHeader("fileuri", fileUri);

                FileUtils.deleteDirectory(tempDirectory);

                if (log.isDebugEnabled()) {
                    log.debug("Uploaded file " + fileName + "(" + item.getSize() + "b)"
                            + " saved to " + fileUri);
                }

            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't parse uploaded files", ex);
                }
                throw new ServletException("Can't parse uploaded files", ex);
            }

        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("Request is not multipart");
            }
            response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
        }
    }
}
