package com.franciaflex.faxtomail.ui.swing.content.pdfeditor;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: PDFEditorUIHandler.java 660 2014-10-02 11:30:33Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-1.0/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/content/pdfeditor/PDFEditorUIHandler.java $
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.awt.*;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;

import javax.media.jai.PlanarImage;
import javax.swing.JComponent;
import javax.swing.JPanel;

import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import jaxx.runtime.swing.JAXXButtonGroup;
import jaxx.runtime.validator.swing.SwingValidator;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.pdfbox.io.RandomAccess;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.nuiton.jaxx.application.swing.util.Cancelable;
import org.nuiton.jaxx.application.swing.util.CloseableUI;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.ui.swing.actions.GenerateAnnotatedAttachmentAction;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.franciaflex.faxtomail.ui.swing.util.JImagePanel;
import com.itextpdf.text.DocumentException;

/**
 * @author Kevin Morin (Code Lutin)
 *
 */
public class PDFEditorUIHandler extends AbstractFaxToMailUIHandler<PDFEditorUIModel, PDFEditorUI> implements CloseableUI, Cancelable {

    private static final Log log = LogFactory.getLog(PDFEditorUIHandler.class);

    protected ComponentMover cm = new ComponentMover();
    protected ComponentResizer cr = new ComponentResizer();

    protected PDDocument pdf;

    @Override
    public void afterInit(PDFEditorUI pdfEditorUI) {
        initUI(pdfEditorUI);

        cm.setDragInsets(cr.getDragInsets());
        cm.setEdgeInsets(new Insets(0, 0, 0, 0));

        final PDFEditorUIModel model = getModel();

        JAXXButtonGroup actionGroup = ui.getActionGroup();
        actionGroup.addPropertyChangeListener(JAXXButtonGroup.SELECTED_VALUE_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String value = (String) evt.getNewValue();
                model.setSelectedComponent(PDFEditorUIModel.EditionComponent.valueOf(value));
            }
        });

        String value = (String) actionGroup.getSelectedValue();
        model.setSelectedComponent(PDFEditorUIModel.EditionComponent.valueOf(value));

        ui.getContainer().addContainerListener(new ContainerListener() {
            @Override
            public void componentAdded(ContainerEvent e) {
                Component child = e.getChild();
                PDFEditorUIModel.Page currentPage = model.getCurrentPage();
                if (child.getClass().isAssignableFrom(PDFEditorNoteUI.class)) {
                    currentPage.addNote((PDFEditorNoteUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorCrossUI.class)) {
                    currentPage.addCross((PDFEditorCrossUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorLineUI.class)) {
                    currentPage.addLine((PDFEditorLineUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorHighlighterUI.class)) {
                    currentPage.addHighlighter((PDFEditorHighlighterUI) child);
                }
            }

            @Override
            public void componentRemoved(ContainerEvent e) {
                Component child = e.getChild();
                PDFEditorUIModel.Page currentPage = model.getCurrentPage();
                if (child.getClass().isAssignableFrom(PDFEditorNoteUI.class)) {
                    currentPage.removeNote((PDFEditorNoteUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorCrossUI.class)) {
                    currentPage.removeCross((PDFEditorCrossUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorLineUI.class)) {
                    currentPage.removeLine((PDFEditorLineUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorHighlighterUI.class)) {
                    currentPage.removeHighlighter((PDFEditorHighlighterUI) child);
                }
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_CURRENT_PAGE_INDEX,
                                        new PropertyChangeListener() {
                                            @Override
                                            public void propertyChange(PropertyChangeEvent evt) {
                                                Integer pageNb = (Integer) evt.getNewValue();
                                                Integer prevPageNb = (Integer) evt.getOldValue();
                                                updatePageNumber(pageNb, prevPageNb);
                                            }
                                        }
        );

        model.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (Attachment.PROPERTY_ORIGINAL_FILE.equals(evt.getPropertyName())
                        || Attachment.PROPERTY_EDITED_FILE.equals(evt.getPropertyName())) {

                    PDFEditorUIModel model = (PDFEditorUIModel) evt.getSource();
                    AttachmentFile attachmentFile = model.getNotNullFile();

                    if (attachmentFile != null) {
                        File file = attachmentFile.getFile();
                        if (!FaxToMailUIUtil.isFileAPDF(attachmentFile)) {
                            try {
                                file = convertFileToPdf(attachmentFile);

                            } catch (IOException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.convertToPdf.error"));

                            } catch (DocumentException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.convertToPdf.error"));
                            }
                        }
                        if (FaxToMailUIUtil.isFileAPDF(attachmentFile)) {
                            try {
                                pdf = PDDocument.load(file);

                                model.setPageNumber(pdf.getNumberOfPages());
                                model.setCurrentPageIndex(0);

                            } catch (IOException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.readPdf.error"));
                            }
                        }
                        getUI().setCursor(Cursor.getDefaultCursor());
                        model.firePropertyChanged(PDFEditorUIModel.PROPERTY_CURRENT_PAGE_INDEX, null, 0);
                    }
                }
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_ZOOM, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                displayPage(model.getCurrentPageIndex(),
                            (Float) evt.getOldValue(),
                            model.getRotation());
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_ROTATION, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                displayPage(model.getCurrentPageIndex(),
                            model.getZoom(),
                            (Integer) evt.getOldValue());
            }
        });
    }

    @Override
    protected JComponent getComponentToFocus() {
        return null;
    }

    @Override
    public SwingValidator<PDFEditorUIModel> getValidator() {
        return null;
    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen2(
                true,
                getModel().isModify(),
                null,
                t("faxtomail.pdfEditor.askSaveBeforeLeaving.save"),
                getContext().getActionFactory().createLogicAction(this, GenerateAnnotatedAttachmentAction.class)
        );
        return result;
    }

    @Override
    public Component getTopestUI() {
        return getUI();
    }

    @Override
    public void onCloseUI() {

    }

    public void addEditionComponent(MouseEvent event) {
        PDFEditorUIModel.EditionComponent editionComponent = getModel().getSelectedComponent();
        switch (editionComponent) {
            case NOTE:
                addNote(event.getX(), event.getY());
                break;
            case CROSS:
                addCross(event.getX(), event.getY());
                break;
            case HLINE:
                addHLine(event.getX(), event.getY());
                break;
            case VLINE:
                addVLine(event.getX(), event.getY());
                break;
            case HIGHLIGHTER:
                addHighlighter(event.getX(), event.getY());
                break;
        }
    }

    public void addNote(int x, int y) {
        PDFEditorNoteUI note = new PDFEditorNoteUI();
        note.setZoom(getModel().getZoom());
        String title = decorate(new Date()) + " - " + getContext().getCurrentUser().getTrigraph();
        note.setTitle(title);
        addPanel(note, x, y);
    }

    public void addCross(int x, int y) {
        PDFEditorCrossUI cross = new PDFEditorCrossUI();
        addPanel(cross, x, y);
        cr.registerComponent(cross);
    }

    public void addHLine(int x, int y) {
        PDFEditorLineUI line = new PDFEditorLineUI();
        line.setHorizontal(true);
        addPanel(line, x, y);
        cr.registerComponent(ComponentResizer.DIRECTION_HORIZONTAL, line);
    }

    public void addVLine(int x, int y) {
        PDFEditorLineUI line = new PDFEditorLineUI();
        line.setHorizontal(false);
        addPanel(line, x, y);
        cr.registerComponent(ComponentResizer.DIRECTION_VERTICAL, line);
    }

    public void addHighlighter(int x, int y) {
        PDFEditorHighlighterUI highlighter = new PDFEditorHighlighterUI();
        addPanel(highlighter, x, y);
        cr.registerComponent(highlighter);
    }

    protected void addPanel(JPanel panel, int x, int y) {
        JPanel container = ui.getContainer();
        container.add(panel, 0);

        Insets insets = container.getInsets();
        Dimension size = panel.getPreferredSize();
        panel.setBounds(x + insets.left, y + insets.top,
                        size.width, size.height);

        container.updateUI();

        cm.registerComponent(panel);
    }

    protected void updatePageNumber(Integer pageNb, Integer prevPageNb) {
        if (pdf != null) {
            if (pageNb != null) {
                if (prevPageNb != null) {
                    PDFEditorUIModel.Page p = getModel().getPage(prevPageNb);
                    for (JPanel panel : p.getNotes()) {
                        panel.setVisible(false);
                    }
                    for (JPanel panel : p.getCrosses()) {
                        panel.setVisible(false);
                    }
                }

                displayPage(pageNb, getModel().getZoom(), getModel().getRotation());
            }
        }
    }

    protected void displayPage(int pageNb, float previousZoom, int previousRotation) {
        List<PDPage> pages = pdf.getDocumentCatalog().getAllPages();
        PDPage page = pages.get(pageNb);
        // create the image
        PDRectangle bBox = page.findCropBox();
        Rectangle rect = new Rectangle(0, 0, (int) bBox.getWidth(),
                                       (int) bBox.getHeight());

        float zoom = getModel().getZoom();
        int rotation = getModel().getRotation();

        int width = (int) (zoom * rect.width);
        int height = (int) (zoom * rect.height);
        try {
            Image image = page.convertToImage(BufferedImage.TYPE_INT_RGB, 72);
            //                page.getImage(width, height,    // width & height
            //                                    rect,                       // clip rect
            //                                    null,                       // null for the ImageObserver
            //                                    true,                       // fill background with white
            //                                    true                        // block until drawing is done
            //        );

            JPanel container = getUI().getContainer();
            Dimension containerSize = new Dimension(rotation % 180 == 0 ? width : height,
                                                    rotation % 180 == 0 ? height : width);
            container.setPreferredSize(containerSize);
            container.setMinimumSize(containerSize);
            container.setMaximumSize(containerSize);
            container.setSize(containerSize);

            JImagePanel documentPanel = ui.getDocumentPanel();
            documentPanel.setRotation(rotation);
            documentPanel.setImage(image);

            Insets insets = container.getInsets();
            rect = container.getBounds();

            float zoomRatio = zoom / previousZoom;
            int rotationDiff = rotation - previousRotation;

            PDFEditorUIModel.Page p = getModel().getPage(pageNb);

            for (PDFEditorNoteUI panel : p.getNotes()) {
                panel.setVisible(true);

                panel.setZoom(zoom);
                Rectangle bounds = panel.getBounds();
                int x, y;

                if (rotationDiff == 0) {
                    x = bounds.x;
                    y = bounds.y;

                } else if (rotationDiff == 90 || rotationDiff == -270) {
                    x = rect.width - bounds.height / 2 - bounds.width / 2 - bounds.y;
                    y = bounds.x + bounds.width / 2 - bounds.height / 2;

                } else {
                    x = bounds.y + bounds.height / 2 - bounds.width / 2;
                    y = rect.height - bounds.height / 2 - bounds.width / 2 - bounds.x;
                }

                panel.setBounds((int) (zoomRatio * x) + insets.left,
                                (int) (zoomRatio * y) + insets.top,
                                (int) (zoomRatio * bounds.width), (int) (zoomRatio * bounds.height));

            }

            boolean orientation180 = Math.abs(rotationDiff) % 180 == 0;

            for (PDFEditorCrossUI panel : p.getCrosses()) {
                panel.setVisible(true);

                Rectangle bounds = panel.getBounds();

                int newWidth = orientation180 ? bounds.width : bounds.height;
                int newHeight = orientation180 ? bounds.height : bounds.width;

                int x, y;

                if (rotationDiff == 0) {
                    x = bounds.x;
                    y = bounds.y;

                } else if (rotationDiff == 90 || rotationDiff == -270) {
                    x = rect.width - newWidth - bounds.y;
                    y = bounds.x;

                } else {
                    x = bounds.y;
                    y = rect.height - newHeight - bounds.x;
                }

                panel.setBounds((int) (zoomRatio * x) + insets.left,
                                (int) (zoomRatio * y) + insets.top,
                                (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

            }

            List<PDFEditorLineUI> lines = p.getLines();
            cr.deregisterComponent(lines.toArray(new PDFEditorLineUI[lines.size()]));

            for (PDFEditorLineUI panel : lines) {
                panel.setVisible(true);

                Rectangle bounds = panel.getBounds();

                boolean horizontal = panel.isHorizontal();
                panel.setHorizontal(orientation180 ? horizontal : !horizontal);

                int newWidth = orientation180 ? bounds.width : bounds.height;
                int newHeight = orientation180 ? bounds.height : bounds.width;

                int x, y;

                if (rotationDiff == 0) {
                    x = bounds.x;
                    y = bounds.y;

                } else if (rotationDiff == 90 || rotationDiff == -270) {
                    x = rect.width - newWidth - bounds.y;
                    y = bounds.x;

                } else {
                    x = bounds.y;
                    y = rect.height - newHeight - bounds.x;
                }

                panel.setBounds((int) (zoomRatio * x) + insets.left,
                                (int) (zoomRatio * y) + insets.top,
                                (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

                cr.registerComponent(panel.isHorizontal() ? ComponentResizer.DIRECTION_HORIZONTAL : ComponentResizer.DIRECTION_VERTICAL,
                                     panel);
            }

            for (PDFEditorHighlighterUI panel : p.getHighlighters()) {
                panel.setVisible(true);

                Rectangle bounds = panel.getBounds();

                int newWidth = orientation180 ? bounds.width : bounds.height;
                int newHeight = orientation180 ? bounds.height : bounds.width;

                int x, y;

                if (rotationDiff == 0) {
                    x = bounds.x;
                    y = bounds.y;

                } else if (rotationDiff == 90 || rotationDiff == -270) {
                    x = rect.width - newWidth - bounds.y;
                    y = bounds.x;

                } else {
                    x = bounds.y;
                    y = rect.height - newHeight - bounds.x;
                }

                panel.setBounds((int) (zoomRatio * x) + insets.left,
                                (int) (zoomRatio * y) + insets.top,
                                (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

            }

            container.updateUI();

        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Cannot convert page into image", e);
            }
        }
    }

    public File convertFileToPdf(AttachmentFile attachmentFile) throws IOException, DocumentException {
        File target = FaxToMailUIUtil.convertFileToPdf(attachmentFile);
        // convert content to blob
        FaxToMailServiceContext serviceContext = getContext().newServiceContext();
        AttachmentFile attachmentFileNew = serviceContext.getEmailService().getAttachmentFileFromStream(new FileInputStream(target));
        attachmentFileNew.setFilename(attachmentFile.getFilename() + ".pdf");
        attachmentFileNew.setRotation(0);
        getModel().setEditedFile(attachmentFileNew);

        return target;
    }

    public void zoomOut() {
        PDFEditorUIModel model = getModel();
        float zoom = model.getZoom();
        model.setZoom(zoom - 0.5f);
    }

    public void zoomIn() {
        PDFEditorUIModel model = getModel();
        float zoom = model.getZoom();
        model.setZoom(zoom + 0.5f);
    }

    public void rotateClockwise() {
        int rotation = getModel().getRotation();
        getModel().setRotation((360 + rotation + 90) % 360);
    }

    public void rotateAntiClockwise() {
        int rotation = getModel().getRotation();
        getModel().setRotation((360 + rotation - 90) % 360);
    }

    public void print() {
        PDFEditorUIModel model = getModel();
        AttachmentFile attachmentFile = model.getNotNullFile();
        boolean print = FaxToMailUIUtil.print(attachmentFile, false);

        // TODO kmorin 20140702 à mettre dans une action ou avec un loading ou qqchose
        // j'ai essayé une action vite fait mais ca ferme l'éditeur à la fin de l'action
        if (print) {
            DemandeUIModel demand = model.getDemand();
            FaxToMailServiceContext serviceContext = getContext().newServiceContext();
            Email email = serviceContext.getEmailService().addToHistory(demand.getTopiaId(),
                                                                      HistoryType.PRINTING,
                                                                      getContext().getCurrentUser(),
                                                                      new Date(),
                                                                      attachmentFile.getFilename());
            demand.setHistory(email.getHistory());
        }
    }

    @Override
    public void cancel() {
        closeFrame();
    }
}
